"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appsync_1 = require("aws-cdk-lib/aws-appsync");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
    /**
     * Lambda authorization type
     */
    AuthorizationType["LAMBDA"] = "AWS_LAMBDA";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApi(api);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resourceArns);
            }
            throw error;
        }
        return this.arns.map((arn) => aws_cdk_lib_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
_a = JSII_RTTI_SYMBOL_1;
IamResource[_a] = { fqn: "@aws-cdk/aws-appsync-alpha.IamResource", version: "2.50.0-alpha.0" };
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApiProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GraphqlApi);
            }
            throw error;
        }
        const defaultMode = props.authorizationConfig?.defaultAuthorization ??
            { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = props.authorizationConfig?.additionalAuthorizationModes ?? [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new aws_appsync_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(defaultMode.lambdaAuthorizerConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = props.schema ?? new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (props.domainName) {
            this.domainNameResource = new aws_appsync_1.CfnDomainName(this, 'DomainName', {
                domainName: props.domainName.domainName,
                certificateArn: props.domainName.certificate.certificateArn,
                description: `domain for ${this.name} at ${this.graphqlUrl}`,
            });
            const domainNameAssociation = new aws_appsync_1.CfnDomainNameApiAssociation(this, 'DomainAssociation', {
                domainName: props.domainName.domainName,
                apiId: this.apiId,
            });
            domainNameAssociation.addDependsOn(this.domainNameResource);
        }
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })?.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.LAMBDA)) {
            const config = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.LAMBDA && mode.lambdaAuthorizerConfig;
            })?.lambdaAuthorizerConfig;
            config?.handler.addPermission('appsync', {
                principal: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
                action: 'lambda:InvokeFunction',
            });
        }
        const logGroupName = `/aws/appsync/apis/${this.apiId}`;
        this.logGroup = aws_logs_1.LogGroup.fromLogGroupName(this, 'LogGroup', logGroupName);
        if (props.logConfig?.retention) {
            new aws_logs_1.LogRetention(this, 'LogRetention', {
                logGroupName: this.logGroup.logGroupName,
                retention: props.logConfig.retention,
            });
        }
        ;
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApiAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromGraphqlApiAttributes);
            }
            throw error;
        }
        const arn = attrs.graphqlApiArn ?? aws_cdk_lib_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_IamResource(resources);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.LAMBDA).length > 1) {
            throw new Error('You can only have a single AWS Lambda function configured to authorize your API.');
        }
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing User Pool Configuration');
            }
            if (mode.authorizationType === AuthorizationType.LAMBDA && !mode.lambdaAuthorizerConfig) {
                throw new Error('Missing Lambda Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        if (!config)
            return undefined;
        const logsRoleArn = config.role?.roleArn ?? new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        }).roleArn;
        const fieldLogLevel = config.fieldLogLevel ?? FieldLogLevel.NONE;
        return {
            cloudWatchLogsRoleArn: logsRoleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.env.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupLambdaAuthorizerConfig(config) {
        if (!config)
            return undefined;
        return {
            authorizerResultTtlInSeconds: config.resultsCacheTtl?.toSeconds(),
            authorizerUri: config.handler.functionArn,
            identityValidationExpression: config.validationRegex,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
                lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(mode.lambdaAuthorizerConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        if (config?.expires?.isBefore(aws_cdk_lib_1.Duration.days(1)) || config?.expires?.isAfter(aws_cdk_lib_1.Duration.days(365))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = config?.expires ? config?.expires.toEpoch() : undefined;
        return new aws_appsync_1.CfnApiKey(this, `${config?.name || 'Default'}ApiKey`, {
            expires,
            description: config?.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_IIntermediateType(type);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addType);
            }
            throw error;
        }
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addQuery);
            }
            throw error;
        }
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMutation);
            }
            throw error;
        }
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSubscription);
            }
            throw error;
        }
        return this.schema.addSubscription(fieldName, field);
    }
    /**
     * The AppSyncDomainName of the associated custom domain
     */
    get appSyncDomainName() {
        if (!this.domainNameResource) {
            throw new Error('Cannot retrieve the appSyncDomainName without a domainName configuration');
        }
        return this.domainNameResource.attrAppSyncDomainName;
    }
}
exports.GraphqlApi = GraphqlApi;
_b = JSII_RTTI_SYMBOL_1;
GraphqlApi[_b] = { fqn: "@aws-cdk/aws-appsync-alpha.GraphqlApi", version: "2.50.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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