"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const apigateway = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const core_2 = require("@aws-cdk/core");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const core_3 = require("@aws-solutions-constructs/core");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_3.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_14_X,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: core_2.Duration.hours(8),
    minTtl: core_2.Duration.hours(5),
    maxTtl: core_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: core_2.Duration.minutes(0),
    minTtl: core_2.Duration.minutes(0),
    maxTtl: core_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.NONE,
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    }
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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