"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const events = require("@aws-cdk/aws-events");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventsRuleToSqs(stack, 'test-events-rule-sqs', props);
}
function deployStackWithNewEventBus(stack) {
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {}
    };
    return new lib_1.EventsRuleToSqs(stack, 'test-eventsrule-sqs-new-bus', props);
}
test('check the sqs queue properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SQS::Queue', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testeventsrulesqstesteventsrulesqsWEncryptionKey59B6B2A9",
                "Arn"
            ]
        },
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testeventsrulesqstesteventsrulesqsWdeadLetterQueue6C5AAA92",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
});
test('check the sqs queue properties with existing KMS key', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKey: key
    };
    new lib_1.EventsRuleToSqs(stack, 'test-events-rule-sqs', props);
    expect(stack).toHaveResource('AWS::SQS::Queue', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        },
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testeventsrulesqstesteventsrulesqsWdeadLetterQueue6C5AAA92",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
    expect(stack).toHaveResource('AWS::KMS::Key', {
        Description: "my-key",
        EnableKeyRotation: true
    });
});
test('check if the event rule has permission/policy in place in sqs queue for it to be able to send messages to the queue.', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SQS::QueuePolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:DeleteMessage",
                        "sqs:ReceiveMessage",
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:RemovePermission",
                        "sqs:AddPermission",
                        "sqs:SetQueueAttributes",
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition",
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId",
                                    },
                                    ":root"
                                ],
                            ],
                        },
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqstesteventsrulesqsWqueue0E3B047B",
                            "Arn",
                        ],
                    },
                    Sid: "QueueOwnerOnlyAccess",
                },
                {
                    Action: "SQS:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false",
                        },
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqstesteventsrulesqsWqueue0E3B047B",
                            "Arn",
                        ],
                    },
                    Sid: "HttpsOnly",
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqstesteventsrulesqsWqueue0E3B047B",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Queues: [
            {
                Ref: "testeventsrulesqstesteventsrulesqsWqueue0E3B047B",
            }
        ]
    });
});
test('check if the dead letter queue policy is setup', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SQS::QueuePolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:DeleteMessage",
                        "sqs:ReceiveMessage",
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:RemovePermission",
                        "sqs:AddPermission",
                        "sqs:SetQueueAttributes",
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition",
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ],
                            ],
                        },
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqstesteventsrulesqsWdeadLetterQueue6C5AAA92",
                            "Arn",
                        ],
                    },
                    Sid: "QueueOwnerOnlyAccess",
                },
                {
                    Action: "SQS:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false",
                        },
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqstesteventsrulesqsWdeadLetterQueue6C5AAA92",
                            "Arn",
                        ],
                    },
                    Sid: "HttpsOnly",
                },
            ],
            Version: "2012-10-17",
        },
        Queues: [
            {
                Ref: "testeventsrulesqstesteventsrulesqsWdeadLetterQueue6C5AAA92",
            },
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    expect(construct.eventsRule !== null);
    expect(construct.sqsQueue !== null);
    expect(construct.encryptionKey !== null);
    expect(construct.deadLetterQueue !== null);
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const construct = deployStackWithNewEventBus(stack);
    expect(construct.eventsRule !== null);
    expect(construct.sqsQueue !== null);
    expect(construct.encryptionKey !== null);
    expect(construct.deadLetterQueue !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    expect(stack).toHaveResource('AWS::Events::EventBus');
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {},
        existingEventBusInterface: new events.EventBus(stack, `test-existing-new-eventbus`, {})
    };
    const app = () => {
        new lib_1.EventsRuleToSqs(stack, 'test-eventsrule-sqs', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const props = {
        eventBusProps: {
            eventBusName: 'testcustomeventbus'
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new lib_1.EventsRuleToSqs(stack, 'test-new-eventsrule-sqs', props);
    expect(stack).toHaveResource('AWS::Events::EventBus', {
        Name: 'testcustomeventbus'
    });
});
//# sourceMappingURL=data:application/json;base64,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