"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
/**
 * @stability stable
 */
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener.
 *
 * Routes to a fleet of of instances in a VPC.
 *
 * @stability stable
 */
class LoadBalancer extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer.
         *
         * @stability stable
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (_c = props.crossZone) !== null && _c !== void 0 ? _c : true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        if (props.accessLoggingPolicy !== undefined) {
            this.elb.accessLoggingPolicy = props.accessLoggingPolicy;
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer.
     *
     * @returns A ListenerPort object that controls connections to the listener port
     * @stability stable
     */
    addListener(listener) {
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    /**
     * @stability stable
     */
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancing.LoadBalancer", version: "1.125.0" };
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *     const listener = elb.addListener(...);
 *
 *     listener.connections.allowDefaultPortFromAnyIPv4();
 *     // or
 *     instance.connections.allowToDefaultPort(listener);
 *
 * @stability stable
 */
class ListenerPort {
    /**
     * @stability stable
     */
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
_b = JSII_RTTI_SYMBOL_1;
ListenerPort[_b] = { fqn: "@aws-cdk/aws-elasticloadbalancing.ListenerPort", version: "1.125.0" };
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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