"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirewallDomainList = exports.FirewallDomains = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const route53resolver_generated_1 = require("./route53resolver.generated");
/**
 * A list of domains
 */
class FirewallDomains {
    /**
     * Firewall domains created from a list of domains
     *
     * @param list the list of domains
     */
    static fromList(list) {
        for (const domain of list) {
            if (!/^([\w-.]{1,255}|\*[\w-.]{1,254})$/.test(domain)) {
                throw new Error(`Invalid domain: ${domain}. Domain can optionally start with *. Max length of 255. Valid characters: A-Z, a-z, 0-9, _, -, .`);
            }
        }
        return {
            bind(_scope) {
                return { domains: list };
            },
        };
    }
    /**
     * Firewall domains created from the URL of a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param url S3 bucket url (s3://bucket/prefix/objet).
     */
    static fromS3Url(url) {
        if (!core_1.Token.isUnresolved(url) && !url.startsWith('s3://')) {
            throw new Error(`The S3 URL must start with s3://, got ${url}`);
        }
        return {
            bind(_scope) {
                return { domainFileUrl: url };
            },
        };
    }
    /**
     * Firewall domains created from a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param bucket S3 bucket
     * @param key S3 key
     */
    static fromS3(bucket, key) {
        return this.fromS3Url(bucket.s3UrlForObject(key));
    }
    /**
     * Firewall domains created from a local disk path to a text file.
     * The file must be a text file (`.txt` extension) and must contain a single
     * domain per line. It will be uploaded to S3.
     *
     * @param assetPath path to the text file
     */
    static fromAsset(assetPath) {
        // cdk-assets will correctly set the content type for the S3 object
        // if the file has the correct extension
        if (path.extname(assetPath) !== '.txt') {
            throw new Error(`FirewallDomains.fromAsset() expects a file with the .txt extension, got ${assetPath}`);
        }
        return {
            bind(scope) {
                const asset = new aws_s3_assets_1.Asset(scope, 'Domains', { path: assetPath });
                if (!asset.isFile) {
                    throw new Error('FirewallDomains.fromAsset() expects a file');
                }
                return { domainFileUrl: asset.s3ObjectUrl };
            },
        };
    }
}
exports.FirewallDomains = FirewallDomains;
_a = JSII_RTTI_SYMBOL_1;
FirewallDomains[_a] = { fqn: "@aws-cdk/aws-route53resolver.FirewallDomains", version: "1.158.0" };
/**
 * A Firewall Domain List
 */
class FirewallDomainList extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_route53resolver_FirewallDomainListProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.name && !core_1.Token.isUnresolved(props.name) && !/^[\w-.]{1,128}$/.test(props.name)) {
            throw new Error(`Invalid domain list name: ${props.name}. The name must have 1-128 characters. Valid characters: A-Z, a-z, 0-9, _, -, .`);
        }
        const domainsConfig = props.domains.bind(this);
        const domainList = new route53resolver_generated_1.CfnFirewallDomainList(this, 'Resource', {
            name: props.name,
            domainFileUrl: domainsConfig.domainFileUrl,
            domains: domainsConfig.domains,
        });
        this.firewallDomainListId = domainList.attrId;
        this.firewallDomainListArn = domainList.attrArn;
        this.firewallDomainListCreationTime = domainList.attrCreationTime;
        this.firewallDomainListCreatorRequestId = domainList.attrCreatorRequestId;
        this.firewallDomainListDomainCount = domainList.attrDomainCount;
        this.firewallDomainListManagedOwnerName = domainList.attrManagedOwnerName;
        this.firewallDomainListModificationTime = domainList.attrModificationTime;
        this.firewallDomainListStatus = domainList.attrStatus;
        this.firewallDomainListStatusMessage = domainList.attrStatusMessage;
    }
    /**
     * Import an existing Firewall Rule Group
     */
    static fromFirewallDomainListId(scope, id, firewallDomainListId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.firewallDomainListId = firewallDomainListId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FirewallDomainList = FirewallDomainList;
_b = JSII_RTTI_SYMBOL_1;
FirewallDomainList[_b] = { fqn: "@aws-cdk/aws-route53resolver.FirewallDomainList", version: "1.158.0" };
//# sourceMappingURL=data:application/json;base64,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