"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const dns_validated_certificate_1 = require("../lib/dns-validated-certificate");
module.exports = {
    'creates CloudFormation Custom Resource'(test) {
        const stack = new core_1.Stack();
        const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com'
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            hostedZone: exampleDotComZone,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            DomainName: 'test.example.com',
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertificateCertificateRequestorFunction5E845413',
                    'Arn'
                ]
            },
            HostedZoneId: {
                Ref: 'ExampleDotCom4D1B83AA',
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Handler: 'index.certificateRequestHandler',
            Runtime: 'nodejs10.x',
            Timeout: 900,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyName: 'CertificateCertificateRequestorFunctionServiceRoleDefaultPolicy3C8845BC',
            Roles: [
                {
                    Ref: 'CertificateCertificateRequestorFunctionServiceRoleC04C13DA',
                }
            ],
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: [
                            'acm:RequestCertificate',
                            'acm:DescribeCertificate',
                            'acm:DeleteCertificate'
                        ],
                        Effect: 'Allow',
                        Resource: '*'
                    },
                    {
                        Action: 'route53:GetChange',
                        Effect: 'Allow',
                        Resource: '*'
                    },
                    {
                        Action: 'route53:changeResourceRecordSets',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':route53:::hostedzone/',
                                    { Ref: 'ExampleDotCom4D1B83AA' }
                                ]
                            ]
                        }
                    },
                ],
            }
        }));
        test.done();
    },
    'adds validation error on domain mismatch'(test) {
        const stack = new core_1.Stack();
        const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
            zoneName: 'hello.com'
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'example.com',
            hostedZone: helloDotComZone,
        });
        // a bit of a hack: expect(stack) will trigger validation.
        test.throws(() => assert_1.expect(stack), /DNS zone hello.com is not authoritative for certificate domain name example.com/);
        test.done();
    },
    'test root certificate'(test) {
        const stack = new core_1.Stack();
        const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com'
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'example.com',
            hostedZone: exampleDotComZone,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertCertificateRequestorFunction98FDF273',
                    'Arn'
                ]
            },
            DomainName: 'example.com',
            HostedZoneId: {
                Ref: 'ExampleDotCom4D1B83AA'
            }
        }));
        test.done();
    },
    'works with imported zone'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { account: '12345678', region: 'us-blue-5' },
        });
        const imported = aws_route53_1.HostedZone.fromLookup(stack, 'ExampleDotCom', {
            domainName: 'mydomain.com',
        });
        // WHEN
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'mydomain.com',
            hostedZone: imported,
            route53Endpoint: "https://api.route53.xxx.com",
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertCertificateRequestorFunction98FDF273',
                    'Arn'
                ]
            },
            DomainName: 'mydomain.com',
            HostedZoneId: 'DUMMY',
            Route53Endpoint: 'https://api.route53.xxx.com'
        }));
        test.done();
    },
    'works with imported role'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { account: '12345678', region: 'us-blue-5' },
        });
        const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
            zoneName: 'hello.com'
        });
        const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::account-id:role/role-name');
        // WHEN
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'hello.com',
            hostedZone: helloDotComZone,
            customResourceRole: role
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Role: 'arn:aws:iam::account-id:role/role-name'
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5kbnMtdmFsaWRhdGVkLWNlcnRpZmljYXRlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5kbnMtdmFsaWRhdGVkLWNlcnRpZmljYXRlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBdUQ7QUFDdkQsd0NBQXdDO0FBQ3hDLHNEQUFvRTtBQUNwRSx3Q0FBMkM7QUFFM0MsZ0ZBQTJFO0FBRTNFLGlCQUFTO0lBQ1Asd0NBQXdDLENBQUMsSUFBVTtRQUNqRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE1BQU0saUJBQWlCLEdBQUcsSUFBSSw4QkFBZ0IsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1lBQ3JFLFFBQVEsRUFBRSxhQUFhO1NBQ3hCLENBQUMsQ0FBQztRQUVILElBQUksbURBQXVCLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtZQUNoRCxVQUFVLEVBQUUsa0JBQWtCO1lBQzlCLFVBQVUsRUFBRSxpQkFBaUI7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHFDQUFxQyxFQUFFO1lBQ25FLFVBQVUsRUFBRSxrQkFBa0I7WUFDOUIsWUFBWSxFQUFFO2dCQUNaLFlBQVksRUFBRTtvQkFDWixpREFBaUQ7b0JBQ2pELEtBQUs7aUJBQ047YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixHQUFHLEVBQUUsdUJBQXVCO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7WUFDckQsT0FBTyxFQUFFLGlDQUFpQztZQUMxQyxPQUFPLEVBQUUsWUFBWTtZQUNyQixPQUFPLEVBQUUsR0FBRztTQUNiLENBQUMsQ0FBQyxDQUFDO1FBQ0osZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ2hELFVBQVUsRUFBRSx5RUFBeUU7WUFDckYsS0FBSyxFQUFFO2dCQUNMO29CQUNFLEdBQUcsRUFBRSw0REFBNEQ7aUJBQ2xFO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUU7NEJBQ04sd0JBQXdCOzRCQUN4Qix5QkFBeUI7NEJBQ3pCLHVCQUF1Qjt5QkFDeEI7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFLEdBQUc7cUJBQ2Q7b0JBQ0Q7d0JBQ0UsTUFBTSxFQUFFLG1CQUFtQjt3QkFDM0IsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFLEdBQUc7cUJBQ2Q7b0JBQ0Q7d0JBQ0UsTUFBTSxFQUFFLGtDQUFrQzt3QkFDMUMsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFOzRCQUNSLFVBQVUsRUFBRTtnQ0FDVixFQUFFO2dDQUNGO29DQUNFLE1BQU07b0NBQ04sRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0NBQ3pCLHdCQUF3QjtvQ0FDeEIsRUFBRSxHQUFHLEVBQUUsdUJBQXVCLEVBQUU7aUNBQ2pDOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwQ0FBMEMsQ0FBQyxJQUFVO1FBQ25ELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxlQUFlLEdBQUcsSUFBSSw4QkFBZ0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1lBQ2pFLFFBQVEsRUFBRSxXQUFXO1NBQ3RCLENBQUMsQ0FBQztRQUVILElBQUksbURBQXVCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUN6QyxVQUFVLEVBQUUsYUFBYTtZQUN6QixVQUFVLEVBQUUsZUFBZTtTQUM1QixDQUFDLENBQUM7UUFFSCwwREFBMEQ7UUFDMUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxlQUFNLENBQUMsS0FBSyxDQUFDLEVBQUUsaUZBQWlGLENBQUMsQ0FBQztRQUNwSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUJBQXVCLENBQUMsSUFBVTtRQUNoQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE1BQU0saUJBQWlCLEdBQUcsSUFBSSw4QkFBZ0IsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1lBQ3JFLFFBQVEsRUFBRSxhQUFhO1NBQ3hCLENBQUMsQ0FBQztRQUVILElBQUksbURBQXVCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUN6QyxVQUFVLEVBQUUsYUFBYTtZQUN6QixVQUFVLEVBQUUsaUJBQWlCO1NBQzlCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxxQ0FBcUMsRUFBRTtZQUNqRSxZQUFZLEVBQUU7Z0JBQ2QsWUFBWSxFQUFFO29CQUNaLDBDQUEwQztvQkFDMUMsS0FBSztpQkFDSjthQUNGO1lBQ0QsVUFBVSxFQUFFLGFBQWE7WUFDekIsWUFBWSxFQUFFO2dCQUNaLEdBQUcsRUFBRSx1QkFBdUI7YUFDN0I7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNOLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwQkFBMEIsQ0FBQyxJQUFVO1FBQ25DLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7WUFDcEMsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO1NBQ2xELENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLHdCQUFVLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7WUFDN0QsVUFBVSxFQUFFLGNBQWM7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksbURBQXVCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUN6QyxVQUFVLEVBQUUsY0FBYztZQUMxQixVQUFVLEVBQUUsUUFBUTtZQUNwQixlQUFlLEVBQUUsNkJBQTZCO1NBQy9DLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMscUNBQXFDLEVBQUU7WUFDakUsWUFBWSxFQUFFO2dCQUNkLFlBQVksRUFBRTtvQkFDWiwwQ0FBMEM7b0JBQzFDLEtBQUs7aUJBQ0o7YUFDRjtZQUNELFVBQVUsRUFBRSxjQUFjO1lBQzFCLFlBQVksRUFBRSxPQUFPO1lBQ3JCLGVBQWUsRUFBRSw2QkFBNkI7U0FDL0MsQ0FBQyxDQUFDLENBQUM7UUFFTixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMEJBQTBCLENBQUMsSUFBVTtRQUNuQyxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxFQUFFO1lBQ3BDLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtTQUNsRCxDQUFDLENBQUM7UUFDSCxNQUFNLGVBQWUsR0FBRyxJQUFJLDhCQUFnQixDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDakUsUUFBUSxFQUFFLFdBQVc7U0FDdEIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSx3Q0FBd0MsQ0FBQyxDQUFDO1FBRTNGLE9BQU87UUFDUCxJQUFJLG1EQUF1QixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDekMsVUFBVSxFQUFFLFdBQVc7WUFDdkIsVUFBVSxFQUFFLGVBQWU7WUFDM0Isa0JBQWtCLEVBQUUsSUFBSTtTQUN6QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO1lBQ3JELElBQUksRUFBRSx3Q0FBd0M7U0FDL0MsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEhvc3RlZFpvbmUsIFB1YmxpY0hvc3RlZFpvbmUgfSBmcm9tICdAYXdzLWNkay9hd3Mtcm91dGU1Myc7XG5pbXBvcnQgeyBBcHAsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUgfSBmcm9tICcuLi9saWIvZG5zLXZhbGlkYXRlZC1jZXJ0aWZpY2F0ZSc7XG5cbmV4cG9ydCA9IHtcbiAgJ2NyZWF0ZXMgQ2xvdWRGb3JtYXRpb24gQ3VzdG9tIFJlc291cmNlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGV4YW1wbGVEb3RDb21ab25lID0gbmV3IFB1YmxpY0hvc3RlZFpvbmUoc3RhY2ssICdFeGFtcGxlRG90Q29tJywge1xuICAgICAgem9uZU5hbWU6ICdleGFtcGxlLmNvbSdcbiAgICB9KTtcblxuICAgIG5ldyBEbnNWYWxpZGF0ZWRDZXJ0aWZpY2F0ZShzdGFjaywgJ0NlcnRpZmljYXRlJywge1xuICAgICAgZG9tYWluTmFtZTogJ3Rlc3QuZXhhbXBsZS5jb20nLFxuICAgICAgaG9zdGVkWm9uZTogZXhhbXBsZURvdENvbVpvbmUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDbG91ZEZvcm1hdGlvbjo6Q3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICBEb21haW5OYW1lOiAndGVzdC5leGFtcGxlLmNvbScsXG4gICAgICBTZXJ2aWNlVG9rZW46IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0NlcnRpZmljYXRlQ2VydGlmaWNhdGVSZXF1ZXN0b3JGdW5jdGlvbjVFODQ1NDEzJyxcbiAgICAgICAgICAnQXJuJ1xuICAgICAgICBdXG4gICAgICB9LFxuICAgICAgSG9zdGVkWm9uZUlkOiB7XG4gICAgICAgIFJlZjogJ0V4YW1wbGVEb3RDb200RDFCODNBQScsXG4gICAgICB9XG4gICAgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgICBIYW5kbGVyOiAnaW5kZXguY2VydGlmaWNhdGVSZXF1ZXN0SGFuZGxlcicsXG4gICAgICBSdW50aW1lOiAnbm9kZWpzMTAueCcsXG4gICAgICBUaW1lb3V0OiA5MDAsXG4gICAgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5TmFtZTogJ0NlcnRpZmljYXRlQ2VydGlmaWNhdGVSZXF1ZXN0b3JGdW5jdGlvblNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTNDODg0NUJDJyxcbiAgICAgIFJvbGVzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBSZWY6ICdDZXJ0aWZpY2F0ZUNlcnRpZmljYXRlUmVxdWVzdG9yRnVuY3Rpb25TZXJ2aWNlUm9sZUMwNEMxM0RBJyxcbiAgICAgICAgfVxuICAgICAgXSxcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdhY206UmVxdWVzdENlcnRpZmljYXRlJyxcbiAgICAgICAgICAgICAgJ2FjbTpEZXNjcmliZUNlcnRpZmljYXRlJyxcbiAgICAgICAgICAgICAgJ2FjbTpEZWxldGVDZXJ0aWZpY2F0ZSdcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogJyonXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdyb3V0ZTUzOkdldENoYW5nZScsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogJyonXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdyb3V0ZTUzOmNoYW5nZVJlc291cmNlUmVjb3JkU2V0cycsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgICAgICAgICAgICc6cm91dGU1Mzo6Omhvc3RlZHpvbmUvJyxcbiAgICAgICAgICAgICAgICAgIHsgUmVmOiAnRXhhbXBsZURvdENvbTREMUI4M0FBJyB9XG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkcyB2YWxpZGF0aW9uIGVycm9yIG9uIGRvbWFpbiBtaXNtYXRjaCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBjb25zdCBoZWxsb0RvdENvbVpvbmUgPSBuZXcgUHVibGljSG9zdGVkWm9uZShzdGFjaywgJ0hlbGxvRG90Q29tJywge1xuICAgICAgem9uZU5hbWU6ICdoZWxsby5jb20nXG4gICAgfSk7XG5cbiAgICBuZXcgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUoc3RhY2ssICdDZXJ0Jywge1xuICAgICAgZG9tYWluTmFtZTogJ2V4YW1wbGUuY29tJyxcbiAgICAgIGhvc3RlZFpvbmU6IGhlbGxvRG90Q29tWm9uZSxcbiAgICB9KTtcblxuICAgIC8vIGEgYml0IG9mIGEgaGFjazogZXhwZWN0KHN0YWNrKSB3aWxsIHRyaWdnZXIgdmFsaWRhdGlvbi5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBleHBlY3Qoc3RhY2spLCAvRE5TIHpvbmUgaGVsbG8uY29tIGlzIG5vdCBhdXRob3JpdGF0aXZlIGZvciBjZXJ0aWZpY2F0ZSBkb21haW4gbmFtZSBleGFtcGxlLmNvbS8pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0ZXN0IHJvb3QgY2VydGlmaWNhdGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgZXhhbXBsZURvdENvbVpvbmUgPSBuZXcgUHVibGljSG9zdGVkWm9uZShzdGFjaywgJ0V4YW1wbGVEb3RDb20nLCB7XG4gICAgICB6b25lTmFtZTogJ2V4YW1wbGUuY29tJ1xuICAgIH0pO1xuXG4gICAgbmV3IERuc1ZhbGlkYXRlZENlcnRpZmljYXRlKHN0YWNrLCAnQ2VydCcsIHtcbiAgICAgIGRvbWFpbk5hbWU6ICdleGFtcGxlLmNvbScsXG4gICAgICBob3N0ZWRab25lOiBleGFtcGxlRG90Q29tWm9uZSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpDdXN0b21SZXNvdXJjZScsIHtcbiAgICAgICAgU2VydmljZVRva2VuOiB7XG4gICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICdDZXJ0Q2VydGlmaWNhdGVSZXF1ZXN0b3JGdW5jdGlvbjk4RkRGMjczJyxcbiAgICAgICAgICAnQXJuJ1xuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgRG9tYWluTmFtZTogJ2V4YW1wbGUuY29tJyxcbiAgICAgICAgSG9zdGVkWm9uZUlkOiB7XG4gICAgICAgICAgUmVmOiAnRXhhbXBsZURvdENvbTREMUI4M0FBJ1xuICAgICAgICB9XG4gICAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3dvcmtzIHdpdGggaW1wb3J0ZWQgem9uZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJywge1xuICAgICAgZW52OiB7IGFjY291bnQ6ICcxMjM0NTY3OCcsIHJlZ2lvbjogJ3VzLWJsdWUtNScgfSxcbiAgICB9KTtcbiAgICBjb25zdCBpbXBvcnRlZCA9IEhvc3RlZFpvbmUuZnJvbUxvb2t1cChzdGFjaywgJ0V4YW1wbGVEb3RDb20nLCB7XG4gICAgICBkb21haW5OYW1lOiAnbXlkb21haW4uY29tJyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUoc3RhY2ssICdDZXJ0Jywge1xuICAgICAgZG9tYWluTmFtZTogJ215ZG9tYWluLmNvbScsXG4gICAgICBob3N0ZWRab25lOiBpbXBvcnRlZCxcbiAgICAgIHJvdXRlNTNFbmRwb2ludDogXCJodHRwczovL2FwaS5yb3V0ZTUzLnh4eC5jb21cIixcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDbG91ZEZvcm1hdGlvbjo6Q3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICAgIFNlcnZpY2VUb2tlbjoge1xuICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAnQ2VydENlcnRpZmljYXRlUmVxdWVzdG9yRnVuY3Rpb245OEZERjI3MycsXG4gICAgICAgICAgJ0FybidcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIERvbWFpbk5hbWU6ICdteWRvbWFpbi5jb20nLFxuICAgICAgICBIb3N0ZWRab25lSWQ6ICdEVU1NWScsXG4gICAgICAgIFJvdXRlNTNFbmRwb2ludDogJ2h0dHBzOi8vYXBpLnJvdXRlNTMueHh4LmNvbSdcbiAgICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd3b3JrcyB3aXRoIGltcG9ydGVkIHJvbGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdTdGFjaycsIHtcbiAgICAgIGVudjogeyBhY2NvdW50OiAnMTIzNDU2NzgnLCByZWdpb246ICd1cy1ibHVlLTUnIH0sXG4gICAgfSk7XG4gICAgY29uc3QgaGVsbG9Eb3RDb21ab25lID0gbmV3IFB1YmxpY0hvc3RlZFpvbmUoc3RhY2ssICdIZWxsb0RvdENvbScsIHtcbiAgICAgIHpvbmVOYW1lOiAnaGVsbG8uY29tJ1xuICAgIH0pO1xuICAgIGNvbnN0IHJvbGUgPSBpYW0uUm9sZS5mcm9tUm9sZUFybihzdGFjaywgJ1JvbGUnLCAnYXJuOmF3czppYW06OmFjY291bnQtaWQ6cm9sZS9yb2xlLW5hbWUnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUoc3RhY2ssICdDZXJ0Jywge1xuICAgICAgZG9tYWluTmFtZTogJ2hlbGxvLmNvbScsXG4gICAgICBob3N0ZWRab25lOiBoZWxsb0RvdENvbVpvbmUsXG4gICAgICBjdXN0b21SZXNvdXJjZVJvbGU6IHJvbGVcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgUm9sZTogJ2Fybjphd3M6aWFtOjphY2NvdW50LWlkOnJvbGUvcm9sZS1uYW1lJ1xuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxufTtcbiJdfQ==