"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'apex domain selection by default'(test) {
        const stack = new core_1.Stack();
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com'
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'test.example.com',
            DomainValidationOptions: [{
                    DomainName: "test.example.com",
                    ValidationDomain: "example.com"
                }]
        }));
        test.done();
    },
    'validation domain can be overridden'(test) {
        const stack = new core_1.Stack();
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            validationDomains: {
                'test.example.com': 'test.example.com'
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CertificateManager::Certificate', {
            DomainValidationOptions: [{
                    DomainName: "test.example.com",
                    ValidationDomain: "test.example.com"
                }]
        }));
        test.done();
    },
    'export and import'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const c = lib_1.Certificate.fromCertificateArn(stack, 'Imported', 'cert-arn');
        // THEN
        test.deepEqual(c.certificateArn, 'cert-arn');
        test.done();
    },
    'can configure validation method'(test) {
        const stack = new core_1.Stack();
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            validationMethod: lib_1.ValidationMethod.DNS
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'test.example.com',
            ValidationMethod: 'DNS',
        }));
        test.done();
    },
    'needs validation domain supplied if domain contains a token'(test) {
        const stack = new core_1.Stack();
        test.throws(() => {
            const domainName = core_1.Lazy.stringValue({ produce: () => 'example.com' });
            new lib_1.Certificate(stack, 'Certificate', {
                domainName,
            });
        }, /'validationDomains' needs to be supplied/);
        test.done();
    },
    'validationdomains can be given for a Token'(test) {
        const stack = new core_1.Stack();
        const domainName = core_1.Lazy.stringValue({ produce: () => 'my.example.com' });
        new lib_1.Certificate(stack, 'Certificate', {
            domainName,
            validationDomains: {
                [domainName]: 'example.com'
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'my.example.com',
            DomainValidationOptions: [{
                    DomainName: 'my.example.com',
                    ValidationDomain: 'example.com'
                }]
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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