"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * (experimental) Elasticsearch version.
 *
 * @experimental
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * (experimental) Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @experimental
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/**
 * (experimental) AWS Elasticsearch 1.5.
 *
 * @experimental
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * (experimental) AWS Elasticsearch 2.3.
 *
 * @experimental
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * (experimental) AWS Elasticsearch 5.1.
 *
 * @experimental
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * (experimental) AWS Elasticsearch 5.3.
 *
 * @experimental
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * (experimental) AWS Elasticsearch 5.5.
 *
 * @experimental
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * (experimental) AWS Elasticsearch 5.6.
 *
 * @experimental
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * (experimental) AWS Elasticsearch 6.0.
 *
 * @experimental
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * (experimental) AWS Elasticsearch 6.2.
 *
 * @experimental
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * (experimental) AWS Elasticsearch 6.3.
 *
 * @experimental
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * (experimental) AWS Elasticsearch 6.4.
 *
 * @experimental
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * (experimental) AWS Elasticsearch 6.5.
 *
 * @experimental
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * (experimental) AWS Elasticsearch 6.7.
 *
 * @experimental
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * (experimental) AWS Elasticsearch 6.8.
 *
 * @experimental
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * (experimental) AWS Elasticsearch 7.1.
 *
 * @experimental
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * (experimental) AWS Elasticsearch 7.4.
 *
 * @experimental
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * (experimental) AWS Elasticsearch 7.7.
 *
 * @experimental
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * (experimental) AWS Elasticsearch 7.8.
 *
 * @experimental
 */
ElasticsearchVersion.V7_8 = ElasticsearchVersion.of('7.8');
/**
 * (experimental) AWS Elasticsearch 7.9.
 *
 * @experimental
 */
ElasticsearchVersion.V7_9 = ElasticsearchVersion.of('7.9');
/**
 * (experimental) The minimum TLS version required for traffic to the domain.
 *
 * @experimental
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * (experimental) Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Return the given named metric for this Domain.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @experimental
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @experimental
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @experimental
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * (experimental) Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * (experimental) Provides an Elasticsearch domain.
 *
 * @experimental
 */
class Domain extends DomainBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31, _32, _33, _34;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_r = props.vpcOptions) === null || _r === void 0 ? void 0 : _r.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master and data node instance types must end with ".elasticsearch".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_s = props.useUnsignedBasicAuth) !== null && _s !== void 0 ? _s : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_t = props.encryptionAtRest) === null || _t === void 0 ? void 0 : _t.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_u = props.fineGrainedAccessControl) === null || _u === void 0 ? void 0 : _u.masterUserArn;
        const masterUserNameProps = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_y = (_x = props.encryptionAtRest) === null || _x === void 0 ? void 0 : _x.enabled) !== null && _y !== void 0 ? _y : (((_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_0 = props.nodeToNodeEncryption) !== null && _0 !== void 0 ? _0 : unsignedBasicAuthEnabled;
        const volumeSize = (_2 = (_1 = props.ebs) === null || _1 === void 0 ? void 0 : _1.volumeSize) !== null && _2 !== void 0 ? _2 : 10;
        const volumeType = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeType) !== null && _4 !== void 0 ? _4 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.enabled) !== null && _6 !== void 0 ? _6 : true;
        const enforceHttps = (_7 = props.enforceHttps) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_8 = props.logging) === null || _8 === void 0 ? void 0 : _8.slowIndexLogEnabled) || ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled)
                || ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_13 = props.logging) === null || _13 === void 0 ? void 0 : _13.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_14 = props.logging.slowSearchLogGroup) !== null && _14 !== void 0 ? _14 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_15 = props.logging) === null || _15 === void 0 ? void 0 : _15.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_16 = props.logging.slowIndexLogGroup) !== null && _16 !== void 0 ? _16 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_17 = props.logging) === null || _17 === void 0 ? void 0 : _17.appLogEnabled) {
            this.appLogGroup = (_18 = props.logging.appLogGroup) !== null && _18 !== void 0 ? _18 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_19 = props.logging) === null || _19 === void 0 ? void 0 : _19.auditLogEnabled) {
            this.auditLogGroup = (_20 = props.logging.auditLogGroup) !== null && _20 !== void 0 ? _20 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_21 = props.ebs) === null || _21 === void 0 ? void 0 : _21.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_23 = (_22 = props.encryptionAtRest) === null || _22 === void 0 ? void 0 : _22.kmsKey) === null || _23 === void 0 ? void 0 : _23.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: {
                AUDIT_LOGS: {
                    enabled: this.auditLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_24 = this.auditLogGroup) === null || _24 === void 0 ? void 0 : _24.logGroupArn,
                },
                ES_APPLICATION_LOGS: {
                    enabled: this.appLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_25 = this.appLogGroup) === null || _25 === void 0 ? void 0 : _25.logGroupArn,
                },
                SEARCH_SLOW_LOGS: {
                    enabled: this.slowSearchLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_26 = this.slowSearchLogGroup) === null || _26 === void 0 ? void 0 : _26.logGroupArn,
                },
                INDEX_SLOW_LOGS: {
                    enabled: this.slowIndexLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_27 = this.slowIndexLogGroup) === null || _27 === void 0 ? void 0 : _27.logGroupArn,
                },
            },
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_28 = props.cognitoKibanaAuth) === null || _28 === void 0 ? void 0 : _28.identityPoolId,
                roleArn: (_29 = props.cognitoKibanaAuth) === null || _29 === void 0 ? void 0 : _29.role.roleArn,
                userPoolId: (_30 = props.cognitoKibanaAuth) === null || _30 === void 0 ? void 0 : _30.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_31 = props.tlsSecurityPolicy) !== null && _31 !== void 0 ? _31 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_32 = this.masterUserPassword) === null || _32 === void 0 ? void 0 : _32.toString(),
                    },
                }
                : undefined,
        });
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_33 = props.accessPolicies) !== null && _33 !== void 0 ? _33 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            if ((_34 = props.encryptionAtRest) === null || _34 === void 0 ? void 0 : _34.kmsKey) {
                // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/encryption-at-rest.html
                // these permissions are documented as required during domain creation.
                // while not strictly documented for updates as well, it stands to reason that an update
                // operation might require these in case the cluster uses a kms key.
                // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                    actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                    resources: [props.encryptionAtRest.kmsKey.keyArn],
                    effect: iam.Effect.ALLOW,
                }));
            }
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @experimental
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @experimental
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,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