"use strict";
const assert_1 = require("@aws-cdk/assert");
const kinesis = require("@aws-cdk/aws-kinesis");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'sufficiently complex example'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON,
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'kinesis:DescribeStreamSummary',
                            'kinesis:GetRecords',
                            'kinesis:GetShardIterator',
                            'kinesis:ListShards',
                            'kinesis:SubscribeToShard',
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'S509448A1',
                                'Arn',
                            ],
                        },
                    },
                    {
                        'Action': 'kinesis:DescribeStream',
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'S509448A1',
                                'Arn',
                            ],
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'FnServiceRoleDefaultPolicyC6A839BF',
            'Roles': [{
                    'Ref': 'FnServiceRoleB9001A96',
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            'EventSourceArn': {
                'Fn::GetAtt': [
                    'S509448A1',
                    'Arn',
                ],
            },
            'FunctionName': {
                'Ref': 'Fn9270CBC0',
            },
            'BatchSize': 100,
            'StartingPosition': 'TRIM_HORIZON',
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST,
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            'EventSourceArn': {
                'Fn::GetAtt': [
                    'S509448A1',
                    'Arn',
                ],
            },
            'FunctionName': {
                'Ref': 'Fn9270CBC0',
            },
            'BatchSize': 50,
            'StartingPosition': 'LATEST',
        }));
        test.done();
    },
    'fails if batch size < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        test.throws(() => fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 0,
            startingPosition: lambda.StartingPosition.LATEST,
        })), /Maximum batch size must be between 1 and 10000 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size > 10000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        test.throws(() => fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 10001,
            startingPosition: lambda.StartingPosition.LATEST,
        })), /Maximum batch size must be between 1 and 10000 inclusive \(given 10001\)/);
        test.done();
    },
    'specific maxBatchingWindow'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            maxBatchingWindow: cdk.Duration.minutes(2),
            startingPosition: lambda.StartingPosition.LATEST,
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            'EventSourceArn': {
                'Fn::GetAtt': [
                    'S509448A1',
                    'Arn',
                ],
            },
            'FunctionName': {
                'Ref': 'Fn9270CBC0',
            },
            'MaximumBatchingWindowInSeconds': 120,
            'StartingPosition': 'LATEST',
        }));
        test.done();
    },
    'contains eventSourceMappingId after lambda binding'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        const eventSource = new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON,
        });
        // WHEN
        fn.addEventSource(eventSource);
        // THEN
        test.ok(eventSource.eventSourceMappingId);
        test.done();
    },
    'eventSourceMappingId throws error before binding to lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const stream = new kinesis.Stream(stack, 'S');
        const eventSource = new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON,
        });
        // WHEN/THEN
        test.throws(() => eventSource.eventSourceMappingId, /KinesisEventSource is not yet bound to an event source mapping/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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