"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
function deployLambdaToOpenSearch(stack) {
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain'
    };
    return new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
}
function getDefaultTestLambdaProps() {
    return {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
    };
}
test('Check pattern properties', () => {
    const stack = new cdk.Stack();
    const construct = deployLambdaToOpenSearch(stack);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.openSearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudWatchAlarms).toBeDefined();
    expect(construct.openSearchRole).toBeDefined();
});
test('Check properties with no CloudWatch alarms ', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.openSearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudWatchAlarms).toBeUndefined();
    expect(construct.openSearchRole).toBeDefined();
});
test('Check for an existing Lambda object', () => {
    const stack = new cdk.Stack();
    const existingLambdaObj = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            functionName: 'test-function'
        }
    });
    const props = {
        openSearchDomainName: 'test-domain',
        existingLambdaObj
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'test-function'
    });
});
test('Check Lambda function custom environment variable', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        domainEndpointEnvironmentVariableName: 'CUSTOM_DOMAIN_ENDPOINT'
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_DOMAIN_ENDPOINT: {
                    'Fn::GetAtt': [
                        'testlambdaopensearchstackOpenSearchDomain46D6A86E',
                        'DomainEndpoint'
                    ]
                }
            }
        }
    });
});
test('Check domain name', () => {
    const stack = new cdk.Stack();
    deployLambdaToOpenSearch(stack);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain",
        UserPoolId: {
            Ref: "testlambdaopensearchstackCognitoUserPoolF5169460"
        }
    });
    expect(stack).toHaveResource('AWS::OpenSearchService::Domain', {
        DomainName: "test-domain",
    });
});
test('Check cognito domain name override', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        cognitoDomainName: 'test-cogn-domain'
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: 'test-cogn-domain'
    });
});
test('Check engine version override', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        openSearchDomainProps: {
            engineVersion: 'OpenSearch_1.0',
        }
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    expect(stack).toHaveResource('AWS::OpenSearchService::Domain', {
        EngineVersion: 'OpenSearch_1.0'
    });
});
test("Test minimal deployment that deploys a VPC in 2 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {});
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaopensearchstackReplaceDefaultSecurityGroupsecuritygroup293B90D7",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                }
            ],
        },
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test minimal deployment that deploys a VPC in 3 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaopensearchstackReplaceDefaultSecurityGroupsecuritygroup293B90D7",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D",
                },
            ],
        },
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test cluster deploy to 1 AZ when user set zoneAwarenessEnabled to false", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const openSearchDomainProps = {
        clusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            zoneAwarenessEnabled: false,
            instanceCount: 3
        }
    };
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        openSearchDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 1
        }
    });
    expect(stack).toHaveResource("AWS::OpenSearchService::Domain", {
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: false,
        }
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                }
            ]
        }
    });
});
test("Test cluster deploy to 2 AZ when user set availabilityZoneCount to 2", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const openSearchDomainProps = {
        clusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            instanceCount: 2,
            zoneAwarenessEnabled: true,
            zoneAwarenessConfig: {
                availabilityZoneCount: 2
            }
        }
    };
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        openSearchDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 2
        }
    });
    expect(stack).toHaveResource("AWS::OpenSearchService::Domain", {
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 2,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 2,
            },
            ZoneAwarenessEnabled: true,
        }
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
});
test('Test minimal deployment with an existing isolated VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack, false, {
        vpcName: "existing-isolated-vpc-test"
    });
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: "test",
        existingVpc: vpc
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-isolated-vpc-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with an existing private VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = core_1.getTestVpc(stack, true, {
        vpcName: "existing-private-vpc-test"
    });
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: "test",
        existingVpc: vpc
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-private-vpc-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcPrivateSubnet1Subnet536B997A"
                },
                {
                    Ref: "VpcPrivateSubnet2Subnet3788AAA1"
                },
                {
                    Ref: "VpcPrivateSubnet3SubnetF258B56E"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with VPC construct props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: "test",
        deployVpc: true,
        vpcProps: {
            vpcName: "vpc-props-test"
        }
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "vpc-props-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test error for vpcProps and undefined deployVpc prop', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            openSearchDomainName: "test",
            vpcProps: {
                vpcName: "existing-vpc-test"
            }
        });
    };
    expect(app).toThrowError("Error - deployVpc must be true when defining vpcProps");
});
test('Test error for Lambda function VPC props', () => {
    const stack = new cdk.Stack();
    const vpc = defaults.getTestVpc(stack);
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: defaults.consolidateProps(getDefaultTestLambdaProps(), { vpc }),
            openSearchDomainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not Lambda function props");
});
test('Test error for the OpenSearch domain VPC props', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            openSearchDomainProps: {
                vpcOptions: {
                    subnetIds: ['fake-ids'],
                    securityGroupIds: ['fake-sgs']
                }
            },
            openSearchDomainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not the OpenSearch Service props");
});
test('Test error for missing existingLambdaObj or lambdaFunctionProps', () => {
    const stack = new cdk.Stack();
    const props = {
        openSearchDomainName: 'test-domain'
    };
    try {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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