"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventsRuleToSqs = void 0;
const events = require("@aws-cdk/aws-events");
const defaults = require("@aws-solutions-constructs/core");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
class EventsRuleToSqs extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the EventsRuleToSqs class.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        let enableEncryptionParam = props.enableEncryptionWithCustomerManagedKey;
        if (props.enableEncryptionWithCustomerManagedKey === undefined ||
            props.enableEncryptionWithCustomerManagedKey === true) {
            enableEncryptionParam = true;
        }
        // Setup the queue
        [this.sqsQueue, this.encryptionKey] = defaults.buildQueue(this, 'queue', {
            existingQueueObj: props.existingQueueObj,
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue,
            enableEncryptionWithCustomerManagedKey: enableEncryptionParam,
            encryptionKey: props.encryptionKey,
            encryptionKeyProps: props.encryptionKeyProps
        });
        const sqsEventTarget = {
            bind: () => ({
                id: this.sqsQueue.queueName,
                arn: this.sqsQueue.queueArn
            })
        };
        const defaultEventsRuleProps = defaults.DefaultEventsRuleProps([sqsEventTarget]);
        const eventsRuleProps = core_2.overrideProps(defaultEventsRuleProps, props.eventRuleProps, true);
        this.eventsRule = new events.Rule(this, 'EventsRule', eventsRuleProps);
        // Enable queue purging permissions for the event rule, if enabled
        if (props.enableQueuePurging) {
            this.sqsQueue.grantPurge(new aws_iam_1.ServicePrincipal('events.amazonaws.com'));
        }
        // Policy for event to be able to send messages to the queue and Grant Event Bridge service access to the SQS queue encryption key
        this.sqsQueue.grantSendMessages(new aws_iam_1.ServicePrincipal('events.amazonaws.com'));
    }
}
exports.EventsRuleToSqs = EventsRuleToSqs;
//# sourceMappingURL=data:application/json;base64,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