"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ram_1 = require("aws-cdk-lib/aws-ram");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const validation_1 = require("./private/validation");
const aws_servicecatalogappregistry_1 = require("aws-cdk-lib/aws-servicecatalogappregistry");
const APPLICATION_READ_ONLY_RAM_PERMISSION_ARN = 'arn:aws:ram::aws:permission/AWSRAMPermissionServiceCatalogAppRegistryApplicationReadOnly';
const APPLICATION_ALLOW_ACCESS_RAM_PERMISSION_ARN = 'arn:aws:ram::aws:permission/AWSRAMPermissionServiceCatalogAppRegistryApplicationAllowAssociation';
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedAttributeGroups = new Set();
        this.associatedResources = new Set();
    }
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup) {
        if (!this.associatedAttributeGroups.has(attributeGroup.node.addr)) {
            const hashId = this.generateUniqueHash(attributeGroup.node.addr);
            new aws_servicecatalogappregistry_1.CfnAttributeGroupAssociation(this, `AttributeGroupAssociation${hashId}`, {
                application: this.applicationId,
                attributeGroup: attributeGroup.attributeGroupId,
            });
            this.associatedAttributeGroups.add(attributeGroup.node.addr);
        }
    }
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     */
    associateStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            const hashId = this.generateUniqueHash(stack.node.addr);
            new aws_servicecatalogappregistry_1.CfnResourceAssociation(this, `ResourceAssociation${hashId}`, {
                application: this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
        }
    }
    /**
     * Share an application with accounts, organizations and OUs, and IAM roles and users.
     * The application will become available to end users within those principals.
     *
     * @param shareOptions The options for the share.
     */
    shareApplication(shareOptions) {
        const principals = common_1.getPrincipalsforSharing(shareOptions);
        const shareName = `RAMShare${common_1.hashValues(aws_cdk_lib_1.Names.nodeUniqueId(this.node), this.node.children.length.toString())}`;
        new aws_ram_1.CfnResourceShare(this, shareName, {
            name: shareName,
            allowExternalPrincipals: false,
            principals: principals,
            resourceArns: [this.applicationArn],
            permissionArns: [this.getApplicationSharePermissionARN(shareOptions)],
        });
    }
    /**
     * Get the correct permission ARN based on the SharePermission
     */
    getApplicationSharePermissionARN(shareOptions) {
        switch (shareOptions.sharePermission) {
            case common_1.SharePermission.ALLOW_ACCESS:
                return APPLICATION_ALLOW_ACCESS_RAM_PERMISSION_ARN;
            case common_1.SharePermission.READ_ONLY:
                return APPLICATION_READ_ONLY_RAM_PERMISSION_ARN;
            default:
                return shareOptions.sharePermission ?? APPLICATION_READ_ONLY_RAM_PERMISSION_ARN;
        }
    }
}
/**
 * A Service Catalog AppRegistry Application.
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalogappregistry_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        this.validateApplicationProps(props);
        const application = new aws_servicecatalogappregistry_1.CfnApplication(this, 'Resource', {
            name: props.applicationName,
            description: props.description,
        });
        this.applicationArn = application.attrArn;
        this.applicationId = application.attrId;
        this.nodeAddress = cdk.Names.nodeUniqueId(application.node);
    }
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const arn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME);
        const applicationId = arn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required Application ID from Application ARN: ' + applicationArn);
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationArn = applicationArn;
                this.applicationId = applicationId;
            }
            generateUniqueHash(resourceAddress) {
                return common_1.hashValues(this.applicationArn, resourceAddress);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: applicationArn,
        });
    }
    generateUniqueHash(resourceAddress) {
        return common_1.hashValues(this.nodeAddress, resourceAddress);
    }
    validateApplicationProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'application name', 1, 256, props.applicationName);
        validation_1.InputValidator.validateRegex(this.node.path, 'application name', /^[a-zA-Z0-9-_]+$/, props.applicationName);
        validation_1.InputValidator.validateLength(this.node.path, 'application description', 0, 1024, props.description);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-servicecatalogappregistry-alpha.Application", version: "2.44.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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