"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Task = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const chain_1 = require("../chain");
const fields_1 = require("../fields");
const util_1 = require("../util");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Task state in the state machine
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 *
 * @deprecated - replaced by service integration specific classes (i.e. LambdaInvoke, SnsPublish)
 */
class Task extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.timeout = props.timeout;
        const taskProps = props.task.bind(this);
        this.taskProps = {
            ...taskProps,
            parameters: util_1.noEmptyObject({ ...taskProps.parameters || {}, ...props.parameters || {} }),
        };
        this.endStates = [this];
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderInputOutput(),
            Type: state_type_1.StateType.TASK,
            Comment: this.comment,
            Resource: this.taskProps.resourceArn,
            Parameters: this.taskProps.parameters && fields_1.FieldUtils.renderObject(this.taskProps.parameters),
            ResultPath: state_1.renderJsonPath(this.resultPath),
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
            HeartbeatSeconds: this.taskProps.heartbeat && this.taskProps.heartbeat.toSeconds(),
        };
    }
    /**
     * Return the given named metric for this Task
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: this.taskProps.metricDimensions,
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricRunTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default average over 5 minutes
     */
    metricScheduleTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default sum over 5 minutes
     */
    metricScheduled(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Started', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Failed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskProps.policyStatements || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('This Task Resource does not expose metrics');
        }
        return this.metric(prefix + suffix, props);
    }
}
exports.Task = Task;
//# sourceMappingURL=data:application/json;base64,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