# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['asynkit', 'asynkit.experimental']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=4.4.0,<5.0.0']

setup_kwargs = {
    'name': 'asynkit',
    'version': '0.6.0',
    'description': 'Async toolkit for advanced scheduling',
    'long_description': '# asynkit: a toolkit for coroutines\n\n[![CI](https://github.com/kristjanvalur/py-asynkit/actions/workflows/ci.yml/badge.svg)](https://github.com/kristjanvalur/py-asynkit/actions/workflows/ci.yml)\n\nThis module provides some handy tools for those wishing to have better control over the\nway Python\'s `asyncio` module does things.\n\n- Helper tools for controlling coroutine execution, such as `CoroStart` and `Monitor`\n- Utility classes such as `GeneratorObject`\n- `asyncio` event-loop extensions\n- _eager_ execution of Tasks\n- Limited support for `anyio` and `trio`.\n\n# Installation\n\n```bash\n$ pip install asynkit\n```\n\n# Coroutine Tools\n\n## `eager()` - lower latency IO\n\nDid you ever wish that your _coroutines_ started right away, and only returned control to\nthe caller once they become blocked?  Like the way the `async` and `await` keywords work in the __C#__ language?\n\nNow they can. Just decorate or convert them with `acynkit.eager`:\n\n```python\n@asynkit.eager\nasync def get_slow_remote_data():\n    result = await execute_remote_request()\n    return result.important_data\n\nasync def my_complex_thing():\n    # kick off the request as soon as possible\n    future = get_slow_remote_data()\n    # The remote execution may now already be in flight. Do some work taking time\n    intermediate_result = await some_local_computation()\n    # wait for the result of the request\n    return compute_result(intermediate_result, await future)\n```\n\nBy decorating your function with `eager`, the coroutine will start executing __right away__ and\ncontrol will return to the calling function as soon as it _suspends_, _returns_, or _raises_\nan exception. In case it is suspended, a _Task_ is created and returned, ready to resume\nexecution from that point.\n\nNotice how, in either case, control is returned __directly__ back to the\ncalling function, maintaining synchronous execution. In effect, conventional code\ncalling order is maintained as much as possible. We call this _depth-first-execution_.\n\nThis allows you to prepare and dispatch long running operations __as soon as possible__ while\nstill being able to asynchronously wait for the result.\n\n`asynckit.eager` can also be used directly on the returned coroutine:\n```python\nlog = []\nasync def test():\n    log.append(1)\n    await asyncio.sleep(0.2) # some long IO\n    log.append(2)\n\nasync def caller(convert):\n    del log[:]\n    log.append("a")\n    future = convert(test())\n    log.append("b")\n    await asyncio.sleep(0.1) # some other IO\n    log.append("c")\n    await future\n\n# do nothing\nasyncio.run(caller(lambda c:c))\nassert log == ["a", "b", "c", 1, 2]\n\n# Create a Task\nasyncio.run(caller(asyncio.create_task))\nassert log == ["a", "b", 1, "c", 2]\n\n# eager\nasyncio.run(caller(asynkit.eager))\nassert log == ["a", 1, "b", "c", 2]\n```\n\n`eager()` is actually a convenience function, invoking either `coro_eager()` or `func_eager()` (see below) depending on context.\nDecorating your function makes sense if you __always__ intend\nTo _await_ its result at some later point. Otherwise, just apply it at the point\nof invocation in each such case. \n\n## `coro_eager()`, `func_eager()`\n\n`coro_eager()` is the magic coroutine wrapper providing the __eager__ behaviour:\n\n1. It copies the current _context_\n1. It initializes a `CoroStart()` object for the coroutine, starting it in the copied context.\n2. If it subsequently is `done()` It returns `CoroStart.as_future()`, ortherwise\n   it creates and returns a `Task` (using `asyncio.create_task` by default.)\n\nThe result is an _awaitable_ which can be either directly awaited or passed\nto `asyncio.gather()`. The coroutine is executed in its own copy of the current context,\njust as would happen if it were directly turned into a `Task`.\n\n`func_eager()` is a decorator which automatically applies `coro_eager()` to the coroutine returned by an async function.\n\n## `CoroStart`\n\nThis class manages the state of a partially run coroutine and is what what powers the `coro_eager()` function. \nWhen initialized, it will _start_ the coroutine, running it until it either suspends, returns, or raises\nan exception.\n\nSimilarly to a `Future`, it has these methods:\n\n- `done()` - returns `True` if the coroutine finished without blocking. In this case, the following two methods may be called to get the result.\n- `result()` - Returns the _return value_ of the coroutine or **raises** any _exception_ that it produced.\n- `exception()` - Returns any _exception_ raised, or `None` otherwise.\n\n But more importly it has these:\n\n- `as_coroutine()` - Returns an coroutine encapsulating the original coroutine\'s _continuation_.\n  If it has already finished, awaiting this coroutine is the same as calling `result()`, otherwise it continues the original coroutine\'s execution.\n- `as_future()` - If `done()`, returns a `Future` holding its result, otherwise, a `RuntimeError`\n  is raised. This is suitable for using with\n  `asyncio.gather()` to avoid wrapping the result of an already completed coroutine into a `Task`.\n\nCoroStart can be provided with a `contextvars.Context` object, in which case the coroutine will be run using that\ncontext.\n\n## Context helper\n\n`coro_await()` is a helper function to await a coroutine, optionally with a `contextvars.Context`\nobject to activate:\n\n```python\nvar1 = contextvars.ContextVar("myvar")\n\nasync def my_method():\n    var1.set("foo")\n    \nasync def main():\n    context=contextvars.copy_context()\n    var1.set("bar")\n    await asynkit.coro_await(my_method(), context=context)\n    # the coroutine didn\'t modify _our_ context\n    assert var1.get() == "bar"\n    # ... but it did modify the copied context\n    assert context.get(var1) == "foo"\n```\n\nThis is similar to `contextvars.Context.run()` but works for async functions.  This function is\nimplemented using `CoroStart`\n\n## `Monitor`\n\nA `Monitor` object can be used to await a coroutine, while listening for _out of band_ messages\nfrom the coroutine.  As the coroutine sends messages, it is suspended, until the caller resumes\nawaiting for it.\n\n```python\nasync def coro(monitor):\n    await monitor.oob("hello")\n    await asyncio.sleep(0)\n    await monitor.oob("dolly")\n    return "done"\n\nasync def runner():\n    m = Monitor()\n    c = coro(m)\n    while True:\n        try:\n            print(await m.aawait(c))\n        except OOBData as oob:\n            print(oob.data)\n```\n\nwhich will result in the output\n\n```\nhello\ndolly\ndone\n```\n\nThe caller can also pass in data to the coroutine via the `Monitor.aawait(coro, data:None)` method and\nit will become the result of the `Monitor.oob()` call inside the monitor.   `Monitor.athrow()` can be\nused to raise an exception inside the coroutine.\n\nA Monitor can be used when a coroutine wants to suspend itself, maybe waiting for some extenal\ncondition, without resorting to the relatively heavy mechanism of creating, managing and synchronizing\n`Task` objects.  This can be useful if the coroutine needs to maintain state.\n\nConsider the following scenario. A _parser_ wants to read a line from a buffer, but fails, signalling\nthis to the monitor:\n\n```python\n    async def readline(m, buffer):\n        l = buffer.readline()\n        while not l.endswith("\\n"):\n            await m.oob(None)  # ask for more data in the buffer\n            l += buffer.readline()\n        return l\n\n    async def manager(buffer, io):\n        m = Monitor()\n        c = readline(m, buffer)\n        while True:\n            try:\n                return await m.aawait(c)\n            except OOBData:\n                try:\n                    buffer.fill(await io.read())\n                except Exception as exc:\n                    await m.athrow(c, exc)\n```\n\nIn this example, `readline()` is trivial, but if this were a complicated parser with hierarchical\ninvocation structure, then this pattern allows the decoupling of IO and the parsing of buffered data, maintaining the state of the parser while _the caller_ fills up the buffer.\n\n## `GeneratorObject`\n\nA GeneratorObject builds on top of the `Monitor` to create an `AsyncGenerator`.  It is in many ways\nsimilar to an _asynchronous generator_ constructed using the _generator function_ syntax.\nBut wheras those return values using the `yield` keyword,\na GeneratorObject has an `ayield()` method, which means that data can be sent to the generator\nby anyone.\nIt leverages the `Monitor.oob()` method to deliver the yielded data to whomever is iterating over it:\n\n```python\nasync def generator(gen_obj):\n    # yield directly to the generator\n    await gen_obj.ayield(1):\n    # have someone else yield to it\n    async def helper():\n        await gen_obj.ayield(2)\n    await asyncio.create_task(helper())\n\nasync def runner():\n    gen_obj = GeneratorObject()\n    values = [val async for val in gen_obj(generator(gen_obj))]\n    assert values == [1, 2]\n```\n\nThe `GeneratorObject`, when called, returns a `GeneratorObjectIterator` which behaves in\nthe same way as an `AsyncGenerator` object.  It can be iterated over and supports the\n`asend()`, `athrow()` and `aclose()` methods.\n\nA GeneratorObject is a flexible way to asynchronously generate results without\nresorting to Tasks and Queues.\n\n\n# Event loop tools\n\nAlso provided is a mixin for the built-in event loop implementations in python, providing some primitives for advanced\nscheduling of tasks.\n\n## `SchedulingMixin` mixin class\n\nThis class adds some handy scheduling functions to the event loop. They primarily\nwork with the _ready queue_, a queue of callbacks representing tasks ready\nto be executed.\n\n- `ready_len(self)` - returns the length of the ready queue\n- `ready_pop(self, pos=-1)` - pops an entry off the queue\n- `ready_insert(self, pos, element)` - inserts a previously popped element into the queue\n- `ready_rotate(self, n)` - rotates the queue\n- `call_insert(self, pos, ...)` - schedules a callback at position `pos` in the queue\n\n## Concrete event loop classes\n\nConcrete subclasses of Python\'s built-in event loop classes are provided.\n\n- `SchedulingSelectorEventLoop` is a subclass of `asyncio.SelectorEventLoop` with the `SchedulingMixin`\n- `SchedulingProactorEventLoop` is a subclass of `asyncio.ProactorEventLoop` with the `SchedulingMixin` on those platforms that support it.\n\n## Event Loop Policy\n\nA policy class is provided to automatically create the appropriate event loops.\n\n- `SchedulingEventLoopPolicy` is a subclass of `asyncio.DefaultEventLoopPolicy` which instantiates either of the above event loop classes as appropriate.\n\nUse this either directly:\n\n```python\nasyncio.set_event_loop_policy(asynkit.SchedulingEventLoopPolicy())\nasyncio.run(myprogram())\n```\n\nor with a context manager:\n\n```python\nwith asynkit.event_loop_policy():\n    asyncio.run(myprogram())\n```\n\n## Scheduling functions\n\nA couple of functions are provided making use of these scheduling features.\nThey require a `SchedulingMixin` event loop to be current.\n\n### `sleep_insert(pos)`\n\nSimilar to `asyncio.sleep()` but sleeps only for `pos` places in the runnable queue.\nWhereas `asyncio.sleep(0)` will place the executing task at the end of the queue, which is\nappropriate for fair scheduling, in some advanced cases you want to wake up sooner than that, perhaps\nafter a specific task.\n\n### `task_reinsert(task, pos)`\n\nTakes a _runnable_ task (for example just created with `asyncio.create_task()` or similar) and\nreinserts it at a given position in the queue. \nSimilarly as for `sleep_insert()`, this can be useful to achieve\ncertain scheduling goals.\n\n### `task_switch(task, *, insert_pos=None)`\n\nImmediately moves the given task to the head of the ready queue and switches to it, assuming it is runnable.\nIf `insert_pos is not None`, the current task will be\nput to sleep at that position, using `sleep_insert()`. Otherwise the current task is put at the end\nof the ready queue.  If `insert_pos == 1` the current task will be inserted directly after the target\ntask, making it the next to be run.  If `insert_pos == 0`, the current task will execute _before_ the target.\n\n### `task_is_blocked(task)`\n\nReturns True if the task is waiting for some awaitable, such as a Future or another Task, and is thus not\non the ready queue.\n\n### `task_is_runnable(task)`\n\nRoughly the opposite of `task_is_blocked()`, returns True if the task is neither `done()` nor __blocked__ and\nawaits execution.\n\n### `create_task_descend(coro)`\n\nImplements depth-first task scheduling.\n\nSimilar to `asyncio.create_task()` this creates a task but starts it running right away, and positions the caller to be woken\nup right after it blocks. The effect is similar to using `asynkit.eager()` but\nit achieves its goals solely by modifying the runnable queue. A `Task` is always\ncreated, unlike `eager`, which only creates a task if the target blocks.\n\n## Runnable task helpers\n\nA few functions are added to help working with tasks.\nThey require a `SchedulingMixin` event loop to be current.\n\nThe following identity applies:\n```python\nasyncio.all_tasks(loop) = asynkit.runnable_tasks(loop) | asynkit.blocked_tasks(loop) | {asyncio.current_task(loop)}\n```\n\n### `runnable_tasks(loop=None)`\n\nReturns a set of the tasks that are currently runnable in the given loop\n\n### `blocked_tasks(loop=None)`\n\nReturns a set of the tasks that are currently blocked on some future in the given loop.\n\n# Coroutine helpers\n\nA couple of functions are provided to introspect the state of coroutine objects. They\nwork on both regular __async__ coroutines, __classic__ coroutines (using `yield from`) and\n__async generators__.\n\n- `coro_is_new(coro)` -\n  Returns true if the object has just been created and hasn\'t started executing yet\n\n- `coro_is_suspended(coro)` - Returns true if the object is in a suspended state.\n\n- `coro_is_done(coro)` - Returns true if the object has finished executing, e.g. by returning or raising an exception.\n\n- `coro_get_frame(coro)` - Returns the current frame object of the coroutine, if it has one, or `None`.\n\n# `anyio` support\n\nThe library has been tested to work with the `anyio`.  However, not everything is supported on the `trio` backend.\nCurrently only the `asyncio` backend can be assumed to work reliably.\n\nWhen using the asyncio backend, the module `asynkit.experimental.anyio` can be used, to provide "eager"-like\nbehaviour to task creation.  It will return an `EagerTaskGroup` context manager:\n\n```python\nfrom asynkit.experimental.anyio import create_eager_task_group\nfrom anyio import run, sleep\n\nasync def func(task_status):\n    print("hello")\n    task_status->started("world")\n    await sleep(0.01)\n    print("goodbye")\n\nasync def main():\n\n    async with create_eager_task_group() as tg:\n        start = tg.start(func)\n        print("fine")\n        print(await start)\n    print ("world")\n\nrun(main, backend="asyncio")\n```\n\nThis will result in the following output:\n\n```\nhello\nfine\nworld\ngoodbye\nworld\n```\n\nThe first part of the function `func` is run even before calling `await` on the result from `EagerTaskGroup.start()`\n\nSimilarly, `EagerTaskGroup.start_soon()` will run the provided coroutine up to its first blocking point before\nreturning.\n\n## `trio` limitations\n\n`trio` differs significantly from `asyncio` and therefore enjoys only limited support.\n\n- The event loop is completely different and proprietary and so the event loop extensions don\'t work\n  for `trio`.\n\n- `CoroStart` when used with `Task` objects, such as by using `EagerTaskGroup`,\n  does not work reliably with `trio`.\n  This is because the syncronization primitives\n  are not based on `Future` objects but rather perform `Task`-based actions both before going to sleep\n  and upon waking up.  If a `CoroStart` initially blocks on a primitive such as `Event.wait()` or\n  `sleep(x)` it will be surprised and throw an error when it wakes up on in a different\n  `Task` than when it was in when it fell asleep.\n\n`CoroStart` works by intercepting a `Future` being passed up via the `await` protocol to \nthe event loop to perform the task scheduling.  If any part of the task scheduling has happened\nbefore this, and the _continuation_ happens on a different `Task` then things may break\nin various ways.   For `asyncio`, the event loop never sees the `Future` object until\n`as_coroutine()` has been called and awaited, and so if this happens in a new task, all is good.\n',
    'author': 'Kristján Valur Jónsson',
    'author_email': 'sweskman@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/kristjanvalur/py-asynkit',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
