"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test for error with existingLambdaObj=undefined (not supplied by user).
// --------------------------------------------------------------
test('Error on existingLambdaObj=undefined', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {};
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test with lambdaFunctionProps.
// --------------------------------------------------------------
test('Test with lambdaFunctionProps', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                OVERRIDE_STATUS: 'true'
            }
        },
        apiGatewayProps: {
            description: "sampleApiProp"
        }
    };
    new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                OVERRIDE_STATUS: "true",
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1"
            }
        }
    });
});
// --------------------------------------------------------------
// Test getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    };
    const app = new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction !== null);
    // Assertion 2
    expect(app.apiGateway !== null);
    expect(app.apiGatewayCloudWatchRole !== null);
    expect(app.apiGatewayLogGroup !== null);
});
// --------------------------------------------------------------
// Test for error with lambdaFunctionProps=undefined (not supplied by user).
// --------------------------------------------------------------
test('Error on lambdaFunctionProps=undefined', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {};
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway AuthorizationType to NONE
// -----------------------------------------------------------------
test('Test deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToLambda(stack, 'api-gateway-lambda', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        },
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "ANY",
        AuthorizationType: "NONE"
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway cloudWatchRole = false
// -----------------------------------------------------------------
test('Test deployment ApiGateway override cloudWatchRole = false', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToLambda(stack, 'api-gateway-lambda', {
        apiGatewayProps: {
            cloudWatchRole: false
        },
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Account", {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "apigatewaylambdaLambdaRestApiCloudWatchRoleA759E8AC",
                "Arn"
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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