"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachine = exports.LogLevel = exports.StateMachineType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const state_graph_1 = require("./state-graph");
const stepfunctions_canned_metrics_generated_1 = require("./stepfunctions-canned-metrics.generated");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @default STANDARD
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 * @stability stable
 */
var StateMachineType;
(function (StateMachineType) {
    StateMachineType["EXPRESS"] = "EXPRESS";
    StateMachineType["STANDARD"] = "STANDARD";
})(StateMachineType = exports.StateMachineType || (exports.StateMachineType = {}));
/**
 * Defines which category of execution history events are logged.
 *
 * @default ERROR
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 * @stability stable
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["OFF"] = "OFF";
    LogLevel["ALL"] = "ALL";
    LogLevel["ERROR"] = "ERROR";
    LogLevel["FATAL"] = "FATAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * A new or imported state machine.
 */
class StateMachineBase extends core_1.Resource {
    /**
     * Import a state machine.
     *
     * @stability stable
     */
    static fromStateMachineArn(scope, id, stateMachineArn) {
        class Import extends StateMachineBase {
            constructor() {
                super(...arguments);
                this.stateMachineArn = stateMachineArn;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity permissions to start an execution of this state machine.
     *
     * @stability stable
     */
    grantStartExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to start a synchronous execution of this state machine.
     *
     * @stability stable
     */
    grantStartSyncExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartSyncExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to read results from state machine.
     *
     * @stability stable
     */
    grantRead(identity) {
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListExecutions',
                'states:ListStateMachines',
            ],
            resourceArns: [this.stateMachineArn],
        });
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:DescribeExecution',
                'states:DescribeStateMachineForExecution',
                'states:GetExecutionHistory',
            ],
            resourceArns: [`${this.executionArn()}:*`],
        });
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListActivities',
                'states:DescribeStateMachine',
                'states:DescribeActivity',
            ],
            resourceArns: ['*'],
        });
    }
    /**
     * Grant the given identity task response permissions on a state machine.
     *
     * @stability stable
     */
    grantTaskResponse(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:SendTaskSuccess',
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
            ],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions on all executions of the state machine.
     *
     * @stability stable
     */
    grantExecution(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [`${this.executionArn()}:*`],
        });
    }
    /**
     * Grant the given identity custom permissions.
     *
     * @stability stable
     */
    grant(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Return the given named metric for this State Machine's executions.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensionsMap: { StateMachineArn: this.stateMachineArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of executions that failed.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricFailed(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsFailedAverage, {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Metric for the number of executions that were throttled.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricThrottled(props) {
        // There's a typo in the "canned" version of this
        return this.metric('ExecutionThrottled', props);
    }
    /**
     * Metric for the number of executions that were aborted.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricAborted(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsAbortedAverage, {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Metric for the number of executions that succeeded.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricSucceeded(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsSucceededAverage, {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Metric for the number of executions that timed out.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricTimedOut(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsTimedOutAverage, {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Metric for the number of executions that were started.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricStarted(props) {
        return this.metric('ExecutionsStarted', props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes.
     *
     * @default - average over 5 minutes
     * @stability stable
     */
    metricTime(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionTimeAverage, props);
    }
    /**
     * Returns the pattern for the execution ARN's of the state machine
     */
    executionArn() {
        return core_1.Stack.of(this).formatArn({
            resource: 'execution',
            service: 'states',
            resourceName: core_1.Arn.split(this.stateMachineArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ StateMachineArn: this.stateMachineArn }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Define a StepFunctions State Machine.
 *
 * @stability stable
 */
class StateMachine extends StateMachineBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.stateMachineName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateMachineProps(props);
        if (props.stateMachineName !== undefined) {
            this.validateStateMachineName(props.stateMachineName);
        }
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        const graph = new state_graph_1.StateGraph(props.definition.startState, `State Machine ${id} definition`);
        graph.timeout = props.timeout;
        this.stateMachineType = (_b = props.stateMachineType) !== null && _b !== void 0 ? _b : StateMachineType.STANDARD;
        const resource = new stepfunctions_generated_1.CfnStateMachine(this, 'Resource', {
            stateMachineName: this.physicalName,
            stateMachineType: (_c = props.stateMachineType) !== null && _c !== void 0 ? _c : undefined,
            roleArn: this.role.roleArn,
            definitionString: core_1.Stack.of(this).toJsonString(graph.toGraphJson()),
            loggingConfiguration: props.logs ? this.buildLoggingConfiguration(props.logs) : undefined,
            tracingConfiguration: props.tracingEnabled ? this.buildTracingConfiguration() : undefined,
        });
        resource.node.addDependency(this.role);
        for (const statement of graph.policyStatements) {
            this.addToRolePolicy(statement);
        }
        this.stateMachineName = this.getResourceNameAttribute(resource.attrName);
        this.stateMachineArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'stateMachine',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * The principal this state machine is running as.
     *
     * @stability stable
     */
    get grantPrincipal() {
        return this.role.grantPrincipal;
    }
    /**
     * Add the given statement to the role's policy.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    validateStateMachineName(stateMachineName) {
        if (!core_1.Token.isUnresolved(stateMachineName)) {
            if (stateMachineName.length < 1 || stateMachineName.length > 80) {
                throw new Error(`State Machine name must be between 1 and 80 characters. Received: ${stateMachineName}`);
            }
            if (!stateMachineName.match(/^[a-z0-9\+\!\@\.\(\)\-\=\_\']+$/i)) {
                throw new Error(`State Machine name must match "^[a-z0-9+!@.()-=_']+$/i". Received: ${stateMachineName}`);
            }
        }
    }
    buildLoggingConfiguration(logOptions) {
        // https://docs.aws.amazon.com/step-functions/latest/dg/cw-logs.html#cloudwatch-iam-policy
        this.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogDelivery',
                'logs:GetLogDelivery',
                'logs:UpdateLogDelivery',
                'logs:DeleteLogDelivery',
                'logs:ListLogDeliveries',
                'logs:PutResourcePolicy',
                'logs:DescribeResourcePolicies',
                'logs:DescribeLogGroups',
            ],
            resources: ['*'],
        }));
        return {
            destinations: [{
                    cloudWatchLogsLogGroup: { logGroupArn: logOptions.destination.logGroupArn },
                }],
            includeExecutionData: logOptions.includeExecutionData,
            level: logOptions.level || 'ERROR',
        };
    }
    buildTracingConfiguration() {
        this.addToRolePolicy(new iam.PolicyStatement({
            // https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#xray-permissions-resources
            // https://docs.aws.amazon.com/step-functions/latest/dg/xray-iam.html
            actions: [
                'xray:PutTraceSegments',
                'xray:PutTelemetryRecords',
                'xray:GetSamplingRules',
                'xray:GetSamplingTargets',
            ],
            resources: ['*'],
        }));
        return {
            enabled: true,
        };
    }
}
exports.StateMachine = StateMachine;
_a = JSII_RTTI_SYMBOL_1;
StateMachine[_a] = { fqn: "@aws-cdk/aws-stepfunctions.StateMachine", version: "1.136.0" };
//# sourceMappingURL=data:application/json;base64,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