"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const certificate_base_1 = require("./certificate-base");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate.
 *
 * @stability stable
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _c;
        this.props = props;
        this.method = (_c = props.method) !== null && _c !== void 0 ? _c : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certificate with DNS.
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created.
     * @stability stable
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certificate with automatically created DNS records in multiple Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created for the domains in the certificate.
     * @stability stable
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certificate with Email.
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *   admin@domain.com
     *   administrator@domain.com
     *   hostmaster@domain.com
     *   postmaster@domain.com
     *   webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate.
     * @stability stable
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "@aws-cdk/aws-certificatemanager.CertificateValidation", version: "1.114.0" };
/**
 * A certificate managed by AWS Certificate Manager.
 *
 * @stability stable
 */
class Certificate extends certificate_base_1.CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate.
     *
     * @stability stable
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends certificate_base_1.CertificateBase {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "@aws-cdk/aws-certificatemanager.Certificate", version: "1.114.0" };
/**
 * Method used to assert ownership of the domain.
 *
 * @stability stable
 */
var ValidationMethod;
(function (ValidationMethod) {
    ValidationMethod["EMAIL"] = "EMAIL";
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _c, _d, _e;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_d = (_c = validation.props.hostedZones) === null || _c === void 0 ? void 0 : _c[domainName]) !== null && _d !== void 0 ? _d : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_e = validation.props.validationDomains) === null || _e === void 0 ? void 0 : _e[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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