"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("aws-cdk-lib/aws-events");
const index_1 = require("../lib/index");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployNewStateMachine(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            schedule: events.Schedule.rate(aws_cdk_lib_1.Duration.minutes(5))
        }
    };
    return new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions', props);
}
function deployNewStateMachineAndEventBus(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' }
    };
    return new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions-eventbus', props);
}
test('check events rule role policy permissions', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testeventbridgestepfunctionsStateMachineDD09BCB6"
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check events rule properties', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "testeventbridgestepfunctionsStateMachineDD09BCB6"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "testeventbridgestepfunctionsEventsRuleRoleFFAAD2A8",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachine !== null);
    expect(construct.eventsRule !== null);
    expect(construct.stateMachineLogGroup !== null);
});
test('check properties with no CW Alarms', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            schedule: events.Schedule.rate(aws_cdk_lib_1.Duration.minutes(5))
        },
        createCloudWatchAlarms: false
    };
    const construct = new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions', props);
    expect(construct.cloudwatchAlarms === null);
    expect(construct.stateMachine !== null);
    expect(construct.eventsRule !== null);
    expect(construct.stateMachineLogGroup !== null);
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachineAndEventBus(stack);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachine !== null);
    expect(construct.eventsRule !== null);
    expect(construct.stateMachineLogGroup !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Events::EventBus', 1);
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' },
        existingEventBusInterface: new events.EventBus(stack, `test-existing-new-eventbus`, { eventBusName: 'test' })
    };
    const app = () => {
        new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventBusProps: {
            eventBusName: 'testcustomeventbus'
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new index_1.EventbridgeToStepfunctions(stack, 'test-new-eventbridge-stepfunctions', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::EventBus', {
        Name: 'testcustomeventbus'
    });
});
test('check LogGroup name', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    // Perform some fancy stuff to examine the specifics of the LogGroupName
    const expectedPrefix = '/aws/vendedlogs/states/constructs/';
    const lengthOfDatetimeSuffix = 13;
    const template = assertions_1.Template.fromStack(stack);
    const LogGroup = template.findResources("AWS::Logs::LogGroup");
    const logName = LogGroup.testeventbridgestepfunctionsStateMachineLogGroup826A5B74.Properties.LogGroupName;
    const suffix = logName.slice(-lengthOfDatetimeSuffix);
    // Look for the expected Prefix and the 13 digit time suffix
    expect(logName.slice(0, expectedPrefix.length)).toEqual(expectedPrefix);
    expect(IsWholeNumber(suffix)).not.toBe(false);
});
function IsWholeNumber(target) {
    const numberPattern = /[0-9]{13}/;
    return target.match(numberPattern) !== null;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnRicmlkZ2Utc3RlcGZ1bmN0aW9ucy50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZXZlbnRicmlkZ2Utc3RlcGZ1bmN0aW9ucy50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxpREFBaUQ7QUFDakQsd0NBQTJGO0FBQzNGLDZDQUF1QztBQUN2QyxxREFBcUQ7QUFDckQsbUNBQW1DO0FBQ25DLHVEQUFrRDtBQUVsRCxTQUFTLHFCQUFxQixDQUFDLEtBQWdCO0lBRTdDLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFFckQsTUFBTSxLQUFLLEdBQW9DO1FBQzdDLGlCQUFpQixFQUFFO1lBQ2pCLFVBQVUsRUFBRSxVQUFVO1NBQ3ZCO1FBQ0QsY0FBYyxFQUFFO1lBQ2QsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLHNCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3BEO0tBQ0YsQ0FBQztJQUVGLE9BQU8sSUFBSSxrQ0FBMEIsQ0FBQyxLQUFLLEVBQUUsZ0NBQWdDLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDeEYsQ0FBQztBQUVELFNBQVMsZ0NBQWdDLENBQUMsS0FBZ0I7SUFFeEQsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUVyRCxNQUFNLEtBQUssR0FBb0M7UUFDN0MsaUJBQWlCLEVBQUU7WUFDakIsVUFBVSxFQUFFLFVBQVU7U0FDdkI7UUFDRCxjQUFjLEVBQUU7WUFDZCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLENBQUMscUJBQXFCLENBQUM7YUFDaEM7U0FDRjtRQUNELGFBQWEsRUFBRSxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUU7S0FDeEMsQ0FBQztJQUVGLE9BQU8sSUFBSSxrQ0FBMEIsQ0FBQyxLQUFLLEVBQUUseUNBQXlDLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDakcsQ0FBQztBQUVELElBQUksQ0FBQywyQ0FBMkMsRUFBRSxHQUFHLEVBQUU7SUFDckQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIscUJBQXFCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFN0IsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO1FBQ2pELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUUsdUJBQXVCO29CQUMvQixNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUU7d0JBQ1IsR0FBRyxFQUFFLGtEQUFrRDtxQkFDeEQ7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsOEJBQThCLEVBQUUsR0FBRyxFQUFFO0lBQ3hDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTdCLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtRQUNsRCxrQkFBa0IsRUFBRSxpQkFBaUI7UUFDckMsS0FBSyxFQUFFLFNBQVM7UUFDaEIsT0FBTyxFQUFFO1lBQ1A7Z0JBQ0UsR0FBRyxFQUFFO29CQUNILEdBQUcsRUFBRSxrREFBa0Q7aUJBQ3hEO2dCQUNELEVBQUUsRUFBRSxTQUFTO2dCQUNiLE9BQU8sRUFBRTtvQkFDUCxZQUFZLEVBQUU7d0JBQ1osb0RBQW9EO3dCQUNwRCxLQUFLO3FCQUNOO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtJQUM1QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLFNBQVMsR0FBK0IscUJBQXFCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0UsTUFBTSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUM1QyxNQUFNLENBQUMsU0FBUyxDQUFDLFlBQVksS0FBSyxJQUFJLENBQUMsQ0FBQztJQUN4QyxNQUFNLENBQUMsU0FBUyxDQUFDLFVBQVUsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUN0QyxNQUFNLENBQUMsU0FBUyxDQUFDLG9CQUFvQixLQUFLLElBQUksQ0FBQyxDQUFDO0FBQ2xELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM5QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFVBQVUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBRXJELE1BQU0sS0FBSyxHQUFvQztRQUM3QyxpQkFBaUIsRUFBRTtZQUNqQixVQUFVLEVBQUUsVUFBVTtTQUN2QjtRQUNELGNBQWMsRUFBRTtZQUNkLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNwRDtRQUNELHNCQUFzQixFQUFFLEtBQUs7S0FDOUIsQ0FBQztJQUVGLE1BQU0sU0FBUyxHQUFnQyxJQUFJLGtDQUEwQixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUU5SCxNQUFNLENBQUMsU0FBUyxDQUFDLGdCQUFnQixLQUFLLElBQUksQ0FBQyxDQUFDO0lBQzVDLE1BQU0sQ0FBQyxTQUFTLENBQUMsWUFBWSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3hDLE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQW9CLEtBQUssSUFBSSxDQUFDLENBQUM7QUFDbEQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO0lBQy9ELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sU0FBUyxHQUErQixnQ0FBZ0MsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLENBQUMsU0FBUyxDQUFDLGdCQUFnQixLQUFLLElBQUksQ0FBQyxDQUFDO0lBQzVDLE1BQU0sQ0FBQyxTQUFTLENBQUMsWUFBWSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3hDLE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQW9CLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDaEQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEtBQUssSUFBSSxDQUFDLENBQUM7SUFFcEMsZ0NBQWdDO0lBQ2hDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDdkQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsZ0VBQWdFLEVBQUUsR0FBRyxFQUFFO0lBQzFFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFFckQsTUFBTSxLQUFLLEdBQW9DO1FBQzdDLGlCQUFpQixFQUFFO1lBQ2pCLFVBQVUsRUFBRSxVQUFVO1NBQ3ZCO1FBQ0QsY0FBYyxFQUFFO1lBQ2QsWUFBWSxFQUFFO2dCQUNaLE1BQU0sRUFBRSxDQUFDLHFCQUFxQixDQUFDO2FBQ2hDO1NBQ0Y7UUFDRCxhQUFhLEVBQUUsRUFBRSxZQUFZLEVBQUUsTUFBTSxFQUFFO1FBQ3ZDLHlCQUF5QixFQUFFLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsNEJBQTRCLEVBQUUsRUFBRyxZQUFZLEVBQUUsTUFBTSxFQUFHLENBQUM7S0FDaEgsQ0FBQztJQUVGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksa0NBQTBCLENBQUMsS0FBSyxFQUFFLGdDQUFnQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2pGLENBQUMsQ0FBQztJQUNGLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2REFBNkQsRUFBRSxHQUFHLEVBQUU7SUFDdkUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUVyRCxNQUFNLEtBQUssR0FBb0M7UUFDN0MsaUJBQWlCLEVBQUU7WUFDakIsVUFBVSxFQUFFLFVBQVU7U0FDdkI7UUFDRCxhQUFhLEVBQUU7WUFDYixZQUFZLEVBQUUsb0JBQW9CO1NBQ25DO1FBQ0QsY0FBYyxFQUFFO1lBQ2QsWUFBWSxFQUFFO2dCQUNaLE1BQU0sRUFBRSxDQUFDLHFCQUFxQixDQUFDO2FBQ2hDO1NBQ0Y7S0FDRixDQUFDO0lBQ0YsSUFBSSxrQ0FBMEIsQ0FBQyxLQUFLLEVBQUUsb0NBQW9DLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFbkYsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELElBQUksRUFBRSxvQkFBb0I7S0FDM0IsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO0lBQy9CLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTdCLHdFQUF3RTtJQUN4RSxNQUFNLGNBQWMsR0FBRyxvQ0FBb0MsQ0FBQztJQUM1RCxNQUFNLHNCQUFzQixHQUFHLEVBQUUsQ0FBQztJQUVsQyxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFFL0QsTUFBTSxPQUFPLEdBQUcsUUFBUSxDQUFDLHdEQUF3RCxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUM7SUFDMUcsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFFdEQsNERBQTREO0lBQzVELE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDeEUsTUFBTSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDaEQsQ0FBQyxDQUFDLENBQUM7QUFFSCxTQUFTLGFBQWEsQ0FBQyxNQUFjO0lBQ25DLE1BQU0sYUFBYSxHQUFHLFdBQVcsQ0FBQztJQUNsQyxPQUFPLE1BQU0sQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLEtBQUssSUFBSSxDQUFDO0FBQzlDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgKiBhcyBldmVudHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cyc7XG5pbXBvcnQgeyBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9ucywgRXZlbnRicmlkZ2VUb1N0ZXBmdW5jdGlvbnNQcm9wcyB9IGZyb20gJy4uL2xpYi9pbmRleCc7XG5pbXBvcnQgeyBEdXJhdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgVGVtcGxhdGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hc3NlcnRpb25zJztcblxuZnVuY3Rpb24gZGVwbG95TmV3U3RhdGVNYWNoaW5lKHN0YWNrOiBjZGsuU3RhY2spIHtcblxuICBjb25zdCBzdGFydFN0YXRlID0gbmV3IHNmbi5QYXNzKHN0YWNrLCAnU3RhcnRTdGF0ZScpO1xuXG4gIGNvbnN0IHByb3BzOiBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9uc1Byb3BzID0ge1xuICAgIHN0YXRlTWFjaGluZVByb3BzOiB7XG4gICAgICBkZWZpbml0aW9uOiBzdGFydFN0YXRlXG4gICAgfSxcbiAgICBldmVudFJ1bGVQcm9wczoge1xuICAgICAgc2NoZWR1bGU6IGV2ZW50cy5TY2hlZHVsZS5yYXRlKER1cmF0aW9uLm1pbnV0ZXMoNSkpXG4gICAgfVxuICB9O1xuXG4gIHJldHVybiBuZXcgRXZlbnRicmlkZ2VUb1N0ZXBmdW5jdGlvbnMoc3RhY2ssICd0ZXN0LWV2ZW50YnJpZGdlLXN0ZXBmdW5jdGlvbnMnLCBwcm9wcyk7XG59XG5cbmZ1bmN0aW9uIGRlcGxveU5ld1N0YXRlTWFjaGluZUFuZEV2ZW50QnVzKHN0YWNrOiBjZGsuU3RhY2spIHtcblxuICBjb25zdCBzdGFydFN0YXRlID0gbmV3IHNmbi5QYXNzKHN0YWNrLCAnU3RhcnRTdGF0ZScpO1xuXG4gIGNvbnN0IHByb3BzOiBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9uc1Byb3BzID0ge1xuICAgIHN0YXRlTWFjaGluZVByb3BzOiB7XG4gICAgICBkZWZpbml0aW9uOiBzdGFydFN0YXRlXG4gICAgfSxcbiAgICBldmVudFJ1bGVQcm9wczoge1xuICAgICAgZXZlbnRQYXR0ZXJuOiB7XG4gICAgICAgIHNvdXJjZTogWydzb2x1dGlvbnNjb25zdHJ1Y3RzJ11cbiAgICAgIH1cbiAgICB9LFxuICAgIGV2ZW50QnVzUHJvcHM6IHsgZXZlbnRCdXNOYW1lOiAndGVzdCcgfVxuICB9O1xuXG4gIHJldHVybiBuZXcgRXZlbnRicmlkZ2VUb1N0ZXBmdW5jdGlvbnMoc3RhY2ssICd0ZXN0LWV2ZW50YnJpZGdlLXN0ZXBmdW5jdGlvbnMtZXZlbnRidXMnLCBwcm9wcyk7XG59XG5cbnRlc3QoJ2NoZWNrIGV2ZW50cyBydWxlIHJvbGUgcG9saWN5IHBlcm1pc3Npb25zJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBkZXBsb3lOZXdTdGF0ZU1hY2hpbmUoc3RhY2spO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpJQU06OlBvbGljeVwiLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBcInN0YXRlczpTdGFydEV4ZWN1dGlvblwiLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBSZWY6IFwidGVzdGV2ZW50YnJpZGdlc3RlcGZ1bmN0aW9uc1N0YXRlTWFjaGluZUREMDlCQ0I2XCJcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgZXZlbnRzIHJ1bGUgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgZGVwbG95TmV3U3RhdGVNYWNoaW5lKHN0YWNrKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFdmVudHM6OlJ1bGUnLCB7XG4gICAgU2NoZWR1bGVFeHByZXNzaW9uOiBcInJhdGUoNSBtaW51dGVzKVwiLFxuICAgIFN0YXRlOiBcIkVOQUJMRURcIixcbiAgICBUYXJnZXRzOiBbXG4gICAgICB7XG4gICAgICAgIEFybjoge1xuICAgICAgICAgIFJlZjogXCJ0ZXN0ZXZlbnRicmlkZ2VzdGVwZnVuY3Rpb25zU3RhdGVNYWNoaW5lREQwOUJDQjZcIlxuICAgICAgICB9LFxuICAgICAgICBJZDogXCJUYXJnZXQwXCIsXG4gICAgICAgIFJvbGVBcm46IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJ0ZXN0ZXZlbnRicmlkZ2VzdGVwZnVuY3Rpb25zRXZlbnRzUnVsZVJvbGVGRkFBRDJBOFwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIF1cbiAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgY29uc3RydWN0OiBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9ucyA9IGRlcGxveU5ld1N0YXRlTWFjaGluZShzdGFjayk7XG5cbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZHdhdGNoQWxhcm1zICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5zdGF0ZU1hY2hpbmUgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LmV2ZW50c1J1bGUgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LnN0YXRlTWFjaGluZUxvZ0dyb3VwICE9PSBudWxsKTtcbn0pO1xuXG50ZXN0KCdjaGVjayBwcm9wZXJ0aWVzIHdpdGggbm8gQ1cgQWxhcm1zJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3Qgc3RhcnRTdGF0ZSA9IG5ldyBzZm4uUGFzcyhzdGFjaywgJ1N0YXJ0U3RhdGUnKTtcblxuICBjb25zdCBwcm9wczogRXZlbnRicmlkZ2VUb1N0ZXBmdW5jdGlvbnNQcm9wcyA9IHtcbiAgICBzdGF0ZU1hY2hpbmVQcm9wczoge1xuICAgICAgZGVmaW5pdGlvbjogc3RhcnRTdGF0ZVxuICAgIH0sXG4gICAgZXZlbnRSdWxlUHJvcHM6IHtcbiAgICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUucmF0ZShEdXJhdGlvbi5taW51dGVzKDUpKVxuICAgIH0sXG4gICAgY3JlYXRlQ2xvdWRXYXRjaEFsYXJtczogZmFsc2VcbiAgfTtcblxuICBjb25zdCBjb25zdHJ1Y3Q6IEV2ZW50YnJpZGdlVG9TdGVwZnVuY3Rpb25zID0gIG5ldyBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9ucyhzdGFjaywgJ3Rlc3QtZXZlbnRicmlkZ2Utc3RlcGZ1bmN0aW9ucycsIHByb3BzKTtcblxuICBleHBlY3QoY29uc3RydWN0LmNsb3Vkd2F0Y2hBbGFybXMgPT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LnN0YXRlTWFjaGluZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuZXZlbnRzUnVsZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3Quc3RhdGVNYWNoaW5lTG9nR3JvdXAgIT09IG51bGwpO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGV2ZW50YnVzIHByb3BlcnR5LCBzbmFwc2hvdCAmIGV2ZW50YnVzIGV4aXN0cycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgY29uc3RydWN0OiBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9ucyA9IGRlcGxveU5ld1N0YXRlTWFjaGluZUFuZEV2ZW50QnVzKHN0YWNrKTtcblxuICBleHBlY3QoY29uc3RydWN0LmNsb3Vkd2F0Y2hBbGFybXMgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LnN0YXRlTWFjaGluZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuZXZlbnRzUnVsZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3Quc3RhdGVNYWNoaW5lTG9nR3JvdXAgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LmV2ZW50QnVzICE9PSBudWxsKTtcblxuICAvLyBDaGVjayB3aGV0aGVyIGV2ZW50YnVzIGV4aXN0c1xuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpFdmVudHM6OkV2ZW50QnVzJywgMSk7XG59KTtcblxudGVzdCgnY2hlY2sgZXhjZXB0aW9uIHdoaWxlIHBhc3NpbmcgZXhpc3RpbmdFdmVudEJ1cyAmIGV2ZW50QnVzUHJvcHMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBzdGFydFN0YXRlID0gbmV3IHNmbi5QYXNzKHN0YWNrLCAnU3RhcnRTdGF0ZScpO1xuXG4gIGNvbnN0IHByb3BzOiBFdmVudGJyaWRnZVRvU3RlcGZ1bmN0aW9uc1Byb3BzID0ge1xuICAgIHN0YXRlTWFjaGluZVByb3BzOiB7XG4gICAgICBkZWZpbml0aW9uOiBzdGFydFN0YXRlXG4gICAgfSxcbiAgICBldmVudFJ1bGVQcm9wczoge1xuICAgICAgZXZlbnRQYXR0ZXJuOiB7XG4gICAgICAgIHNvdXJjZTogWydzb2x1dGlvbnNjb25zdHJ1Y3RzJ11cbiAgICAgIH1cbiAgICB9LFxuICAgIGV2ZW50QnVzUHJvcHM6IHsgZXZlbnRCdXNOYW1lOiAndGVzdCcgfSxcbiAgICBleGlzdGluZ0V2ZW50QnVzSW50ZXJmYWNlOiBuZXcgZXZlbnRzLkV2ZW50QnVzKHN0YWNrLCBgdGVzdC1leGlzdGluZy1uZXctZXZlbnRidXNgLCB7ICBldmVudEJ1c05hbWU6ICd0ZXN0JyAgfSlcbiAgfTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgbmV3IEV2ZW50YnJpZGdlVG9TdGVwZnVuY3Rpb25zKHN0YWNrLCAndGVzdC1ldmVudGJyaWRnZS1zdGVwZnVuY3Rpb25zJywgcHJvcHMpO1xuICB9O1xuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG50ZXN0KCdjaGVjayBjdXN0b20gZXZlbnQgYnVzIHJlc291cmNlIHdpdGggcHJvcHMgd2hlbiBkZXBsb3k6dHJ1ZScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG5cbiAgY29uc3QgcHJvcHM6IEV2ZW50YnJpZGdlVG9TdGVwZnVuY3Rpb25zUHJvcHMgPSB7XG4gICAgc3RhdGVNYWNoaW5lUHJvcHM6IHtcbiAgICAgIGRlZmluaXRpb246IHN0YXJ0U3RhdGVcbiAgICB9LFxuICAgIGV2ZW50QnVzUHJvcHM6IHtcbiAgICAgIGV2ZW50QnVzTmFtZTogJ3Rlc3RjdXN0b21ldmVudGJ1cydcbiAgICB9LFxuICAgIGV2ZW50UnVsZVByb3BzOiB7XG4gICAgICBldmVudFBhdHRlcm46IHtcbiAgICAgICAgc291cmNlOiBbJ3NvbHV0aW9uc2NvbnN0cnVjdHMnXVxuICAgICAgfVxuICAgIH1cbiAgfTtcbiAgbmV3IEV2ZW50YnJpZGdlVG9TdGVwZnVuY3Rpb25zKHN0YWNrLCAndGVzdC1uZXctZXZlbnRicmlkZ2Utc3RlcGZ1bmN0aW9ucycsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFdmVudHM6OkV2ZW50QnVzJywge1xuICAgIE5hbWU6ICd0ZXN0Y3VzdG9tZXZlbnRidXMnXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIExvZ0dyb3VwIG5hbWUnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGRlcGxveU5ld1N0YXRlTWFjaGluZShzdGFjayk7XG5cbiAgLy8gUGVyZm9ybSBzb21lIGZhbmN5IHN0dWZmIHRvIGV4YW1pbmUgdGhlIHNwZWNpZmljcyBvZiB0aGUgTG9nR3JvdXBOYW1lXG4gIGNvbnN0IGV4cGVjdGVkUHJlZml4ID0gJy9hd3MvdmVuZGVkbG9ncy9zdGF0ZXMvY29uc3RydWN0cy8nO1xuICBjb25zdCBsZW5ndGhPZkRhdGV0aW1lU3VmZml4ID0gMTM7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICBjb25zdCBMb2dHcm91cCA9IHRlbXBsYXRlLmZpbmRSZXNvdXJjZXMoXCJBV1M6OkxvZ3M6OkxvZ0dyb3VwXCIpO1xuXG4gIGNvbnN0IGxvZ05hbWUgPSBMb2dHcm91cC50ZXN0ZXZlbnRicmlkZ2VzdGVwZnVuY3Rpb25zU3RhdGVNYWNoaW5lTG9nR3JvdXA4MjZBNUI3NC5Qcm9wZXJ0aWVzLkxvZ0dyb3VwTmFtZTtcbiAgY29uc3Qgc3VmZml4ID0gbG9nTmFtZS5zbGljZSgtbGVuZ3RoT2ZEYXRldGltZVN1ZmZpeCk7XG5cbiAgLy8gTG9vayBmb3IgdGhlIGV4cGVjdGVkIFByZWZpeCBhbmQgdGhlIDEzIGRpZ2l0IHRpbWUgc3VmZml4XG4gIGV4cGVjdChsb2dOYW1lLnNsaWNlKDAsIGV4cGVjdGVkUHJlZml4Lmxlbmd0aCkpLnRvRXF1YWwoZXhwZWN0ZWRQcmVmaXgpO1xuICBleHBlY3QoSXNXaG9sZU51bWJlcihzdWZmaXgpKS5ub3QudG9CZShmYWxzZSk7XG59KTtcblxuZnVuY3Rpb24gSXNXaG9sZU51bWJlcih0YXJnZXQ6IHN0cmluZyk6IGJvb2xlYW4ge1xuICBjb25zdCBudW1iZXJQYXR0ZXJuID0gL1swLTldezEzfS87XG4gIHJldHVybiB0YXJnZXQubWF0Y2gobnVtYmVyUGF0dGVybikgIT09IG51bGw7XG59XG4iXX0=