"""
.. _stopwords:

Stopwords in Several Languages
==============================

List of stopwords by the spaCy [#spaCy]_ package, useful in text mining,
analyzing content of social media posts, tweets, web pages, keywords, etc.

Each list is accessible as part of a dictionary ``stopwords`` which is a normal
Python dictionary.

Available Languages:
^^^^^^^^^^^^^^^^^^^^

* Arabic
* Azerbaijani
* Bengali
* Catalan
* Chinese
* Croatian
* Danish
* Dutch
* English
* Finnish
* French
* German
* Greek
* Hebrew
* Hindi
* Hungarian
* Indonesian
* Irish
* Italian
* Japanese
* Kazakh
* Nepali
* Norwegian
* Persian
* Polish
* Portuguese
* Romanian
* Russian
* Sinhala
* Spanish
* Swedish
* Tagalog
* Tamil
* Tatar
* Telugu
* Thai
* Turkish
* Ukrainian
* Urdu
* Vietnamese

>>> import advertools as adv
>>> sorted(adv.stopwords['english'])[:5]
["a", "about", "above", "across", "after"]

>>> sorted(adv.stopwords['german'])[:5]
["a", "ab", "aber", "ach", "acht"]

To get a list of all available languages, run

>>> adv.stopwords.keys()
dict_keys(['arabic', 'azerbaijani', 'bengali', 'catalan', 'chinese',
'croatian', 'danish', 'dutch', 'english', 'finnish', 'french',
'german', 'greek', 'hebrew', 'hindi', 'hungarian', 'indonesian',
'irish', 'italian', 'japanese', 'kazakh', 'nepali', 'norwegian',
'persian', 'polish', 'portuguese', 'romanian', 'russian', 'sinhala',
'spanish', 'swedish', 'tagalog', 'tamil', 'tatar', 'telugu', 'thai',
'turkish', 'ukrainian', 'urdu', 'vietnamese'])

.. rubric:: Footnotes

.. [#spaCy] Copyright (C) 2016 ExplosionAI UG (haftungsbeschränkt), 2016 spaCy GmbH,
   2015 Matthew Honnibal

"""

stopwords = dict(


arabic={
    "آل", "آه", "آها", "آي", "أبو", "أثناء", "أحد", "أصبح", "أضحى", "أف",
    "أقل", "أكثر", "ألا", "أم", "أما", "أمام", "أمسى", "أن", "أنا", "أنت",
    "أنتم", "أنتما", "أنتن", "أنه", "أنى", "أو", "أولئك", "أولاء", "أوه", "أي",
    "أين", "أينما", "أيها", "إذ", "إذا", "إذما", "إذن", "إلا", "إلى", "إلي",
    "إليك", "إليكم", "إليكما", "إليكن", "إما", "إن", "إنا", "إنما", "إنه",
    "إي", "إيه", "ابن", "اذا", "اصبح", "اضحى", "الا", "التي", "الحالي", "الذى",
    "الذي", "الذين", "اللائي", "اللاتي", "اللتان", "اللتي", "اللتيا", "اللتين",
    "اللذان", "اللذين", "اللواتي", "الى", "الي", "اليه", "اليها", "اليوم",
    "اما", "امسى", "ان", "انه", "انهم", "او", "اول", "اي", "ايضا", "بإن",
    "بات", "باللتي", "بتلك", "بحيث", "بخ", "بد", "بدلا", "بذلك", "بس", "بعد",
    "بعض", "بك", "بكم", "بكما", "بكن", "بل", "بلا", "بلى", "بما", "بماذا",
    "بمن", "بنا", "به", "بها", "بهذا", "بهذه", "بهم", "بهما", "بهن", "بي",
    "بيد", "بين", "بينما", "بيننا", "بينهم", "بينهن", "تحت", "تكون", "تلك",
    "تلكم", "تلكما", "ته", "تي", "تين", "تينك", "ثم", "ثمة", "جدا", "حاشا",
    "حبذا", "حتى", "حول", "حيث", "حيثما", "حين", "خلا", "دون", "ذا", "ذات",
    "ذاك", "ذان", "ذانك", "ذلك", "ذلكم", "ذلكما", "ذلكن", "ذه", "ذو", "ذوا",
    "ذواتا", "ذواتي", "ذي", "ذين", "ذينك", "ريث", "ستكون", "سوف", "سوى",
    "شتان", "صار", "ضد", "ضمن", "ظل", "عدا", "عسى", "عل", "على", "علي", "عليك",
    "عليه", "عليها", "عما", "عن", "عند", "عنه", "عنها", "غير", "فإذا", "فإن",
    "فاللتي", "فانه", "فقد", "فقط", "فكان", "فلا", "فمن", "فهو", "في", "فيم",
    "فيما", "فيه", "فيها", "فيهم", "قال", "قالت", "قبل", "قد", "كأن", "كأنما",
    "كأي", "كأين", "كان", "كانت", "كذا", "كذلك", "كل", "كلا", "كلاهما", "كلتا",
    "كلما", "كليكما", "كليهما", "كم", "كما", "كي", "كيت", "كيف", "كيفما",
    "لأن", "لئن", "لا", "لازال", "لاسيما", "لايزال", "لدى", "لدي", "لذا",
    "لذلك", "لست", "لستم", "لستما", "لستن", "لسن", "لسنا", "لسوف", "لعل",
    "لقد", "لك", "لكم", "لكما", "لكن", "لكنما", "لكنه", "لكي", "لكيلا", "لم",
    "لما", "لماذا", "لن", "لنا", "له", "لها", "لهذا", "لهم", "لهما", "لهن",
    "لو", "لولا", "لوما", "لي", "ليت", "ليس", "ليسا", "ليست", "ليستا", "ليسوا",
    "ما", "ماانفك", "مابرح", "ماذا", "مازال", "مافتئ", "مايزال", "متى", "مثل",
    "مثلا", "مذ", "مساء", "مع", "معه", "مما", "ممن", "من", "منذ", "منه",
    "منها", "مه", "مهما", "نحن", "نحو", "نعم", "هؤلاء", "ها", "هاتان", "هاته",
    "هاتي", "هاتين", "هاك", "هاهنا", "هذا", "هذان", "هذه", "هذي", "هذين",
    "هكذا", "هل", "هلا", "هم", "هما", "هن", "هنا", "هناك", "هنالك", "هو", "هي",
    "هيا", "هيت", "هيهات", "و", "وأبو", "وأثناء", "وأن", "وإذ", "وإذا", "وإن",
    "واذا", "والتي", "والذي", "والذين", "واللتي", "والى", "وان", "وايضا",
    "وبما", "وبين", "وبينما", "وبينهم", "وثي", "وسوف", "وعلى", "وفي", "وقبل",
    "وقد", "وكان", "وكانت", "وكذلك", "وكل", "ولا", "ولايزال", "ولذا", "ولسوف",
    "ولعل", "ولقد", "ولكن", "ولم", "ولما", "ولماذا", "وله", "ولهذا", "ولو",
    "وليس", "وليست", "وما", "وماذا", "ومع", "ومن", "ومنها", "وهذا", "وهل",
    "وهو", "وهي", "يا", "يكون", "يلي", "يمكن", "يوم"
},

azerbaijani={
    "a", "ad", "altı", "altmış", "amma", "arasında", "artıq", "ay", "az",
    "bax", "belə", "bəli", "bəlkə", "beş", "bəy", "bəzən", "bəzi",
    "bilər", "bir", "biraz", "biri", "birşey", "biz", "bizim", "bizlər",
    "bu", "buna", "bundan", "bunların", "bunu", "bunun", "buradan",
    "bütün", "ci", "cı", "çox", "cu", "cü", "çünki", "da", "daha", "də",
    "dedi", "dək", "dən", "dəqiqə", "deyil", "dir", "doqquz", "doqsan",
    "dörd", "düz", "ə", "edən", "edir", "əgər", "əlbəttə", "elə", "əlli",
    "ən", "əslində", "et", "etdi", "etmə", "etmək", "faiz", "gilə",
    "görə", "ha", "haqqında", "harada", "hə", "heç", "həm", "həmin",
    "həmişə", "hər", "ı", "idi", "iki", "il", "ildə", "ilə", "ilk", "in",
    "indi", "isə", "istifadə", "iyirmi", "ki", "kim", "kimə", "kimi",
    "lakin", "lap", "məhz", "mən", "mənə", "mirşey", "nə", "nəhayət",
    "niyə", "o", "obirisi", "of", "olan", "olar", "olaraq", "oldu",
    "olduğu", "olmadı", "olmaz", "olmuşdur", "olsun", "olur", "on", "ona",
    "ondan", "onlar", "onlardan", "onların", "onsuzda", "onu", "onun",
    "oradan", "otuz", "öz", "özü", "qarşı", "qədər", "qırx", "saat",
    "sadəcə", "saniyə", "səhv", "səkkiz", "səksən", "sən", "sənə",
    "sənin", "siz", "sizin", "sizlər", "sonra", "təəssüf", "ü", "üç",
    "üçün", "var", "və", "xan", "xanım", "xeyr", "ya", "yalnız", "yaxşı",
    "yeddi", "yenə", "yəni", "yetmiş", "yox", "yoxdur", "yoxsa", "yüz",
    "zaman"
},

bengali={
    "অতএব", "অথচ", "অথবা", "অনুযায়ী", "অনুযায়ী", "অনেক", "অনেকে", "অনেকেই",
    "অন্তত", "অন্য", "অবধি", "অবশ্য", "অর্থাৎ", "অর্ধভাগে", "আগামী", "আগে",
    "আগেই", "আছে", "আজ", "আদ্যভাগে", "আপনার", "আপনি", "আবার", "আমরা", "আমাকে",
    "আমাদের", "আমার", "আমি", "আর", "আরও", "ইত্যাদি", "ইহা", "উচিত", "উত্তর",
    "উনি", "উপর", "উপরে", "এ", "এঁদের", "এঁরা", "এই", "এক", "একই", "একজন",
    "একটা", "একটি", "একবার", "একে", "এখন", "এখনও", "এখানে", "এখানেই", "এটা",
    "এটাই", "এটি", "এত", "এতটাই", "এতে", "এদের", "এবং", "এবার", "এমন", "এমনকি",
    "এমনি", "এর", "এরা", "এলো", "এস", "এসে", "এসো", "ঐ", "ও", "ওঁদের", "ওঁর",
    "ওঁরা", "ওই", "ওকে", "ওখানে", "ওদের", "ওর", "ওরা", "কখনও", "কত", "কথা",
    "কবে", "কয়েক", "কয়েকটি", "করছে", "করছেন", "করতে", "করবে", "করবেন",
    "করলে", "করলেন", "করা", "করাই", "করায়", "করার", "করায়", "করি", "করিতে",
    "করিয়া", "করিয়ে", "করিয়া", "করিয়ে", "করে", "করেই", "করেছিলেন", "করেছে",
    "করেছেন", "করেন", "কাউকে", "কাছ", "কাছে", "কাজ", "কাজে", "কারও", "কারণ",
    "কি", "কিংবা", "কিছু", "কিছুই", "কিন্তু", "কী", "কে", "কেউ", "কেউই", "কেন",
    "কেমনে", "কোটি", "কোন", "কোনও", "কোনো", "ক্ষেত্রে", "কয়েক", "কয়েকটি",
    "খুব", "গিয়ে", "গিয়েছে", "গিয়ে", "গিয়েছে", "গুলি", "গেছে", "গেল", "গেলে",
    "গোটা", "চলে", "চান", "চায়", "চার", "চালু", "চায়", "চেয়ে", "চেষ্টা",
    "চেয়ে", "ছাড়া", "ছাড়াও", "ছাড়া", "ছাড়াও", "ছিল", "ছিলেন", "জন", "জনকে",
    "জনের", "জন্য", "জন্যে", "জানতে", "জানা", "জানানো", "জানায়", "জানায়",
    "জানিয়ে", "জানিয়েছে", "জানিয়েছে", "জাানিয়ে", "টি", "ঠিক", "তখন", "তত",
    "তথা", "তবু", "তবে", "তা", "তাঁকে", "তাঁদের", "তাঁর", "তাঁরা", "তাঁহারা",
    "তাই", "তাও", "তাকে", "তাতে", "তাদের", "তার", "তারই", "তারপর", "তারা",
    "তাহলে", "তাহা", "তাহাতে", "তাহার", "তিনই", "তিনি", "তিনিও", "তুই", "তুমি",
    "তুলে", "তেমন", "তো", "তোদের", "তোমাদের", "তোমার", "তোর", "তোরা", "থাকবে",
    "থাকবেন", "থাকা", "থাকায়", "থাকায়", "থাকে", "থাকেন", "থেকে", "থেকেই",
    "থেকেও", "দিকে", "দিতে", "দিন", "দিয়ে", "দিয়েছে", "দিয়েছেন", "দিলেন",
    "দিয়ে", "দিয়েছে", "দিয়েছেন", "দু", "দুই", "দুটি", "দুটো", "দেওয়া",
    "দেওয়ার", "দেওয়া", "দেওয়ার", "দেখতে", "দেখা", "দেখে", "দেন", "দেয়",
    "দেশের", "দেয়", "দ্বারা", "ধরা", "ধরে", "নতুন", "নয়", "না", "নাই", "নাকি",
    "নাগাদ", "নানা", "নিজে", "নিজেই", "নিজেদের", "নিজের", "নিতে", "নিয়ে",
    "নিয়ে", "নেই", "নেওয়ার", "নেওয়া", "নয়", "পক্ষে", "পর", "পরে", "পরেই",
    "পরেও", "পর্যন্ত", "পাঁচ", "পাওয়া", "পাওয়া", "পারি", "পারে", "পারেন",
    "পেয়ে", "পেয়ে", "প্রতি", "প্রথম", "প্রভৃতি", "প্রাথমিক", "প্রায়",
    "প্রায়", "ফলে", "ফিরে", "ফের", "বক্তব্য", "বছর", "বদলে", "বন", "বরং",
    "বলতে", "বলল", "বললেন", "বলা", "বলে", "বলেছেন", "বলেন", "বসে", "বহু", "বা",
    "বাদে", "বার", "বিনা", "বিভিন্ন", "বিশেষ", "বিষয়টি", "বেশ", "বেশি",
    "ব্যবহার", "ব্যাপারে", "ভাবে", "ভাবেই", "মত", "মতো", "মতোই", "মধ্যভাগে",
    "মধ্যে", "মধ্যেই", "মধ্যেও", "মনে", "মাত্র", "মাধ্যমে", "মানুষ", "মানুষের",
    "মোট", "মোটেই", "মোদের", "মোর", "যখন", "যত", "যতটা", "যথেষ্ট", "যদি",
    "যদিও", "যা", "যাঁর", "যাঁরা", "যাওয়া", "যাওয়ার", "যাকে", "যাচ্ছে",
    "যাতে", "যাদের", "যান", "যাবে", "যায়", "যার", "যারা", "যায়", "যিনি", "যে",
    "যেখানে", "যেতে", "যেন", "যেমন", "রকম", "রয়েছে", "রাখা", "রেখে", "রয়েছে",
    "লক্ষ", "শুধু", "শুরু", "সঙ্গে", "সঙ্গেও", "সব", "সবার", "সমস্ত",
    "সম্প্রতি", "সময়", "সহ", "সহিত", "সাথে", "সাধারণ", "সামনে", "সুতরাং", "সে",
    "সেই", "সেখান", "সেখানে", "সেটা", "সেটাই", "সেটাও", "সেটি", "স্পষ্ট",
    "স্বয়ং", "হইতে", "হইবে", "হইয়া", "হইয়া", "হওয়া", "হওয়ায়", "হওয়ার",
    "হওয়া", "হওয়ার", "হওয়ায়", "হচ্ছে", "হত", "হতে", "হতেই", "হন", "হবে",
    "হবেন", "হয়", "হয়তো", "হয়নি", "হয়ে", "হয়েই", "হয়েছিল", "হয়েছে",
    "হল", "হলে", "হলেই", "হলেও", "হলো", "হাজার", "হিসাবে", "হিসেবে", "হৈতে",
    "হৈলে", "হোক", "হয়", "হয়তো", "হয়নি", "হয়ে", "হয়েই", "হয়েছিল", "হয়েছে",
    "হয়েছেন"
},

catalan={
    "a", "abans", "ací", "ah", "així", "això", "al", "aleshores", "algun",
    "alguna", "algunes", "alguns", "alhora", "allà", "allí", "allò", "als",
    "altra", "altre", "altres", "amb", "ambdues", "ambdós", "anar", "ans",
    "apa", "aquell", "aquella", "aquelles", "aquells", "aquest", "aquesta",
    "aquestes", "aquests", "aquí", "baix", "bastant", "bé", "cada",
    "cadascuna", "cadascunes", "cadascuns", "cadascú", "com", "consegueixo",
    "conseguim", "conseguir", "consigueix", "consigueixen", "consigueixes",
    "contra", "d'un", "d'una", "d'unes", "d'uns", "dalt", "de", "del", "dels",
    "des", "després", "dins", "dintre", "donat", "doncs", "durant", "e", "eh",
    "el", "elles", "ells", "els", "em", "en", "encara", "ens", "entre", "era",
    "erem", "eren", "eres", "es", "esta", "estan", "estat", "estava",
    "estaven", "estem", "esteu", "estic", "està", "estàvem", "estàveu", "et",
    "etc", "ets", "fa", "faig", "fan", "fas", "fem", "fer", "feu", "fi",
    "fins", "fora", "gairebé", "ha", "han", "has", "haver", "havia", "he",
    "hem", "heu", "hi", "ho", "i", "igual", "iguals", "inclòs", "ja", "jo",
    "l'hi", "la", "les", "li", "li'n", "llarg", "llavors", "m'he", "ma", "mal",
    "malgrat", "mateix", "mateixa", "mateixes", "mateixos", "me", "mentre",
    "meu", "meus", "meva", "meves", "mode", "molt", "molta", "moltes", "molts",
    "mon", "mons", "més", "n'he", "n'hi", "ne", "ni", "no", "nogensmenys",
    "només", "nosaltres", "nostra", "nostre", "nostres", "o", "oh", "oi", "on",
    "pas", "pel", "pels", "per", "perquè", "però", "poc", "poca", "pocs",
    "podem", "poden", "poder", "podeu", "poques", "potser", "primer", "propi",
    "puc", "qual", "quals", "quan", "quant", "que", "quelcom", "qui", "quin",
    "quina", "quines", "quins", "què", "s'ha", "s'han", "sa", "sabem", "saben",
    "saber", "sabeu", "sap", "saps", "semblant", "semblants", "sense", "ser",
    "ses", "seu", "seus", "seva", "seves", "si", "sobre", "sobretot", "soc",
    "solament", "sols", "som", "son", "sons", "sota", "sou", "sóc", "són",
    "t'ha", "t'han", "t'he", "ta", "tal", "també", "tampoc", "tan", "tant",
    "tanta", "tantes", "te", "tene", "tenim", "tenir", "teniu", "teu", "teus",
    "teva", "teves", "tinc", "ton", "tons", "tot", "tota", "totes", "tots",
    "un", "una", "unes", "uns", "us", "va", "vaig", "vam", "van", "vas", "veu",
    "vosaltres", "vostra", "vostre", "vostres", "érem", "éreu", "és", "éssent",
    "últim", "ús"
},

chinese={
    "!", '"', "#", "$", "%", "&", "'", "(", ")", "*", "+", ",", "-", "--", ".",
    "..", "...", "......", "...................", "./", ".一", ".数", ".日", "/",
    "//", "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", ":", "://", "::",
    ";", "<", "=", ">", ">>", "?", "@", "A", "Lex", "[", "]", "^", "_", "`",
    "exp", "sub", "sup", "|", "},", "~", "~~~~", "·", "×", "×××", "Δ", "Ψ",
    "γ", "μ", "φ", "φ．", "В", "—", "——", "———", "‘", "’", "’‘", "“", "”",
    "”，", "…", "……", "…………………………………………………③", "′∈", "′｜", "℃", "Ⅲ", "↑", "→",
    "∈［", "∪φ∈", "≈", "①", "②", "②ｃ", "③", "③］", "④", "⑤", "⑥", "⑦",
    "⑧", "⑨", "⑩", "──", "■", "▲", "、", "。", "〈", "〉", "《", "》", "》），",
    "」", "『", "』", "【", "】",
    "〔", "〕", "〕〔", "㈧", "一", "一.", "一一", "一下", "一个", "一些", "一何", "一切", "一则",
    "一则通过", "一天", "一定", "一方面", "一旦", "一时", "一来", "一样", "一次", "一片", "一番", "一直",
    "一致", "一般", "一起", "一转眼", "一边", "一面", "七", "万一", "三", "三天两头", "三番两次",
    "三番五次", "上", "上下", "上升", "上去", "上来", "上述", "上面", "下", "下列", "下去", "下来",
    "下面", "不", "不一", "不下", "不久", "不了", "不亦乐乎", "不仅", "不仅...而且", "不仅仅", "不仅仅是",
    "不会", "不但", "不但...而且", "不光", "不免", "不再", "不力", "不单", "不变", "不只", "不可",
    "不可开交", "不可抗拒", "不同", "不外", "不外乎", "不够", "不大", "不如", "不妨", "不定", "不对",
    "不少", "不尽", "不尽然", "不巧", "不已", "不常", "不得", "不得不", "不得了", "不得已", "不必",
    "不怎么", "不怕", "不惟", "不成", "不拘", "不择手段", "不敢", "不料", "不断", "不日", "不时", "不是",
    "不曾", "不止", "不止一次", "不比", "不消", "不满", "不然", "不然的话", "不特", "不独", "不由得",
    "不知不觉", "不管", "不管怎样", "不经意", "不胜", "不能", "不能不", "不至于", "不若", "不要", "不论",
    "不起", "不足", "不过", "不迭", "不问", "不限", "与", "与其", "与其说", "与否", "与此同时", "专门",
    "且", "且不说", "且说", "两者", "严格", "严重", "个", "个人", "个别", "中小", "中间", "丰富",
    "串行", "临", "临到", "为", "为主", "为了", "为什么", "为什麽", "为何", "为止", "为此", "为着",
    "主张", "主要", "举凡", "举行", "乃", "乃至", "乃至于", "么", "之", "之一", "之前", "之后", "之後",
    "之所以", "之类", "乌乎", "乎", "乒", "乘", "乘势", "乘机", "乘胜", "乘虚", "乘隙", "九", "也",
    "也好", "也就是说", "也是", "也罢", "了", "了解", "争取", "二", "二来", "二话不说", "二话没说", "于",
    "于是", "于是乎", "云云", "云尔", "互", "互相", "五", "些", "交口", "亦", "产生", "亲口", "亲手",
    "亲眼", "亲自", "亲身", "人", "人人", "人们", "人家", "人民", "什么", "什么样", "什麽", "仅",
    "仅仅", "今", "今后", "今天", "今年", "今後", "介于", "仍", "仍旧", "仍然", "从", "从不", "从严",
    "从中", "从事", "从今以后", "从优", "从古到今", "从古至今", "从头", "从宽", "从小", "从新", "从无到有",
    "从早到晚", "从未", "从来", "从此", "从此以后", "从而", "从轻", "从速", "从重", "他", "他人", "他们",
    "他是", "他的", "代替", "以", "以上", "以下", "以为", "以便", "以免", "以前", "以及", "以后",
    "以外", "以後", "以故", "以期", "以来", "以至", "以至于", "以致", "们", "任", "任何", "任凭",
    "任务", "企图", "伙同", "会", "伟大", "传", "传说", "传闻", "似乎", "似的", "但", "但凡", "但愿",
    "但是", "何", "何乐而不为", "何以", "何况", "何处", "何妨", "何尝", "何必", "何时", "何止", "何苦",
    "何须", "余外", "作为", "你", "你们", "你是", "你的", "使", "使得", "使用", "例如", "依", "依据",
    "依照", "依靠", "便", "便于", "促进", "保持", "保管", "保险", "俺", "俺们", "倍加", "倍感",
    "倒不如", "倒不如说", "倒是", "倘", "倘使", "倘或", "倘然", "倘若", "借", "借以", "借此", "假使",
    "假如", "假若", "偏偏", "做到", "偶尔", "偶而", "傥然", "像", "儿", "允许", "元／吨", "充其极",
    "充其量", "充分", "先不先", "先后", "先後", "先生", "光", "光是", "全体", "全力", "全年", "全然",
    "全身心", "全部", "全都", "全面", "八", "八成", "公然", "六", "兮", "共", "共同", "共总", "关于",
    "其", "其一", "其中", "其二", "其他", "其余", "其后", "其它", "其实", "其次", "具体", "具体地说",
    "具体来说", "具体说来", "具有", "兼之", "内", "再", "再其次", "再则", "再有", "再次", "再者", "再者说",
    "再说", "冒", "冲", "决不", "决定", "决非", "况且", "准备", "凑巧", "凝神", "几", "几乎", "几度",
    "几时", "几番", "几经", "凡", "凡是", "凭", "凭借", "出", "出于", "出去", "出来", "出现", "分别",
    "分头", "分期", "分期分批", "切", "切不可", "切切", "切勿", "切莫", "则", "则甚", "刚", "刚好",
    "刚巧", "刚才", "初", "别", "别人", "别处", "别是", "别的", "别管", "别说", "到", "到了儿", "到处",
    "到头", "到头来", "到底", "到目前为止", "前后", "前此", "前者", "前进", "前面", "加上", "加之", "加以",
    "加入", "加强", "动不动", "动辄", "勃然", "匆匆", "十分", "千", "千万", "千万千万", "半", "单",
    "单单", "单纯", "即", "即令", "即使", "即便", "即刻", "即如", "即将", "即或", "即是说", "即若",
    "却", "却不", "历", "原来", "去", "又", "又及", "及", "及其", "及时", "及至", "双方", "反之",
    "反之亦然", "反之则", "反倒", "反倒是", "反应", "反手", "反映", "反而", "反过来", "反过来说", "取得",
    "取道", "受到", "变成", "古来", "另", "另一个", "另一方面", "另外", "另悉", "另方面", "另行", "只",
    "只当", "只怕", "只是", "只有", "只消", "只要", "只限", "叫", "叫做", "召开", "叮咚", "叮当", "可",
    "可以", "可好", "可是", "可能", "可见", "各", "各个", "各人", "各位", "各地", "各式", "各种",
    "各级", "各自", "合理", "同", "同一", "同时", "同样", "后", "后来", "后者", "后面", "向", "向使",
    "向着", "吓", "吗", "否则", "吧", "吧哒", "吱", "呀", "呃", "呆呆地", "呐", "呕", "呗", "呜",
    "呜呼", "呢", "周围", "呵", "呵呵", "呸", "呼哧", "呼啦", "咋", "和", "咚", "咦", "咧", "咱",
    "咱们", "咳", "哇", "哈", "哈哈", "哉", "哎", "哎呀", "哎哟", "哗", "哗啦", "哟", "哦", "哩",
    "哪", "哪个", "哪些", "哪儿", "哪天", "哪年", "哪怕", "哪样", "哪边", "哪里", "哼", "哼唷", "唉",
    "唯有", "啊", "啊呀", "啊哈", "啊哟", "啐", "啥", "啦", "啪达", "啷当", "喀", "喂", "喏",
    "喔唷", "喽", "嗡", "嗡嗡", "嗬", "嗯", "嗳", "嘎", "嘎嘎", "嘎登", "嘘", "嘛", "嘻", "嘿",
    "嘿嘿", "四", "因", "因为", "因了", "因此", "因着", "因而", "固", "固然", "在", "在下", "在于",
    "地", "均", "坚决", "坚持", "基于", "基本", "基本上", "处在", "处处", "处理", "复杂", "多", "多么",
    "多亏", "多多", "多多少少", "多多益善", "多少", "多年前", "多年来", "多数", "多次", "够瞧的", "大",
    "大不了", "大举", "大事", "大体", "大体上", "大凡", "大力", "大多", "大多数", "大大", "大家",
    "大张旗鼓", "大批", "大抵", "大概", "大略", "大约", "大致", "大都", "大量", "大面儿上", "失去", "奇",
    "奈", "奋勇", "她", "她们", "她是", "她的", "好", "好在", "好的", "好象", "如", "如上", "如上所述",
    "如下", "如今", "如何", "如其", "如前所述", "如同", "如常", "如是", "如期", "如果", "如次", "如此",
    "如此等等", "如若", "始而", "姑且", "存在", "存心", "孰料", "孰知", "宁", "宁可", "宁愿", "宁肯",
    "它", "它们", "它们的", "它是", "它的", "安全", "完全", "完成", "定", "实现", "实际", "宣布",
    "容易", "密切", "对", "对于", "对应", "对待", "对方", "对比", "将", "将才", "将要", "将近", "小",
    "少数", "尔", "尔后", "尔尔", "尔等", "尚且", "尤其", "就", "就地", "就是", "就是了", "就是说",
    "就此", "就算", "就要", "尽", "尽可能", "尽如人意", "尽心尽力", "尽心竭力", "尽快", "尽早", "尽然",
    "尽管", "尽管如此", "尽量", "局外", "居然", "届时", "属于", "屡", "屡屡", "屡次", "屡次三番", "岂",
    "岂但", "岂止", "岂非", "川流不息", "左右", "巨大", "巩固", "差一点", "差不多", "己", "已", "已矣",
    "已经", "巴", "巴巴", "带", "帮助", "常", "常常", "常言说", "常言说得好", "常言道", "平素", "年复一年",
    "并", "并不", "并不是", "并且", "并排", "并无", "并没", "并没有", "并肩", "并非", "广大", "广泛",
    "应当", "应用", "应该", "庶乎", "庶几", "开外", "开始", "开展", "引起", "弗", "弹指之间", "强烈",
    "强调", "归", "归根到底", "归根结底", "归齐", "当", "当下", "当中", "当儿", "当前", "当即", "当口儿",
    "当地", "当场", "当头", "当庭", "当时", "当然", "当真", "当着", "形成", "彻夜", "彻底", "彼",
    "彼时", "彼此", "往", "往往", "待", "待到", "很", "很多", "很少", "後来", "後面", "得", "得了",
    "得出", "得到", "得天独厚", "得起", "心里", "必", "必定", "必将", "必然", "必要", "必须", "快",
    "快要", "忽地", "忽然", "怎", "怎么", "怎么办", "怎么样", "怎奈", "怎样", "怎麽", "怕", "急匆匆",
    "怪", "怪不得", "总之", "总是", "总的来看", "总的来说", "总的说来", "总结", "总而言之", "恍然", "恐怕",
    "恰似", "恰好", "恰如", "恰巧", "恰恰", "恰恰相反", "恰逢", "您", "您们", "您是", "惟其", "惯常",
    "意思", "愤然", "愿意", "慢说", "成为", "成年", "成年累月", "成心", "我", "我们", "我是", "我的",
    "或", "或则", "或多或少", "或是", "或曰", "或者", "或许", "战斗", "截然", "截至", "所", "所以",
    "所在", "所幸", "所有", "所谓", "才", "才能", "扑通", "打", "打从", "打开天窗说亮话", "扩大", "把",
    "抑或", "抽冷子", "拦腰", "拿", "按", "按时", "按期", "按照", "按理", "按说", "挨个", "挨家挨户",
    "挨次", "挨着", "挨门挨户", "挨门逐户", "换句话说", "换言之", "据", "据实", "据悉", "据我所知", "据此",
    "据称", "据说", "掌握", "接下来", "接着", "接著", "接连不断", "放量", "故", "故意", "故此", "故而",
    "敞开儿", "敢", "敢于", "敢情", "数/", "整个", "断然", "方", "方便", "方才", "方能", "方面",
    "旁人", "无", "无宁", "无法", "无论", "既", "既...又", "既往", "既是", "既然", "日复一日", "日渐",
    "日益", "日臻", "日见", "时候", "昂然", "明显", "明确", "是", "是不是", "是以", "是否", "是的",
    "显然", "显著", "普通", "普遍", "暗中", "暗地里", "暗自", "更", "更为", "更加", "更进一步", "曾",
    "曾经", "替", "替代", "最", "最后", "最大", "最好", "最後", "最近", "最高", "有", "有些", "有关",
    "有利", "有力", "有及", "有所", "有效", "有时", "有点", "有的", "有的是", "有着", "有著", "望",
    "朝", "朝着", "末##末", "本", "本人", "本地", "本着", "本身", "权时", "来", "来不及", "来得及",
    "来看", "来着", "来自", "来讲", "来说", "极", "极为", "极了", "极其", "极力", "极大", "极度",
    "极端", "构成", "果然", "果真", "某", "某个", "某些", "某某", "根据", "根本", "格外", "梆", "概",
    "次第", "欢迎", "欤", "正值", "正在", "正如", "正巧", "正常", "正是", "此", "此中", "此后", "此地",
    "此处", "此外", "此时", "此次", "此间", "殆", "毋宁", "每", "每个", "每天", "每年", "每当",
    "每时每刻", "每每", "每逢", "比", "比及", "比如", "比如说", "比方", "比照", "比起", "比较", "毕竟",
    "毫不", "毫无", "毫无例外", "毫无保留地", "汝", "沙沙", "没", "没奈何", "没有", "沿", "沿着", "注意",
    "活", "深入", "清楚", "满", "满足", "漫说", "焉", "然", "然则", "然后", "然後", "然而", "照",
    "照着", "牢牢", "特别是", "特殊", "特点", "犹且", "犹自", "独", "独自", "猛然", "猛然间", "率尔",
    "率然", "现代", "现在", "理应", "理当", "理该", "瑟瑟", "甚且", "甚么", "甚或", "甚而", "甚至",
    "甚至于", "用", "用来", "甫", "甭", "由", "由于", "由是", "由此", "由此可见", "略", "略为", "略加",
    "略微", "白", "白白", "的", "的确", "的话", "皆可", "目前", "直到", "直接", "相似", "相信", "相反",
    "相同", "相对", "相对而言", "相应", "相当", "相等", "省得", "看", "看上去", "看出", "看到", "看来",
    "看样子", "看看", "看见", "看起来", "真是", "真正", "眨眼", "着", "着呢", "矣", "矣乎", "矣哉",
    "知道", "砰", "确定", "碰巧", "社会主义", "离", "种", "积极", "移动", "究竟", "穷年累月", "突出",
    "突然", "窃", "立", "立刻", "立即", "立地", "立时", "立马", "竟", "竟然", "竟而", "第", "第二",
    "等", "等到", "等等", "策略地", "简直", "简而言之", "简言之", "管", "类如", "粗", "精光", "紧接着",
    "累年", "累次", "纯", "纯粹", "纵", "纵令", "纵使", "纵然", "练习", "组成", "经", "经常", "经过",
    "结合", "结果", "给", "绝", "绝不", "绝对", "绝非", "绝顶", "继之", "继后", "继续", "继而", "维持",
    "综上所述", "缕缕", "罢了", "老", "老大", "老是", "老老实实", "考虑", "者", "而", "而且", "而况",
    "而又", "而后", "而外", "而已", "而是", "而言", "而论", "联系", "联袂", "背地里", "背靠背", "能",
    "能否", "能够", "腾", "自", "自个儿", "自从", "自各儿", "自后", "自家", "自己", "自打", "自身",
    "臭", "至", "至于", "至今", "至若", "致", "般的", "良好", "若", "若夫", "若是", "若果", "若非",
    "范围", "莫", "莫不", "莫不然", "莫如", "莫若", "莫非", "获得", "藉以", "虽", "虽则", "虽然",
    "虽说", "蛮", "行为", "行动", "表明", "表示", "被", "要", "要不", "要不是", "要不然", "要么",
    "要是", "要求", "见", "规定", "觉得", "譬喻", "譬如", "认为", "认真", "认识", "让", "许多", "论",
    "论说", "设使", "设或", "设若", "诚如", "诚然", "话说", "该", "该当", "说明", "说来", "说说",
    "请勿", "诸", "诸位", "诸如", "谁", "谁人", "谁料", "谁知", "谨", "豁然", "贼死", "赖以", "赶",
    "赶快", "赶早不赶晚", "起", "起先", "起初", "起头", "起来", "起见", "起首", "趁", "趁便", "趁势",
    "趁早", "趁机", "趁热", "趁着", "越是", "距", "跟", "路经", "转动", "转变", "转贴", "轰然", "较",
    "较为", "较之", "较比", "边", "达到", "达旦", "迄", "迅速", "过", "过于", "过去", "过来", "运用",
    "近", "近几年来", "近年来", "近来", "还", "还是", "还有", "还要", "这", "这一来", "这个", "这么",
    "这么些", "这么样", "这么点儿", "这些", "这会儿", "这儿", "这就是说", "这时", "这样", "这次", "这点",
    "这种", "这般", "这边", "这里", "这麽", "进入", "进去", "进来", "进步", "进而", "进行", "连",
    "连同", "连声", "连日", "连日来", "连袂", "连连", "迟早", "迫于", "适应", "适当", "适用", "逐步",
    "逐渐", "通常", "通过", "造成", "逢", "遇到", "遭到", "遵循", "遵照", "避免", "那", "那个", "那么",
    "那么些", "那么样", "那些", "那会儿", "那儿", "那时", "那末", "那样", "那般", "那边", "那里", "那麽",
    "部分", "都", "鄙人", "采取", "里面", "重大", "重新", "重要", "鉴于", "针对", "长期以来", "长此下去",
    "长线", "长话短说", "问题", "间或", "防止", "阿", "附近", "陈年", "限制", "陡然", "除", "除了",
    "除却", "除去", "除外", "除开", "除此", "除此之外", "除此以外", "除此而外", "除非", "随", "随后",
    "随时", "随着", "随著", "隔夜", "隔日", "难得", "难怪", "难说", "难道", "难道说", "集中", "零",
    "需要", "非但", "非常", "非徒", "非得", "非特", "非独", "靠", "顶多", "顷", "顷刻", "顷刻之间",
    "顷刻间", "顺", "顺着", "顿时", "颇", "风雨无阻", "饱", "首先", "马上", "高低", "高兴", "默然",
    "默默地", "齐", "︿", "！", "＃", "＄", "％", "＆", "＇", "（", "）", "）÷（１－", "）、",
    "＊", "＋", "＋ξ", "＋＋", "，", "，也", "－", "－β", "－－", "－［＊］－", "．", "／", "０",
    "０：２", "１", "１．", "１２％", "２", "２．３％", "３", "４", "５", "５：０", "６", "７", "８",
    "９", "：", "；", "＜", "＜±", "＜Δ", "＜λ", "＜φ", "＜＜", "＝", "＝″", "＝☆", "＝（",
    "＝－", "＝［", "＝｛", "＞", "＞λ", "？", "＠", "Ａ", "ＬＩ", "Ｒ．Ｌ．", "ＺＸＦＩＴＬ", "［",
    "［①①］", "［①②］", "［①③］", "［①④］", "［①⑤］", "［①⑥］", "［①⑦］", "［①⑧］", "［①⑨］",
    "［①Ａ］", "［①Ｂ］", "［①Ｃ］", "［①Ｄ］", "［①Ｅ］", "［①］", "［①ａ］", "［①ｃ］", "［①ｄ］",
    "［①ｅ］", "［①ｆ］", "［①ｇ］", "［①ｈ］", "［①ｉ］", "［①ｏ］", "［②", "［②①］", "［②②］",
    "［②③］", "［②④", "［②⑤］", "［②⑥］", "［②⑦］", "［②⑧］", "［②⑩］", "［②Ｂ］", "［②Ｇ］",
    "［②］", "［②ａ］", "［②ｂ］", "［②ｃ］", "［②ｄ］", "［②ｅ］", "［②ｆ］", "［②ｇ］", "［②ｈ］",
    "［②ｉ］", "［②ｊ］", "［③①］", "［③⑩］", "［③Ｆ］", "［③］", "［③ａ］", "［③ｂ］", "［③ｃ］",
    "［③ｄ］", "［③ｅ］", "［③ｇ］", "［③ｈ］", "［④］", "［④ａ］", "［④ｂ］", "［④ｃ］", "［④ｄ］",
    "［④ｅ］", "［⑤］", "［⑤］］", "［⑤ａ］", "［⑤ｂ］", "［⑤ｄ］", "［⑤ｅ］", "［⑤ｆ］", "［⑥］",
    "［⑦］", "［⑧］", "［⑨］", "［⑩］", "［＊］", "［－", "［］", "］", "］∧′＝［", "］［", "＿",
    "ａ］", "ｂ］", "ｃ］", "ｅ］", "ｆ］", "ｎｇ昉", "｛", "｛－", "｜", "｝", "｝＞", "～", "～±",
    "～＋", "￥"
},

croatian={
    "a", "ah", "aha", "aj", "ako", "al", "ali", "arh", "au", "avaj", "bar",
    "baš", "bez", "bi", "bih", "bijah", "bijahu", "bijasmo", "bijaste",
    "bijaše", "bila", "bili", "bilo", "bio", "bismo", "biste", "biti", "brr",
    "budavši", "bude", "budimo", "budite", "budu", "budući", "bum", "bumo",
    "buć", "da", "daj", "dakle", "de", "deder", "dem", "djelomice",
    "djelomično", "do", "doista", "dok", "dokle", "donekle", "dosad",
    "doskoro", "dotad", "dotle", "dovečer", "drugamo", "drugdje", "duž", "e",
    "eh", "ehe", "ej", "eno", "eto", "evo", "ga", "gdjekakav", "gdjekoje",
    "gic", "god", "halo", "hej", "hm", "hop", "hoće", "hoćemo", "hoćete",
    "hoćeš", "hoću", "htijahu", "htijasmo", "htijaste", "htio", "htjedoh",
    "htjedoste", "htjedoše", "htjela", "htjele", "htjeli", "hura", "i", "iako",
    "ih", "iju", "ijuju", "ikada", "ikakav", "ikakva", "ikakve", "ikakvi",
    "ikakvih", "ikakvim", "ikakvima", "ikakvo", "ikakvog", "ikakvoga",
    "ikakvoj", "ikakvom", "ikakvome", "ili", "im", "iz", "ja", "je", "jedna",
    "jedne", "jedni", "jedno", "jer", "jesam", "jesi", "jesmo", "jest",
    "jeste", "jesu", "jim", "joj", "još", "ju", "kada", "kako", "kao", "koja",
    "koje", "koji", "kojima", "koju", "kroz", "lani", "li", "me", "mene",
    "meni", "mi", "mimo", "moj", "moja", "moje", "moji", "moju", "mu", "na",
    "nad", "nakon", "nam", "nama", "nas", "naš", "naša", "naše", "našeg",
    "naši", "ne", "nego", "neka", "neke", "neki", "nekog", "neku", "nema",
    "netko", "neće", "nećemo", "nećete", "nećeš", "neću", "nešto", "ni",
    "nije", "nikoga", "nikoje", "nikoji", "nikoju", "nisam", "nisi", "nismo",
    "niste", "nisu", "njega", "njegov", "njegova", "njegovo", "njemu",
    "njezin", "njezina", "njezino", "njih", "njihov", "njihova", "njihovo",
    "njim", "njima", "njoj", "nju", "no", "o", "od", "odmah", "on", "ona",
    "one", "oni", "onim", "onima", "ono", "onoj", "onom", "onu", "ova", "ovaj",
    "ovim", "ovima", "ovoj", "pa", "pak", "pljus", "po", "pod", "podalje",
    "poimence", "poizdalje", "ponekad", "pored", "postrance", "potajice",
    "potrbuške", "pouzdano", "prije", "s", "sa", "sam", "samo", "sasvim",
    "sav", "se", "sebe", "sebi", "si", "smo", "ste", "su", "sva", "sve", "svi",
    "svog", "svoj", "svoja", "svoje", "svoju", "svom", "svu", "ta", "tada",
    "taj", "tako", "te", "tebe", "tebi", "ti", "tim", "tima", "to", "toj",
    "tome", "tu", "tvoj", "tvoja", "tvoje", "tvoji", "tvoju", "u", "usprkos",
    "utaman", "uvijek", "uz", "uza", "uzagrapce", "uzalud", "uzduž", "valjda",
    "vam", "vama", "vas", "vaš", "vaša", "vaše", "vašim", "vašima", "već",
    "vi", "vjerojatno", "vjerovatno", "vrh", "vrlo", "za", "zaista", "zar",
    "zatim", "zato", "zbija", "zbog", "zimus", "zum", "će", "ćemo", "ćete",
    "ćeš", "ću", "čijem", "čijim", "čijima", "šic", "šta", "štagod", "što",
    "štogod", "želeći", "željah", "željela", "željele", "željeli", "željelo",
    "željen", "željena", "željene", "željeni", "željenu", "željeo"
},

danish={
    "af", "aldrig", "alene", "alle", "allerede", "alligevel", "alt", "altid",
    "anden", "andet", "andre", "at", "bag", "begge", "blandt", "blev", "blive",
    "bliver", "burde", "bør", "da", "de", "dem", "den", "denne", "dens", "der",
    "derefter", "deres", "derfor", "derfra", "deri", "dermed", "derpå",
    "derved", "det", "dette", "dig", "din", "dine", "disse", "dog", "du",
    "efter", "egen", "eller", "ellers", "en", "end", "endnu", "ene", "eneste",
    "enhver", "ens", "enten", "er", "et", "flere", "flest", "fleste", "for",
    "foran", "fordi", "forrige", "fra", "få", "før", "først", "gennem",
    "gjorde", "gjort", "god", "gør", "gøre", "gørende", "ham", "han", "hans",
    "har", "havde", "have", "hel", "heller", "hen", "hende", "hendes",
    "henover", "her", "herefter", "heri", "hermed", "herpå", "hun", "hvad",
    "hvem", "hver", "hvilke", "hvilken", "hvilkes", "hvis", "hvor", "hvordan",
    "hvorefter", "hvorfor", "hvorfra", "hvorhen", "hvori", "hvorimod",
    "hvornår", "hvorved", "i", "igen", "igennem", "ikke", "imellem", "imens",
    "imod", "ind", "indtil", "ingen", "intet", "jeg", "jer", "jeres", "jo",
    "kan", "kom", "kommer", "kun", "kunne", "lad", "langs", "lav", "lave",
    "lavet", "lidt", "lige", "ligesom", "lille", "længere", "man", "mange",
    "med", "meget", "mellem", "men", "mens", "mere", "mest", "mig", "min",
    "mindre", "mindst", "mine", "mit", "må", "måske", "ned", "nemlig", "nogen",
    "nogensinde", "noget", "nogle", "nok", "nu", "ny", "nyt", "nær", "næste",
    "næsten", "og", "også", "om", "omkring", "op", "os", "over", "overalt",
    "på", "samme", "sammen", "selv", "selvom", "senere", "ses", "siden", "sig",
    "sige", "skal", "skulle", "som", "stadig", "synes", "syntes", "så",
    "sådan", "således", "temmelig", "tidligere", "til", "tilbage", "tit", "ud",
    "uden", "udover", "under", "undtagen", "var", "ved", "vi", "via", "vil",
    "ville", "vore", "vores", "vær", "være", "været", "øvrigt"
},

dutch={
    "aan", "af", "al", "alles", "als", "altijd", "andere", "ben", "bij",
    "daar", "dan", "dat", "de", "der", "deze", "die", "dit", "doch", "doen",
    "door", "dus", "een", "eens", "en", "er", "ge", "geen", "geweest", "haar",
    "had", "heb", "hebben", "heeft", "hem", "het", "hier", "hij", "hoe", "hun",
    "iemand", "iets", "ik", "in", "is", "ja", "je", "kan", "kon", "kunnen",
    "maar", "me", "meer", "men", "met", "mij", "mijn", "moet", "na", "naar",
    "niet", "niets", "nog", "nu", "of", "om", "omdat", "ons", "ook", "op",
    "over", "reeds", "te", "tegen", "toch", "toen", "tot", "u", "uit", "uw",
    "van", "veel", "voor", "want", "waren", "was", "wat", "we", "wel", "werd",
    "wezen", "wie", "wij", "wil", "worden", "zal", "ze", "zei", "zelf", "zich",
    "zij", "zijn", "zo", "zonder", "zou"
},

english={
    "a", "about", "above", "across", "after", "afterwards", "again", "against",
    "all", "almost", "alone", "along", "already", "also", "although", "always",
    "am", "among", "amongst", "amount", "an", "and", "another", "any",
    "anyhow", "anyone", "anything", "anyway", "anywhere", "are", "around",
    "as", "at", "back", "be", "became", "because", "become", "becomes",
    "becoming", "been", "before", "beforehand", "behind", "being", "below",
    "beside", "besides", "between", "beyond", "both", "bottom", "but", "by",
    "ca", "call", "can", "cannot", "could", "did", "do", "does", "doing",
    "done", "down", "due", "during", "each", "eight", "either", "eleven",
    "else", "elsewhere", "empty", "enough", "even", "ever", "every",
    "everyone", "everything", "everywhere", "except", "few", "fifteen",
    "fifty", "first", "five", "for", "former", "formerly", "forty", "four",
    "from", "front", "full", "further", "get", "give", "go", "had", "has",
    "have", "he", "hence", "her", "here", "hereafter", "hereby", "herein",
    "hereupon", "hers", "herself", "him", "himself", "his", "how", "however",
    "hundred", "i", "if", "in", "indeed", "into", "is", "it", "its", "itself",
    "just", "keep", "last", "latter", "latterly", "least", "less", "made",
    "make", "many", "may", "me", "meanwhile", "might", "mine", "more",
    "moreover", "most", "mostly", "move", "much", "must", "my", "myself",
    "name", "namely", "neither", "never", "nevertheless", "next", "nine", "no",
    "nobody", "none", "noone", "nor", "not", "nothing", "now", "nowhere", "of",
    "off", "often", "on", "once", "one", "only", "onto", "or", "other",
    "others", "otherwise", "our", "ours", "ourselves", "out", "over", "own",
    "part", "per", "perhaps", "please", "put", "quite", "rather", "re",
    "really", "regarding", "same", "say", "see", "seem", "seemed", "seeming",
    "seems", "serious", "several", "she", "should", "show", "side", "since",
    "six", "sixty", "so", "some", "somehow", "someone", "something",
    "sometime", "sometimes", "somewhere", "still", "such", "take", "ten",
    "than", "that", "the", "their", "them", "themselves", "then", "thence",
    "there", "thereafter", "thereby", "therefore", "therein", "thereupon",
    "these", "they", "third", "this", "those", "though", "three", "through",
    "throughout", "thru", "thus", "to", "together", "too", "top", "toward",
    "towards", "twelve", "twenty", "two", "under", "unless", "until", "up",
    "upon", "us", "used", "using", "various", "very", "via", "was", "we",
    "well", "were", "what", "whatever", "when", "whence", "whenever", "where",
    "whereafter", "whereas", "whereby", "wherein", "whereupon", "wherever",
    "whether", "which", "while", "whither", "who", "whoever", "whole", "whom",
    "whose", "why", "will", "with", "within", "without", "would", "yet", "you",
    "your", "yours", "yourself", "yourselves"
},

finnish={
    "aiemmin", "aika", "aikaa", "aikaan", "aikaisemmin", "aikaisin", "aikana",
    "aikoina", "aikoo", "aikovat", "aina", "ainakaan", "ainakin", "ainoa",
    "ainoat", "aiomme", "aion", "aiotte", "aivan", "ajan", "alas", "alemmas",
    "alkuisin", "alkuun", "alla", "alle", "aloitamme", "aloitan", "aloitat",
    "aloitatte", "aloitattivat", "aloitettava", "aloitettavaksi", "aloitettu",
    "aloitimme", "aloitin", "aloitit", "aloititte", "aloittaa", "aloittamatta",
    "aloitti", "aloittivat", "alta", "aluksi", "alussa", "alusta", "annettava",
    "annettavaksi", "annettu", "ansiosta", "antaa", "antamatta", "antoi",
    "apu", "asia", "asiaa", "asian", "asiasta", "asiat", "asioiden",
    "asioihin", "asioita", "asti", "avuksi", "avulla", "avun", "avutta",
    "edelle", "edelleen", "edellä", "edeltä", "edemmäs", "edes", "edessä",
    "edestä", "ehkä", "ei", "eikä", "eilen", "eivät", "eli", "ellei",
    "elleivät", "ellemme", "ellen", "ellet", "ellette", "emme", "en",
    "enemmän", "eniten", "ennen", "ensi", "ensimmäinen", "ensimmäiseksi",
    "ensimmäisen", "ensimmäisenä", "ensimmäiset", "ensimmäisiksi",
    "ensimmäisinä", "ensimmäisiä", "ensimmäistä", "ensin", "entinen",
    "entisen", "entisiä", "entisten", "entistä", "enää", "eri", "erittäin",
    "erityisesti", "eräiden", "eräs", "eräät", "esi", "esiin", "esillä",
    "esimerkiksi", "et", "eteen", "etenkin", "ette", "ettei", "että", "halua",
    "haluaa", "haluamatta", "haluamme", "haluan", "haluat", "haluatte",
    "haluavat", "halunnut", "halusi", "halusimme", "halusin", "halusit",
    "halusitte", "halusivat", "halutessa", "haluton", "he", "hei", "heidän",
    "heidät", "heihin", "heille", "heillä", "heiltä", "heissä", "heistä",
    "heitä", "helposti", "heti", "hetkellä", "hieman", "hitaasti",
    "huolimatta", "huomenna", "hyvien", "hyviin", "hyviksi", "hyville",
    "hyviltä", "hyvin", "hyvinä", "hyvissä", "hyvistä", "hyviä", "hyvä",
    "hyvät", "hyvää", "hän", "häneen", "hänelle", "hänellä", "häneltä",
    "hänen", "hänessä", "hänestä", "hänet", "häntä", "ihan", "ilman",
    "ilmeisesti", "itse", "itsensä", "itseään", "ja", "jo", "johon", "joiden",
    "joihin", "joiksi", "joilla", "joille", "joilta", "joina", "joissa",
    "joista", "joita", "joka", "jokainen", "jokin", "joko", "joksi", "joku",
    "jolla", "jolle", "jolloin", "jolta", "jompikumpi", "jona", "jonka",
    "jonkin", "jonne", "joo", "jopa", "jos", "joskus", "jossa", "josta",
    "jota", "jotain", "joten", "jotenkin", "jotenkuten", "jotka", "jotta",
    "jouduimme", "jouduin", "jouduit", "jouduitte", "joudumme", "joudun",
    "joudutte", "joukkoon", "joukossa", "joukosta", "joutua", "joutui",
    "joutuivat", "joutumaan", "joutuu", "joutuvat", "juuri", "jälkeen",
    "jälleen", "jää", "kahdeksan", "kahdeksannen", "kahdella", "kahdelle",
    "kahdelta", "kahden", "kahdessa", "kahdesta", "kahta", "kahteen", "kai",
    "kaiken", "kaikille", "kaikilta", "kaikkea", "kaikki", "kaikkia",
    "kaikkiaan", "kaikkialla", "kaikkialle", "kaikkialta", "kaikkien",
    "kaikkiin", "kaksi", "kannalta", "kannattaa", "kanssa", "kanssaan",
    "kanssamme", "kanssani", "kanssanne", "kanssasi", "kauan", "kauemmas",
    "kaukana", "kautta", "kehen", "keiden", "keihin", "keiksi", "keille",
    "keillä", "keiltä", "keinä", "keissä", "keistä", "keitten", "keittä",
    "keitä", "keneen", "keneksi", "kenelle", "kenellä", "keneltä", "kenen",
    "kenenä", "kenessä", "kenestä", "kenet", "kenettä", "kenties", "kerran",
    "kerta", "kertaa", "keskellä", "kesken", "keskimäärin", "ketkä", "ketä",
    "kiitos", "kohti", "koko", "kokonaan", "kolmas", "kolme", "kolmen",
    "kolmesti", "koska", "koskaan", "kovin", "kuin", "kuinka", "kuinkaan",
    "kuitenkaan", "kuitenkin", "kuka", "kukaan", "kukin", "kumpainen",
    "kumpainenkaan", "kumpi", "kumpikaan", "kumpikin", "kun", "kuten",
    "kuuden", "kuusi", "kuutta", "kylliksi", "kyllä", "kymmenen", "kyse",
    "liian", "liki", "lisäksi", "lisää", "lla", "luo", "luona", "lähekkäin",
    "lähelle", "lähellä", "läheltä", "lähemmäs", "lähes", "lähinnä", "lähtien",
    "läpi", "mahdollisimman", "mahdollista", "me", "meidän", "meidät",
    "meihin", "meille", "meillä", "meiltä", "meissä", "meistä", "meitä",
    "melkein", "melko", "menee", "menemme", "menen", "menet", "menette",
    "menevät", "meni", "menimme", "menin", "menit", "menivät", "mennessä",
    "mennyt", "menossa", "mihin", "miksi", "mikä", "mikäli", "mikään", "mille",
    "milloin", "milloinkan", "millä", "miltä", "minkä", "minne", "minua",
    "minulla", "minulle", "minulta", "minun", "minussa", "minusta", "minut",
    "minuun", "minä", "missä", "mistä", "miten", "mitkä", "mitä", "mitään",
    "moi", "molemmat", "mones", "monesti", "monet", "moni", "moniaalla",
    "moniaalle", "moniaalta", "monta", "muassa", "muiden", "muita", "muka",
    "mukaan", "mukaansa", "mukana", "mutta", "muu", "muualla", "muualle",
    "muualta", "muuanne", "muulloin", "muun", "muut", "muuta", "muutama",
    "muutaman", "muuten", "myöhemmin", "myös", "myöskin", "myöskään", "myötä",
    "ne", "neljä", "neljän", "neljää", "niiden", "niihin", "niiksi", "niille",
    "niillä", "niiltä", "niin", "niinä", "niissä", "niistä", "niitä", "noiden",
    "noihin", "noiksi", "noilla", "noille", "noilta", "noin", "noina",
    "noissa", "noista", "noita", "nopeammin", "nopeasti", "nopeiten", "nro",
    "nuo", "nyt", "näiden", "näihin", "näiksi", "näille", "näillä", "näiltä",
    "näin", "näinä", "näissä", "näistä", "näitä", "nämä", "ohi", "oikea",
    "oikealla", "oikein", "ole", "olemme", "olen", "olet", "olette", "oleva",
    "olevan", "olevat", "oli", "olimme", "olin", "olisi", "olisimme", "olisin",
    "olisit", "olisitte", "olisivat", "olit", "olitte", "olivat", "olla",
    "olleet", "ollut", "oma", "omaa", "omaan", "omaksi", "omalle", "omalta",
    "oman", "omassa", "omat", "omia", "omien", "omiin", "omiksi", "omille",
    "omilta", "omissa", "omista", "on", "onkin", "onko", "ovat", "paikoittain",
    "paitsi", "pakosti", "paljon", "paremmin", "parempi", "parhaillaan",
    "parhaiten", "perusteella", "peräti", "pian", "pieneen", "pieneksi",
    "pienelle", "pienellä", "pieneltä", "pienempi", "pienestä", "pieni",
    "pienin", "poikki", "puolesta", "puolestaan", "päälle", "runsaasti",
    "saakka", "sama", "samaa", "samaan", "samalla", "saman", "samat", "samoin",
    "sata", "sataa", "satojen", "se", "seitsemän", "sekä", "sen", "seuraavat",
    "siellä", "sieltä", "siihen", "siinä", "siis", "siitä", "sijaan", "siksi",
    "sille", "silloin", "sillä", "silti", "siltä", "sinne", "sinua", "sinulla",
    "sinulle", "sinulta", "sinun", "sinussa", "sinusta", "sinut", "sinuun",
    "sinä", "sisäkkäin", "sisällä", "siten", "sitten", "sitä", "ssa", "sta",
    "suoraan", "suuntaan", "suuren", "suuret", "suuri", "suuria", "suurin",
    "suurten", "taa", "taas", "taemmas", "tahansa", "tai", "takaa", "takaisin",
    "takana", "takia", "tallä", "tapauksessa", "tarpeeksi", "tavalla",
    "tavoitteena", "te", "teidän", "teidät", "teihin", "teille", "teillä",
    "teiltä", "teissä", "teistä", "teitä", "tietysti", "todella", "toinen",
    "toisaalla", "toisaalle", "toisaalta", "toiseen", "toiseksi", "toisella",
    "toiselle", "toiselta", "toisemme", "toisen", "toisensa", "toisessa",
    "toisesta", "toista", "toistaiseksi", "toki", "tosin", "tuhannen", "tuhat",
    "tule", "tulee", "tulemme", "tulen", "tulet", "tulette", "tulevat",
    "tulimme", "tulin", "tulisi", "tulisimme", "tulisin", "tulisit",
    "tulisitte", "tulisivat", "tulit", "tulitte", "tulivat", "tulla",
    "tulleet", "tullut", "tuntuu", "tuo", "tuohon", "tuoksi", "tuolla",
    "tuolle", "tuolloin", "tuolta", "tuon", "tuona", "tuonne", "tuossa",
    "tuosta", "tuota", "tuskin", "tykö", "tähän", "täksi", "tälle", "tällä",
    "tällöin", "tältä", "tämä", "tämän", "tänne", "tänä", "tänään", "tässä",
    "tästä", "täten", "tätä", "täysin", "täytyvät", "täytyy", "täällä",
    "täältä", "ulkopuolella", "usea", "useasti", "useimmiten", "usein",
    "useita", "uudeksi", "uudelleen", "uuden", "uudet", "uusi", "uusia",
    "uusien", "uusinta", "uuteen", "uutta", "vaan", "vai", "vaiheessa",
    "vaikea", "vaikean", "vaikeat", "vaikeilla", "vaikeille", "vaikeilta",
    "vaikeissa", "vaikeista", "vaikka", "vain", "varmasti", "varsin",
    "varsinkin", "varten", "vasemmalla", "vasen", "vasta", "vastaan",
    "vastakkain", "vastan", "verran", "vielä", "vierekkäin", "vieressä",
    "vieri", "viiden", "viime", "viimeinen", "viimeisen", "viimeksi", "viisi",
    "voi", "voidaan", "voimme", "voin", "voisi", "voit", "voitte", "voivat",
    "vuoden", "vuoksi", "vuosi", "vuosien", "vuosina", "vuotta", "vähemmän",
    "vähintään", "vähiten", "vähän", "välillä", "yhdeksän", "yhden", "yhdessä",
    "yhteen", "yhteensä", "yhteydessä", "yhteyteen", "yhtä", "yhtäälle",
    "yhtäällä", "yhtäältä", "yhtään", "yhä", "yksi", "yksin", "yksittäin",
    "yleensä", "ylemmäs", "yli", "ylös", "ympäri", "älköön", "älä"
},

french={
    "a", "abord", "absolument", "afin", "ah", "ai", "aie", "ailleurs", "ainsi",
    "ait", "allaient", "allo", "allons", "allô", "alors", "anterieur",
    "anterieure", "anterieures", "apres", "après", "as", "assez", "attendu",
    "au", "aucun", "aucune", "aujourd", "aujourd'hui", "aupres", "auquel",
    "aura", "auraient", "aurait", "auront", "aussi", "autre", "autrefois",
    "autrement", "autres", "autrui", "aux", "auxquelles", "auxquels",
    "avaient", "avais", "avait", "avant", "avec", "avoir", "avons", "ayant",
    "bah", "bas", "basee", "bat", "beau", "beaucoup", "bien", "bigre", "boum",
    "bravo", "brrr", "c'", "car", "ce", "ceci", "cela", "celle", "celle-ci",
    "celle-là", "celles", "celles-ci", "celles-là", "celui", "celui-ci",
    "celui-là", "cent", "cependant", "certain", "certaine", "certaines",
    "certains", "certes", "ces", "cet", "cette", "ceux", "ceux-ci", "ceux-là",
    "chacun", "chacune", "chaque", "cher", "chers", "chez", "chiche", "chut",
    "chère", "chères", "ci", "cinq", "cinquantaine", "cinquante",
    "cinquantième", "cinquième", "clac", "clic", "combien", "comme", "comment",
    "comparable", "comparables", "compris", "concernant", "contre", "couic",
    "crac", "c’", "d'", "da", "dans", "de", "debout", "dedans", "dehors",
    "deja", "delà", "depuis", "dernier", "derniere", "derriere", "derrière",
    "des", "desormais", "desquelles", "desquels", "dessous", "dessus", "deux",
    "deuxième", "deuxièmement", "devant", "devers", "devra", "different",
    "differentes", "differents", "différent", "différente", "différentes",
    "différents", "dire", "directe", "directement", "dit", "dite", "dits",
    "divers", "diverse", "diverses", "dix", "dix-huit", "dix-neuf", "dix-sept",
    "dixième", "doit", "doivent", "donc", "dont", "douze", "douzième", "dring",
    "du", "duquel", "durant", "dès", "désormais", "d’", "effet", "egale",
    "egalement", "egales", "eh", "elle", "elle-même", "elles", "elles-mêmes",
    "en", "encore", "enfin", "entre", "envers", "environ", "es", "est", "et",
    "etaient", "etais", "etait", "etant", "etc", "etre", "eu", "euh", "eux",
    "eux-mêmes", "exactement", "excepté", "extenso", "exterieur", "fais",
    "faisaient", "faisant", "fait", "façon", "feront", "fi", "flac", "floc",
    "font", "gens", "ha", "hein", "hem", "hep", "hi", "ho", "holà", "hop",
    "hormis", "hors", "hou", "houp", "hue", "hui", "huit", "huitième", "hum",
    "hurrah", "hé", "hélas", "i", "il", "ils", "importe", "j'", "je", "jusqu",
    "jusque", "juste", "j’", "l'", "la", "laisser", "laquelle", "las", "le",
    "lequel", "les", "lesquelles", "lesquels", "leur", "leurs", "longtemps",
    "lors", "lorsque", "lui", "lui-meme", "lui-même", "là", "lès", "l’", "m'",
    "ma", "maint", "maintenant", "mais", "malgre", "malgré", "maximale", "me",
    "meme", "memes", "merci", "mes", "mien", "mienne", "miennes", "miens",
    "mille", "mince", "minimale", "moi", "moi-meme", "moi-même", "moindres",
    "moins", "mon", "moyennant", "même", "mêmes", "m’", "n'", "na", "naturel",
    "naturelle", "naturelles", "ne", "neanmoins", "necessaire",
    "necessairement", "neuf", "neuvième", "ni", "nombreuses", "nombreux",
    "non", "nos", "notamment", "notre", "nous", "nous-mêmes", "nouveau", "nul",
    "néanmoins", "nôtre", "nôtres", "n’", "o", "oh", "ohé", "ollé", "olé",
    "on", "ont", "onze", "onzième", "ore", "ou", "ouf", "ouias", "oust",
    "ouste", "outre", "ouvert", "ouverte", "ouverts", "où", "paf", "pan",
    "par", "parce", "parfois", "parle", "parlent", "parler", "parmi",
    "parseme", "partant", "particulier", "particulière", "particulièrement",
    "pas", "passé", "pendant", "pense", "permet", "personne", "peu", "peut",
    "peuvent", "peux", "pff", "pfft", "pfut", "pif", "pire", "plein", "plouf",
    "plus", "plusieurs", "plutôt", "possessif", "possessifs", "possible",
    "possibles", "pouah", "pour", "pourquoi", "pourrais", "pourrait",
    "pouvait", "prealable", "precisement", "premier", "première",
    "premièrement", "pres", "probable", "probante", "procedant", "proche",
    "près", "psitt", "pu", "puis", "puisque", "pur", "pure", "qu'", "quand",
    "quant", "quant-à-soi", "quanta", "quarante", "quatorze", "quatre",
    "quatre-vingt", "quatrième", "quatrièmement", "que", "quel", "quelconque",
    "quelle", "quelles", "quelqu'un", "quelque", "quelques", "quels", "qui",
    "quiconque", "quinze", "quoi", "quoique", "qu’", "rare", "rarement",
    "rares", "relative", "relativement", "remarquable", "rend", "rendre",
    "restant", "reste", "restent", "restrictif", "retour", "revoici",
    "revoilà", "rien", "s'", "sa", "sacrebleu", "sait", "sans", "sapristi",
    "sauf", "se", "sein", "seize", "selon", "semblable", "semblaient",
    "semble", "semblent", "sent", "sept", "septième", "sera", "seraient",
    "serait", "seront", "ses", "seul", "seule", "seulement", "si", "sien",
    "sienne", "siennes", "siens", "sinon", "six", "sixième", "soi", "soi-même",
    "soit", "soixante", "son", "sont", "sous", "souvent", "specifique",
    "specifiques", "speculatif", "stop", "strictement", "subtiles",
    "suffisant", "suffisante", "suffit", "suis", "suit", "suivant", "suivante",
    "suivantes", "suivants", "suivre", "superpose", "sur", "surtout", "s’",
    "t'", "ta", "tac", "tant", "tardive", "te", "tel", "telle", "tellement",
    "telles", "tels", "tenant", "tend", "tenir", "tente", "tes", "tic", "tien",
    "tienne", "tiennes", "tiens", "toc", "toi", "toi-même", "ton", "touchant",
    "toujours", "tous", "tout", "toute", "toutefois", "toutes", "treize",
    "trente", "tres", "trois", "troisième", "troisièmement", "trop", "très",
    "tsoin", "tsouin", "tu", "té", "t’", "un", "une", "unes", "uniformement",
    "unique", "uniques", "uns", "va", "vais", "vas", "vers", "via", "vif",
    "vifs", "vingt", "vivat", "vive", "vives", "vlan", "voici", "voilà",
    "vont", "vos", "votre", "vous", "vous-mêmes", "vu", "vé", "vôtre",
    "vôtres", "zut", "à", "â", "ça", "ès", "étaient", "étais", "était",
    "étant", "été", "être", "ô"
},

german={
    "a", "ab", "aber", "ach", "acht", "achte", "achten", "achter", "achtes",
    "ag", "alle", "allein", "allem", "allen", "aller", "allerdings", "alles",
    "allgemeinen", "als", "also", "am", "an", "andere", "anderen", "andern",
    "anders", "auch", "auf", "aus", "ausser", "ausserdem", "außer", "außerdem",
    "bald", "bei", "beide", "beiden", "beim", "beispiel", "bekannt", "bereits",
    "besonders", "besser", "besten", "bin", "bis", "bisher", "bist", "da",
    "dabei", "dadurch", "dafür", "dagegen", "daher", "dahin", "dahinter",
    "damals", "damit", "danach", "daneben", "dank", "dann", "daran", "darauf",
    "daraus", "darf", "darfst", "darin", "darum", "darunter", "darüber", "das",
    "dasein", "daselbst", "dass", "dasselbe", "davon", "davor", "dazu",
    "dazwischen", "daß", "dein", "deine", "deinem", "deiner", "dem",
    "dementsprechend", "demgegenüber", "demgemäss", "demgemäß", "demselben",
    "demzufolge", "den", "denen", "denn", "denselben", "der", "deren",
    "derjenige", "derjenigen", "dermassen", "dermaßen", "derselbe",
    "derselben", "des", "deshalb", "desselben", "dessen", "deswegen", "dich",
    "die", "diejenige", "diejenigen", "dies", "diese", "dieselbe", "dieselben",
    "diesem", "diesen", "dieser", "dieses", "dir", "doch", "dort", "drei",
    "drin", "dritte", "dritten", "dritter", "drittes", "du", "durch",
    "durchaus", "durfte", "durften", "dürfen", "dürft", "eben", "ebenso",
    "ehrlich", "eigen", "eigene", "eigenen", "eigener", "eigenes", "ein",
    "einander", "eine", "einem", "einen", "einer", "eines", "einigeeinigen",
    "einiger", "einiges", "einmal", "einmaleins", "elf", "en", "ende",
    "endlich", "entweder", "er", "erst", "erste", "ersten", "erster", "erstes",
    "es", "etwa", "etwas", "euch", "früher", "fünf", "fünfte", "fünften",
    "fünfter", "fünftes", "für", "gab", "ganz", "ganze", "ganzen", "ganzer",
    "ganzes", "gar", "gedurft", "gegen", "gegenüber", "gehabt", "gehen",
    "geht", "gekannt", "gekonnt", "gemacht", "gemocht", "gemusst", "genug",
    "gerade", "gern", "gesagt", "geschweige", "gewesen", "gewollt", "geworden",
    "gibt", "ging", "gleich", "gott", "gross", "grosse", "grossen", "grosser",
    "grosses", "groß", "große", "großen", "großer", "großes", "gut", "gute",
    "guter", "gutes", "habe", "haben", "habt", "hast", "hat", "hatte",
    "hatten", "heisst", "heißt", "her", "heute", "hier", "hin", "hinter",
    "hoch", "hätte", "hätten", "ich", "ihm", "ihn", "ihnen", "ihr", "ihre",
    "ihrem", "ihrer", "ihres", "im", "immer", "in", "indem", "infolgedessen",
    "ins", "irgend", "ist", "ja", "jahr", "jahre", "jahren", "je", "jede",
    "jedem", "jeden", "jeder", "jedermann", "jedermanns", "jedoch", "jemand",
    "jemandem", "jemanden", "jene", "jenem", "jenen", "jener", "jenes",
    "jetzt", "kam", "kann", "kannst", "kaum", "kein", "keine", "keinem",
    "keinen", "keiner", "kleine", "kleinen", "kleiner", "kleines", "kommen",
    "kommt", "konnte", "konnten", "kurz", "können", "könnt", "könnte", "lang",
    "lange", "leicht", "leider", "lieber", "los", "machen", "macht", "machte",
    "mag", "magst", "man", "manche", "manchem", "manchen", "mancher",
    "manches", "mehr", "mein", "meine", "meinem", "meinen", "meiner", "meines",
    "mensch", "menschen", "mich", "mir", "mit", "mittel", "mochte", "mochten",
    "morgen", "muss", "musst", "musste", "mussten", "muß", "möchte", "mögen",
    "möglich", "mögt", "müssen", "müsst", "na", "nach", "nachdem", "nahm",
    "natürlich", "neben", "nein", "neue", "neuen", "neun", "neunte", "neunten",
    "neunter", "neuntes", "nicht", "nichts", "nie", "niemand", "niemandem",
    "niemanden", "noch", "nun", "nur", "ob", "oben", "oder", "offen", "oft",
    "ohne", "recht", "rechte", "rechten", "rechter", "rechtes", "richtig",
    "rund", "sagt", "sagte", "sah", "satt", "schlecht", "schon", "sechs",
    "sechste", "sechsten", "sechster", "sechstes", "sehr", "sei", "seid",
    "seien", "sein", "seine", "seinem", "seinen", "seiner", "seines", "seit",
    "seitdem", "selbst", "sich", "sie", "sieben", "siebente", "siebenten",
    "siebenter", "siebentes", "siebte", "siebten", "siebter", "siebtes",
    "sind", "so", "solang", "solche", "solchem", "solchen", "solcher",
    "solches", "soll", "sollen", "sollte", "sollten", "sondern", "sonst",
    "sowie", "später", "statt", "tag", "tage", "tagen", "tat", "teil", "tel",
    "trotzdem", "tun", "uhr", "um", "und", "uns", "unser", "unsere", "unserer",
    "unter", "vergangene", "vergangenen", "viel", "viele", "vielem", "vielen",
    "vielleicht", "vier", "vierte", "vierten", "vierter", "viertes", "vom",
    "von", "vor", "wahr", "wann", "war", "waren", "wart", "warum", "was",
    "wegen", "weil", "weit", "weiter", "weitere", "weiteren", "weiteres",
    "welche", "welchem", "welchen", "welcher", "welches", "wem", "wen",
    "wenig", "wenige", "weniger", "weniges", "wenigstens", "wenn", "wer",
    "werde", "werden", "werdet", "wessen", "wie", "wieder", "will", "willst",
    "wir", "wird", "wirklich", "wirst", "wo", "wohl", "wollen", "wollt",
    "wollte", "wollten", "worden", "wurde", "wurden", "während", "währenddem",
    "währenddessen", "wäre", "würde", "würden", "zehn", "zehnte", "zehnten",
    "zehnter", "zehntes", "zeit", "zu", "zuerst", "zugleich", "zum",
    "zunächst", "zur", "zurück", "zusammen", "zwanzig", "zwar", "zwei",
    "zweite", "zweiten", "zweiter", "zweites", "zwischen", "á", "über",
    "überhaupt", "übrigens"
},

greek={
    "άλλα", "άλλες", "άλλη", "άλλην", "άλλης", "άλλο", "άλλοι", "άλλον",
    "άλλος", "άλλοτε", "άλλους", "άλλων", "άμα", "άμεσα", "άνευ", "άνω",
    "άξαφνα", "άρα", "άραγε", "έγιναν", "έγινε", "έγκαιρα", "έκανε", "έκαστα",
    "έκαστες", "έκαστη", "έκαστην", "έκαστης", "έκαστο", "έκαστοι", "έκαστον",
    "έκαστος", "ένα", "έναν", "ένας", "έξαφνα", "έξι", "έξω", "έπειτα", "έστω",
    "έτερες", "έτερη", "έτερης", "έτερο", "έτεροι", "έτερον", "έτερος",
    "έτερους", "έτσι", "έχει", "έχεις", "έχετε", "έχομε", "έχοντας", "έχουμε",
    "έχουν", "έχω", "έως", "ήδη", "ήμασταν", "ήμαστε", "ήμουν", "ήσασταν",
    "ήσαστε", "ήσουν", "ήταν", "ήτανε", "ήτοι", "ήττον", "ίδια", "ίδιαν",
    "ίδιες", "ίδιο", "ίδιοι", "ίδιον", "ίδιος", "ίδιοσ", "ίδιους", "ίδιων",
    "ίσαμε", "ίσια", "ίσως", "αδιάκοπα", "αι", "ακριβώς", "ακόμα", "ακόμη",
    "αλλά", "αλλαχού", "αλλιώς", "αλλιώτικα", "αλλοιώς", "αλλοιώτικα", "αλλού",
    "αμέσως", "αν", "ανά", "ανάμεσα", "αναμεταξύ", "αντί", "αντίπερα", "αντίς",
    "ανωτέρω", "απ", "απέναντι", "από", "απόψε", "αρκετά", "αρκετές", "αρχικά",
    "ας", "αυτά", "αυτές", "αυτή", "αυτήν", "αυτής", "αυτοί", "αυτού",
    "αυτούς", "αυτό", "αυτόν", "αυτός", "αυτών", "αφού", "αφότου", "αύριο",
    "βέβαια", "βεβαιότατα", "γι", "για", "γιατί", "γρήγορα", "γύρω", "δήθεν",
    "δίπλα", "δίχως", "δα", "δε", "δείνα", "δεν", "δεξιά", "δηλαδή", "δι",
    "δια", "διαρκώς", "δικά", "δικοί", "δικού", "δικούς", "δικό", "δικός",
    "διόλου", "εάν", "είθε", "είμαι", "είμαστε", "είναι", "είσαι", "είσαστε",
    "είστε", "είτε", "είχα", "είχαμε", "είχαν", "είχατε", "είχε", "είχες",
    "εαυτού", "εαυτούς", "εαυτό", "εαυτόν", "εαυτών", "εγκαίρως", "εγώ", "εδώ",
    "ειδεμή", "εις", "εκάστου", "εκάστους", "εκάστων", "εκεί", "εκείνα",
    "εκείνες", "εκείνη", "εκείνην", "εκείνης", "εκείνο", "εκείνοι", "εκείνον",
    "εκείνος", "εκείνου", "εκείνους", "εκείνων", "εκτός", "εμάς", "εμένα",
    "εμείς", "εμπρός", "εν", "εναντίον", "εντελώς", "εντωμεταξύ", "εντός",
    "ενός", "ενώ", "εξ", "εξής", "εξήσ", "εξίσου", "εξαιτίας", "επάνω", "επί",
    "επίσης", "επειδή", "επιπλέον", "επομένως", "επόμενη", "εσάς", "εσένα",
    "εσείς", "εσύ", "ετέρα", "ετέραι", "ετέρας", "ετέρου", "ετέρων", "ετούτα",
    "ετούτες", "ετούτη", "ετούτην", "ετούτης", "ετούτο", "ετούτοι", "ετούτον",
    "ετούτος", "ετούτου", "ετούτους", "ετούτων", "ευθύς", "ευτυχώς", "εφεξής",
    "εχτές", "εύγε", "η", "θα", "ι", "ιδία", "ιδίας", "ιδίου", "ιδίως", "ιι",
    "ιιι", "κάθε", "κάμποσα", "κάμποσες", "κάμποση", "κάμποσην", "κάμποσης",
    "κάμποσο", "κάμποσοι", "κάμποσον", "κάμποσος", "κάμποσου", "κάμποσους",
    "κάμποσων", "κάνεν", "κάποια", "κάποιαν", "κάποιας", "κάποιες", "κάποιο",
    "κάποιοι", "κάποιον", "κάποιος", "κάποιου", "κάποιους", "κάποιων",
    "κάποτε", "κάπου", "κάπως", "κάτι", "κάτω", "καθένα", "καθένας", "καθεμία",
    "καθεμίας", "καθενός", "καθετί", "καθόλου", "καθώς", "και", "κακά",
    "κακώς", "καλά", "καλώς", "καμία", "καμίαν", "καμίας", "κανένα", "κανέναν",
    "κανένας", "κανείς", "κανενός", "κατ", "κατά", "κατιτί", "κατόπιν",
    "κιόλας", "κλπ", "κοντά", "κτλ", "κυρίως", "λίγο", "λιγάκι", "λιγότερο",
    "λοιπά", "λοιπόν", "λόγω", "μάλιστα", "μάλλον", "μέλει", "μέλλεται",
    "μέσα", "μέσω", "μέχρι", "μήδε", "μήπως", "μήτε", "μία", "μα", "μαζί",
    "μακάρι", "μακρυά", "μας", "με", "μείον", "μεθαύριο", "μεμιάς", "μεν",
    "μερικά", "μερικές", "μερικοί", "μερικούς", "μερικών", "μεσώ", "μετ",
    "μετά", "μεταξύ", "μη", "μην", "μια", "μιαν", "μιας", "μολονότι", "μονάχα",
    "μονομιάς", "μου", "μπορεί", "μπορούν", "μπρος", "μόλις", "μόνες", "μόνη",
    "μόνην", "μόνης", "μόνο", "μόνοι", "μόνος", "μόνου", "μόνους", "μόνων",
    "να", "ναι", "νωρίς", "ξανά", "ξαφνικά", "ο", "οι", "ολονέν", "ολωσδιόλου",
    "ολόγυρα", "ολότελα", "οποία", "οποίαν", "οποίας", "οποίες", "οποίο",
    "οποίοι", "οποίος", "οποίου", "οποίους", "οποίων", "οποιαδήποτε",
    "οποιανδήποτε", "οποιασδήποτε", "οποιδήποτε", "οποιεσδήποτε",
    "οποιοδηήποτε", "οποιονδήποτε", "οποιοσδήποτε", "οποιουδήποτε",
    "οποιουσδήποτε", "οποιωνδήποτε", "οποτεδήποτε", "οπουδήποτε", "οπότε",
    "ορισμένα", "ορισμένες", "ορισμένων", "ορισμένως", "ος", "οσαδήποτε",
    "οσεσδήποτε", "οσηδήποτε", "οσηνδήποτε", "οσησδήποτε", "οσοδήποτε",
    "οσοιδήποτε", "οσονδήποτε", "οσοσδήποτε", "οσουδήποτε", "οσουσδήποτε",
    "οσωνδήποτε", "οτιδήποτε", "ου", "ουδέ", "ούτε", "πάλι", "πάντα",
    "πάντοτε", "πάντως", "πάνω", "πάρα", "πέρα", "πέρι", "πέρσι", "πέρυσι",
    "πίσω", "παντού", "παρά", "περί", "περίπου", "περισσότερο", "πια",
    "πιθανόν", "πιο", "πλάι", "πλέον", "πλην", "ποιά", "ποιάν", "ποιάς",
    "ποιές", "ποιοί", "ποιού", "ποιούς", "ποιό", "ποιόν", "ποιός", "ποιών",
    "πολλά", "πολλές", "πολλοί", "πολλούς", "πολύ", "ποτέ", "που", "πουθενά",
    "πού", "πούθε", "πρέπει", "πριν", "προ", "προκειμένου", "προς", "προτού",
    "προχθές", "προχτές", "πρωτύτερα", "πρόκειται", "πρόπερσι", "πρώτα",
    "πρώτες", "πρώτη", "πρώτο", "πρώτος", "πως", "πόσες", "πόση", "πόσην",
    "πόσης", "πόσοι", "πόσος", "πόσους", "πότε", "πώς", "σαν", "σας", "σε",
    "σεις", "σου", "στα", "στη", "στην", "στης", "στις", "στο", "στον", "στου",
    "στους", "στων", "συγχρόνως", "συν", "συνάμα", "συνεπώς", "συχνάς",
    "συχνές", "συχνή", "συχνήν", "συχνής", "συχνοί", "συχνού", "συχνούς",
    "συχνό", "συχνόν", "συχνός", "συχνών", "συχνώς", "σχεδόν", "τάδε", "τάχα",
    "τάχατε", "τέτοια", "τέτοιαν", "τέτοιας", "τέτοιες", "τέτοιο", "τέτοιοι",
    "τέτοιον", "τέτοιος", "τέτοιου", "τέτοιους", "τέτοιων", "τίποτα", "τίποτε",
    "τα", "ταύτα", "ταύτες", "ταύτη", "ταύτην", "ταύτης", "ταύτος",
    "ταύτοταύτον", "ταύτου", "ταύτων", "τελευταία", "τελευταίο", "τελευταίος",
    "τελικά", "τελικώς", "τες", "τη", "την", "της", "τι", "τις", "το", "τοι",
    "τον", "τοσ", "του", "τουλάχιστο", "τουλάχιστον", "τους", "τού", "τούς",
    "τούτα", "τούτες", "τούτη", "τούτην", "τούτης", "τούτο", "τούτοι",
    "τούτοις", "τούτον", "τούτος", "τούτου", "τούτους", "τούτων", "τρία",
    "τρίτη", "τρεις", "τυχόν", "των", "τόσα", "τόσες", "τόση", "τόσην",
    "τόσης", "τόσο", "τόσοι", "τόσον", "τόσος", "τόσου", "τόσους", "τόσων",
    "τότε", "τώρα", "υπ", "υπέρ", "υπό", "υπόψη", "υπόψιν", "χωρίς", "χωριστά",
    "ω", "ως", "ωσάν", "ωστόσο", "ωσότου", "ωχ", "όλα", "όλες", "όλη", "όλην",
    "όλης", "όλο", "όλοι", "όλον", "όλος", "όλου", "όλους", "όλων", "όλως",
    "όμως", "όποια", "όποιες", "όποιο", "όποιοι", "όποιον", "όποιος", "όποτε",
    "όπου", "όπως", "όσα", "όσες", "όση", "όσην", "όσης", "όσο", "όσοι",
    "όσον", "όσος", "όσου", "όσους", "όσων", "όταν", "ότι", "ότου", "όχι",
    "ύστερα", "ώσπου", "ώστε"
},

hebrew={
    "אבל", "או", "אולי", "אותה", "אותו", "אותי", "אותך", "אותם", "אותן",
    "אותנו", "אז", "אחר", "אחרות", "אחרי", "אחרים", "אחרת", "אי", "איזה",
    "איך", "אין", "איפה", "איתה", "איתו", "איתי", "איתך", "איתכם", "איתכן",
    "איתם", "איתן", "איתנו", "אך", "אל", "אלה", "אלו", "אם", "אנחנו", "אני",
    "אס", "אף", "אצל", "אשר", "את", "אתה", "אתכם", "אתכן", "אתם", "אתן",
    "באיזו", "באמצע", "באמצעות", "בגלל", "בו", "בין", "בלי", "במידה", "במקום",
    "ברם", "בשביל", "בשעה", "בתוך", "גם", "דרך", "הוא", "היא", "היה", "היכן",
    "היתה", "היתי", "הם", "הן", "הנה", "הסיבה", "הרי", "ואילו", "ואת", "זאת",
    "זה", "זות", "יהיה", "יוכל", "יוכלו", "יותר", "יכול", "יכולה", "יכולות",
    "יכולים", "יכל", "יכלה", "יכלו", "יש", "כאן", "כאשר", "כולם", "כולן",
    "כזה", "כי", "כיצד", "כך", "ככה", "כל", "כלל", "כמו", "כן", "כפי", "כש",
    "לא", "לאו", "לאיזו", "לאן", "לבין", "לה", "להיות", "להם", "להן", "לו",
    "לי", "לכם", "לכן", "למה", "למטה", "למעלה", "למקום", "למרות", "לנו",
    "לעבר", "לעיכן", "לפיכך", "לפני", "מאד", "מאחורי", "מאיזו", "מאין",
    "מאיפה", "מבלי", "מבעד", "מדוע", "מדי", "מה", "מהיכן", "מול", "מחוץ", "מי",
    "מידה", "מכאן", "מכיוון", "מלבד", "מן", "מנין", "מסוגל", "מעט", "מעטים",
    "מעל", "מצד", "מקום", "מתחת", "מתי", "נגד", "נגר", "נו", "סיבה", "עד",
    "עז", "על", "עלי", "עליה", "עליהם", "עליהן", "עליו", "עליך", "עליכם",
    "עלינו", "עם", "עצמה", "עצמהם", "עצמהן", "עצמו", "עצמי", "עצמם", "עצמן",
    "עצמנו", "פה", "רק", "ש", "שבגללה", "שבו", "שוב", "של", "שלה", "שלהם",
    "שלהן", "שלו", "שלי", "שלך", "שלכם", "שלכן", "שלנו", "שם", "תהיה", "תחת",
    "תכלית"
},

hindi={
    "अंदर", "अत", "अदि", "अप", "अपना", "अपनि", "अपनी", "अपने", "अभि", "अभी",
    "आदि", "आप", "इंहिं", "इंहें", "इंहों", "इतयादि", "इत्यादि", "इन", "इनका",
    "इन्हीं", "इन्हें", "इन्हों", "इस", "इसका", "इसकि", "इसकी", "इसके",
    "इसमें", "इसि", "इसी", "इसे", "उंहिं", "उंहें", "उंहों", "उन", "उनका",
    "उनकि", "उनकी", "उनके", "उनको", "उन्हीं", "उन्हें", "उन्हों", "उस", "उसके",
    "उसि", "उसी", "उसे", "एक", "एवं", "एस", "एसे", "ऐसे", "ओर", "और", "कइ",
    "कई", "कर", "करता", "करते", "करना", "करने", "करें", "कहते", "कहा", "का",
    "काफि", "काफ़ी", "कि", "किंहें", "किंहों", "कितना", "किन्हें", "किन्हों",
    "किया", "किर", "किस", "किसि", "किसी", "किसे", "की", "कुछ", "कुल", "के",
    "को", "कोइ", "कोई", "कोन", "कोनसा", "कौन", "कौनसा", "गया", "घर", "जब",
    "जहाँ", "जहां", "जा", "जिंहें", "जिंहों", "जितना", "जिधर", "जिन",
    "जिन्हें", "जिन्हों", "जिस", "जिसे", "जीधर", "जेसा", "जेसे", "जैसा",
    "जैसे", "जो", "तक", "तब", "तरह", "तिंहें", "तिंहों", "तिन", "तिन्हें",
    "तिन्हों", "तिस", "तिसे", "तो", "था", "थि", "थी", "थे", "दबारा", "दवारा",
    "दिया", "दुसरा", "दुसरे", "दूसरे", "दो", "द्वारा", "न", "नहिं", "नहीं",
    "ना", "निचे", "निहायत", "नीचे", "ने", "पर", "पहले", "पुरा", "पूरा", "पे",
    "फिर", "बनि", "बनी", "बहि", "बही", "बहुत", "बाद", "बाला", "बिलकुल", "भि",
    "भितर", "भी", "भीतर", "मगर", "मानो", "मे", "में", "यदि", "यह", "यहाँ",
    "यहां", "यहि", "यही", "या", "यिह", "ये", "रखें", "रवासा", "रहा", "रहे",
    "ऱ्वासा", "लिए", "लिये", "लेकिन", "व", "वगेरह", "वरग", "वर्ग", "वह",
    "वहाँ", "वहां", "वहिं", "वहीं", "वाले", "वुह", "वे", "वग़ैरह", "संग",
    "सकता", "सकते", "सबसे", "सभि", "सभी", "साथ", "साबुत", "साभ", "सारा", "से",
    "सो", "हि", "ही", "हुअ", "हुआ", "हुइ", "हुई", "हुए", "हे", "हें", "है",
    "हैं", "हो", "होता", "होति", "होती", "होते", "होना", "होने"
},

hungarian={
    "a", "abban", "ahhoz", "ahogy", "ahol", "aki", "akik", "akkor", "akár",
    "alatt", "amely", "amelyek", "amelyekben", "amelyeket", "amelyet",
    "amelynek", "ami", "amikor", "amit", "amolyan", "amíg", "annak", "arra",
    "arról", "az", "azok", "azon", "azonban", "azt", "aztán", "azután",
    "azzal", "azért", "be", "belül", "benne", "bár", "cikk", "cikkek",
    "cikkeket", "csak", "de", "e", "ebben", "eddig", "egy", "egyes",
    "egyetlen", "egyik", "egyre", "egyéb", "egész", "ehhez", "ekkor", "el",
    "ellen", "elo", "eloször", "elott", "elso", "elég", "előtt", "emilyen",
    "ennek", "erre", "ez", "ezek", "ezen", "ezt", "ezzel", "ezért", "fel",
    "felé", "ha", "hanem", "hiszen", "hogy", "hogyan", "hát", "ide", "igen",
    "ill", "ill.", "illetve", "ilyen", "ilyenkor", "inkább", "is", "ismét",
    "ison", "itt", "jobban", "jó", "jól", "kell", "kellett", "keressünk",
    "keresztül", "ki", "kívül", "között", "közül", "le", "legalább", "legyen",
    "lehet", "lehetett", "lenne", "lenni", "lesz", "lett", "ma", "maga",
    "magát", "majd", "meg", "mellett", "mely", "melyek", "mert", "mi", "miatt",
    "mikor", "milyen", "minden", "mindenki", "mindent", "mindig", "mint",
    "mintha", "mit", "mivel", "miért", "mondta", "most", "már", "más", "másik",
    "még", "míg", "nagy", "nagyobb", "nagyon", "ne", "nekem", "neki", "nem",
    "nincs", "néha", "néhány", "nélkül", "o", "oda", "ok", "oket", "olyan",
    "ott", "pedig", "persze", "például", "rá", "s", "saját", "sem", "semmi",
    "sok", "sokat", "sokkal", "stb.", "szemben", "szerint", "szinte",
    "számára", "szét", "talán", "te", "tehát", "teljes", "ti", "tovább",
    "továbbá", "több", "túl", "ugyanis", "utolsó", "után", "utána", "vagy",
    "vagyis", "vagyok", "valaki", "valami", "valamint", "való", "van",
    "vannak", "vele", "vissza", "viszont", "volna", "volt", "voltak", "voltam",
    "voltunk", "által", "általában", "át", "én", "éppen", "és", "így", "ön",
    "össze", "úgy", "új", "újabb", "újra", "ő", "őket"
},

indonesian={
    "ada", "adalah", "adanya", "adapun", "agak", "agaknya", "agar", "akan",
    "akankah", "akhir", "akhiri", "akhirnya", "aku", "akulah", "amat",
    "amatlah", "anda", "andalah", "antar", "antara", "antaranya", "apa",
    "apaan", "apabila", "apakah", "apalagi", "apatah", "artinya", "asal",
    "asalkan", "atas", "atau", "ataukah", "ataupun", "awal", "awalnya",
    "bagai", "bagaikan", "bagaimana", "bagaimanakah", "bagaimanapun", "bagi",
    "bagian", "bahkan", "bahwa", "bahwasanya", "baik", "bakal", "bakalan",
    "balik", "banyak", "bapak", "baru", "bawah", "beberapa", "begini",
    "beginian", "beginikah", "beginilah", "begitu", "begitukah", "begitulah",
    "begitupun", "bekerja", "belakang", "belakangan", "belum", "belumlah",
    "benar", "benarkah", "benarlah", "berada", "berakhir", "berakhirlah",
    "berakhirnya", "berapa", "berapakah", "berapalah", "berapapun", "berarti",
    "berawal", "berbagai", "berdatangan", "beri", "berikan", "berikut",
    "berikutnya", "berjumlah", "berkali-kali", "berkata", "berkehendak",
    "berkeinginan", "berkenaan", "berlainan", "berlalu", "berlangsung",
    "berlebihan", "bermacam", "bermacam-macam", "bermaksud", "bermula",
    "bersama", "bersama-sama", "bersiap", "bersiap-siap", "bertanya",
    "bertanya-tanya", "berturut", "berturut-turut", "bertutur", "berujar",
    "berupa", "besar", "betul", "betulkah", "biasa", "biasanya", "bila",
    "bilakah", "bisa", "bisakah", "boleh", "bolehkah", "bolehlah", "buat",
    "bukan", "bukankah", "bukanlah", "bukannya", "bulan", "bung", "cara",
    "caranya", "cukup", "cukupkah", "cukuplah", "cuma", "dahulu", "dalam",
    "dan", "dapat", "dari", "daripada", "datang", "dekat", "demi", "demikian",
    "demikianlah", "dengan", "depan", "di", "dia", "diakhiri", "diakhirinya",
    "dialah", "diantara", "diantaranya", "diberi", "diberikan", "diberikannya",
    "dibuat", "dibuatnya", "didapat", "didatangkan", "digunakan",
    "diibaratkan", "diibaratkannya", "diingat", "diingatkan", "diinginkan",
    "dijawab", "dijelaskan", "dijelaskannya", "dikarenakan", "dikatakan",
    "dikatakannya", "dikerjakan", "diketahui", "diketahuinya", "dikira",
    "dilakukan", "dilalui", "dilihat", "dimaksud", "dimaksudkan",
    "dimaksudkannya", "dimaksudnya", "diminta", "dimintai", "dimisalkan",
    "dimulai", "dimulailah", "dimulainya", "dimungkinkan", "dini",
    "dipastikan", "diperbuat", "diperbuatnya", "dipergunakan", "diperkirakan",
    "diperlihatkan", "diperlukan", "diperlukannya", "dipersoalkan",
    "dipertanyakan", "dipunyai", "diri", "dirinya", "disampaikan", "disebut",
    "disebutkan", "disebutkannya", "disini", "disinilah", "ditambahkan",
    "ditandaskan", "ditanya", "ditanyai", "ditanyakan", "ditegaskan",
    "ditujukan", "ditunjuk", "ditunjuki", "ditunjukkan", "ditunjukkannya",
    "ditunjuknya", "dituturkan", "dituturkannya", "diucapkan", "diucapkannya",
    "diungkapkan", "dong", "dua", "dulu", "empat", "enggak", "enggaknya",
    "entah", "entahlah", "guna", "gunakan", "hal", "hampir", "hanya",
    "hanyalah", "hari", "harus", "haruslah", "harusnya", "hendak", "hendaklah",
    "hendaknya", "hingga", "ia", "ialah", "ibarat", "ibaratkan", "ibaratnya",
    "ibu", "ikut", "ingat", "ingat-ingat", "ingin", "inginkah", "inginkan",
    "ini", "inikah", "inilah", "itu", "itukah", "itulah", "jadi", "jadilah",
    "jadinya", "jangan", "jangankan", "janganlah", "jauh", "jawab", "jawaban",
    "jawabnya", "jelas", "jelaskan", "jelaslah", "jelasnya", "jika", "jikalau",
    "juga", "jumlah", "jumlahnya", "justru", "kala", "kalau", "kalaulah",
    "kalaupun", "kalian", "kami", "kamilah", "kamu", "kamulah", "kan", "kapan",
    "kapankah", "kapanpun", "karena", "karenanya", "kasus", "kata", "katakan",
    "katakanlah", "katanya", "ke", "keadaan", "kebetulan", "kecil", "kedua",
    "keduanya", "keinginan", "kelamaan", "kelihatan", "kelihatannya", "kelima",
    "keluar", "kembali", "kemudian", "kemungkinan", "kemungkinannya", "kenapa",
    "kepada", "kepadanya", "kesampaian", "keseluruhan", "keseluruhannya",
    "keterlaluan", "ketika", "khususnya", "kini", "kinilah", "kira",
    "kira-kira", "kiranya", "kita", "kitalah", "kok", "kurang", "lagi",
    "lagian", "lah", "lain", "lainnya", "lalu", "lama", "lamanya", "lanjut",
    "lanjutnya", "lebih", "lewat", "lima", "luar", "macam", "maka", "makanya",
    "makin", "malah", "malahan", "mampu", "mampukah", "mana", "manakala",
    "manalagi", "masa", "masalah", "masalahnya", "masih", "masihkah", "masing",
    "masing-masing", "mau", "maupun", "melainkan", "melakukan", "melalui",
    "melihat", "melihatnya", "memang", "memastikan", "memberi", "memberikan",
    "membuat", "memerlukan", "memihak", "meminta", "memintakan", "memisalkan",
    "memperbuat", "mempergunakan", "memperkirakan", "memperlihatkan",
    "mempersiapkan", "mempersoalkan", "mempertanyakan", "mempunyai", "memulai",
    "memungkinkan", "menaiki", "menambahkan", "menandaskan", "menanti",
    "menanti-nanti", "menantikan", "menanya", "menanyai", "menanyakan",
    "mendapat", "mendapatkan", "mendatang", "mendatangi", "mendatangkan",
    "menegaskan", "mengakhiri", "mengapa", "mengatakan", "mengatakannya",
    "mengenai", "mengerjakan", "mengetahui", "menggunakan", "menghendaki",
    "mengibaratkan", "mengibaratkannya", "mengingat", "mengingatkan",
    "menginginkan", "mengira", "mengucapkan", "mengucapkannya",
    "mengungkapkan", "menjadi", "menjawab", "menjelaskan", "menuju",
    "menunjuk", "menunjuki", "menunjukkan", "menunjuknya", "menurut",
    "menuturkan", "menyampaikan", "menyangkut", "menyatakan", "menyebutkan",
    "menyeluruh", "menyiapkan", "merasa", "mereka", "merekalah", "merupakan",
    "meski", "meskipun", "meyakini", "meyakinkan", "minta", "mirip", "misal",
    "misalkan", "misalnya", "mula", "mulai", "mulailah", "mulanya", "mungkin",
    "mungkinkah", "nah", "naik", "namun", "nanti", "nantinya", "nyaris",
    "nyatanya", "oleh", "olehnya", "pada", "padahal", "padanya", "pak",
    "paling", "panjang", "pantas", "para", "pasti", "pastilah", "penting",
    "pentingnya", "per", "percuma", "perlu", "perlukah", "perlunya", "pernah",
    "persoalan", "pertama", "pertama-tama", "pertanyaan", "pertanyakan",
    "pihak", "pihaknya", "pukul", "pula", "pun", "punya", "rasa", "rasanya",
    "rata", "rupanya", "saat", "saatnya", "saja", "sajalah", "saling", "sama",
    "sama-sama", "sambil", "sampai", "sampai-sampai", "sampaikan", "sana",
    "sangat", "sangatlah", "satu", "saya", "sayalah", "se", "sebab",
    "sebabnya", "sebagai", "sebagaimana", "sebagainya", "sebagian", "sebaik",
    "sebaik-baiknya", "sebaiknya", "sebaliknya", "sebanyak", "sebegini",
    "sebegitu", "sebelum", "sebelumnya", "sebenarnya", "seberapa", "sebesar",
    "sebetulnya", "sebisanya", "sebuah", "sebut", "sebutlah", "sebutnya",
    "secara", "secukupnya", "sedang", "sedangkan", "sedemikian", "sedikit",
    "sedikitnya", "seenaknya", "segala", "segalanya", "segera", "seharusnya",
    "sehingga", "seingat", "sejak", "sejauh", "sejenak", "sejumlah", "sekadar",
    "sekadarnya", "sekali", "sekali-kali", "sekalian", "sekaligus",
    "sekalipun", "sekarang", "sekecil", "seketika", "sekiranya", "sekitar",
    "sekitarnya", "sekurang-kurangnya", "sekurangnya", "sela", "selain",
    "selaku", "selalu", "selama", "selama-lamanya", "selamanya", "selanjutnya",
    "seluruh", "seluruhnya", "semacam", "semakin", "semampu", "semampunya",
    "semasa", "semasih", "semata", "semata-mata", "semaunya", "sementara",
    "semisal", "semisalnya", "sempat", "semua", "semuanya", "semula",
    "sendiri", "sendirian", "sendirinya", "seolah", "seolah-olah", "seorang",
    "sepanjang", "sepantasnya", "sepantasnyalah", "seperlunya", "seperti",
    "sepertinya", "sepihak", "sering", "seringnya", "serta", "serupa",
    "sesaat", "sesama", "sesampai", "sesegera", "sesekali", "seseorang",
    "sesuatu", "sesuatunya", "sesudah", "sesudahnya", "setelah", "setempat",
    "setengah", "seterusnya", "setiap", "setiba", "setibanya",
    "setidak-tidaknya", "setidaknya", "setinggi", "seusai", "sewaktu", "siap",
    "siapa", "siapakah", "siapapun", "sini", "sinilah", "soal", "soalnya",
    "suatu", "sudah", "sudahkah", "sudahlah", "supaya", "tadi", "tadinya",
    "tahu", "tahun", "tak", "tambah", "tambahnya", "tampak", "tampaknya",
    "tandas", "tandasnya", "tanpa", "tanya", "tanyakan", "tanyanya", "tapi",
    "tegas", "tegasnya", "telah", "tempat", "tengah", "tentang", "tentu",
    "tentulah", "tentunya", "tepat", "terakhir", "terasa", "terbanyak",
    "terdahulu", "terdapat", "terdiri", "terhadap", "terhadapnya", "teringat",
    "teringat-ingat", "terjadi", "terjadilah", "terjadinya", "terkira",
    "terlalu", "terlebih", "terlihat", "termasuk", "ternyata", "tersampaikan",
    "tersebut", "tersebutlah", "tertentu", "tertuju", "terus", "terutama",
    "tetap", "tetapi", "tiap", "tiba", "tiba-tiba", "tidak", "tidakkah",
    "tidaklah", "tiga", "tinggi", "toh", "tunjuk", "turut", "tutur",
    "tuturnya", "ucap", "ucapnya", "ujar", "ujarnya", "umum", "umumnya",
    "ungkap", "ungkapnya", "untuk", "usah", "usai", "waduh", "wah", "wahai",
    "waktu", "waktunya", "walau", "walaupun", "wong", "yaitu", "yakin",
    "yakni", "yang"
},

irish={
    "a", "ach", "ag", "agus", "an", "aon", "ar", "arna", "as", "ba", "beirt",
    "bhúr", "caoga", "ceathair", "ceathrar", "chomh", "chuig", "chun", "cois",
    "céad", "cúig", "cúigear", "daichead", "dar", "de", "deich", "deichniúr",
    "den", "dhá", "do", "don", "dtí", "dá", "dár", "dó", "faoi", "faoin",
    "faoina", "faoinár", "fara", "fiche", "gach", "gan", "go", "gur", "haon",
    "hocht", "i", "iad", "idir", "in", "ina", "ins", "inár", "is", "le",
    "leis", "lena", "lenár", "mar", "mo", "muid", "mé", "na", "nach", "naoi",
    "naonúr", "ná", "ní", "níor", "nó", "nócha", "ocht", "ochtar", "ochtó",
    "os", "roimh", "sa", "seacht", "seachtar", "seachtó", "seasca", "seisear",
    "siad", "sibh", "sinn", "sna", "sé", "sí", "tar", "thar", "thú", "triúr",
    "trí", "trína", "trínár", "tríocha", "tú", "um", "ár", "é", "éis", "í",
    "ó", "ón", "óna", "ónár"
},

italian={
    "a", "abbastanza", "abbia", "abbiamo", "abbiano", "abbiate", "accidenti",
    "ad", "adesso", "affinche", "agl", "agli", "ahime", "ahimè", "ai", "al",
    "alcuna", "alcuni", "alcuno", "all", "alla", "alle", "allo", "allora",
    "altri", "altrimenti", "altro", "altrove", "altrui", "anche", "ancora",
    "anni", "anno", "ansa", "anticipo", "assai", "attesa", "attraverso",
    "avanti", "avemmo", "avendo", "avente", "aver", "avere", "averlo",
    "avesse", "avessero", "avessi", "avessimo", "aveste", "avesti", "avete",
    "aveva", "avevamo", "avevano", "avevate", "avevi", "avevo", "avrai",
    "avranno", "avrebbe", "avrebbero", "avrei", "avremmo", "avremo", "avreste",
    "avresti", "avrete", "avrà", "avrò", "avuta", "avute", "avuti", "avuto",
    "basta", "bene", "benissimo", "brava", "bravo", "casa", "caso", "cento",
    "certa", "certe", "certi", "certo", "che", "chi", "chicchessia",
    "chiunque", "ci", "ciascuna", "ciascuno", "cima", "cio", "cioe", "circa",
    "citta", "città", "co", "codesta", "codesti", "codesto", "cogli", "coi",
    "col", "colei", "coll", "coloro", "colui", "come", "cominci", "comunque",
    "con", "concernente", "conciliarsi", "conclusione", "consiglio", "contro",
    "cortesia", "cos", "cosa", "cosi", "così", "cui", "da", "dagl", "dagli",
    "dai", "dal", "dall", "dalla", "dalle", "dallo", "dappertutto", "davanti",
    "degl", "degli", "dei", "del", "dell", "della", "delle", "dello", "dentro",
    "detto", "deve", "di", "dice", "dietro", "dire", "dirimpetto", "diventa",
    "diventare", "diventato", "dopo", "dov", "dove", "dovra", "dovrà",
    "dovunque", "due", "dunque", "durante", "ebbe", "ebbero", "ebbi", "ecc",
    "ecco", "ed", "effettivamente", "egli", "ella", "entrambi", "eppure",
    "era", "erano", "eravamo", "eravate", "eri", "ero", "esempio", "esse",
    "essendo", "esser", "essere", "essi", "ex", "fa", "faccia", "facciamo",
    "facciano", "facciate", "faccio", "facemmo", "facendo", "facesse",
    "facessero", "facessi", "facessimo", "faceste", "facesti", "faceva",
    "facevamo", "facevano", "facevate", "facevi", "facevo", "fai", "fanno",
    "farai", "faranno", "fare", "farebbe", "farebbero", "farei", "faremmo",
    "faremo", "fareste", "faresti", "farete", "farà", "farò", "fatto",
    "favore", "fece", "fecero", "feci", "fin", "finalmente", "finche", "fine",
    "fino", "forse", "forza", "fosse", "fossero", "fossi", "fossimo", "foste",
    "fosti", "fra", "frattempo", "fu", "fui", "fummo", "fuori", "furono",
    "futuro", "generale", "gia", "giacche", "giorni", "giorno", "già", "gli",
    "gliela", "gliele", "glieli", "glielo", "gliene", "governo", "grande",
    "grazie", "gruppo", "ha", "haha", "hai", "hanno", "ho", "ieri", "il",
    "improvviso", "in", "inc", "infatti", "inoltre", "insieme", "intanto",
    "intorno", "invece", "io", "la", "lasciato", "lato", "lavoro", "le", "lei",
    "li", "lo", "lontano", "loro", "lui", "lungo", "luogo", "là", "ma",
    "macche", "magari", "maggior", "mai", "male", "malgrado", "malissimo",
    "mancanza", "marche", "me", "medesimo", "mediante", "meglio", "meno",
    "mentre", "mesi", "mezzo", "mi", "mia", "mie", "miei", "mila", "miliardi",
    "milioni", "minimi", "ministro", "mio", "modo", "molti", "moltissimo",
    "molto", "momento", "mondo", "mosto", "nazionale", "ne", "negl", "negli",
    "nei", "nel", "nell", "nella", "nelle", "nello", "nemmeno", "neppure",
    "nessun", "nessuna", "nessuno", "niente", "no", "noi", "non", "nondimeno",
    "nonostante", "nonsia", "nostra", "nostre", "nostri", "nostro", "novanta",
    "nove", "nulla", "nuovo", "od", "oggi", "ogni", "ognuna", "ognuno",
    "oltre", "oppure", "ora", "ore", "osi", "ossia", "ottanta", "otto",
    "paese", "parecchi", "parecchie", "parecchio", "parte", "partendo",
    "peccato", "peggio", "per", "perche", "perché", "percio", "perciò",
    "perfino", "pero", "persino", "persone", "però", "piedi", "pieno",
    "piglia", "piu", "piuttosto", "più", "po", "pochissimo", "poco", "poi",
    "poiche", "possa", "possedere", "posteriore", "posto", "potrebbe",
    "preferibilmente", "presa", "press", "prima", "primo", "principalmente",
    "probabilmente", "proprio", "puo", "pure", "purtroppo", "può", "qualche",
    "qualcosa", "qualcuna", "qualcuno", "quale", "quali", "qualunque",
    "quando", "quanta", "quante", "quanti", "quanto", "quantunque", "quasi",
    "quattro", "quel", "quella", "quelle", "quelli", "quello", "quest",
    "questa", "queste", "questi", "questo", "qui", "quindi", "realmente",
    "recente", "recentemente", "registrazione", "relativo", "riecco", "salvo",
    "sara", "sarai", "saranno", "sarebbe", "sarebbero", "sarei", "saremmo",
    "saremo", "sareste", "saresti", "sarete", "saro", "sarà", "sarò", "scola",
    "scopo", "scorso", "se", "secondo", "seguente", "seguito", "sei", "sembra",
    "sembrare", "sembrato", "sembri", "sempre", "senza", "sette", "si", "sia",
    "siamo", "siano", "siate", "siete", "sig", "solito", "solo", "soltanto",
    "sono", "sopra", "sotto", "spesso", "srl", "sta", "stai", "stando",
    "stanno", "starai", "staranno", "starebbe", "starebbero", "starei",
    "staremmo", "staremo", "stareste", "staresti", "starete", "starà", "starò",
    "stata", "state", "stati", "stato", "stava", "stavamo", "stavano",
    "stavate", "stavi", "stavo", "stemmo", "stessa", "stesse", "stessero",
    "stessi", "stessimo", "stesso", "steste", "stesti", "stette", "stettero",
    "stetti", "stia", "stiamo", "stiano", "stiate", "sto", "su", "sua",
    "subito", "successivamente", "successivo", "sue", "sugl", "sugli", "sui",
    "sul", "sull", "sulla", "sulle", "sullo", "suo", "suoi", "tale", "tali",
    "talvolta", "tanto", "te", "tempo", "ti", "titolo", "torino", "tra",
    "tranne", "tre", "trenta", "troppo", "trovato", "tu", "tua", "tue", "tuo",
    "tuoi", "tutta", "tuttavia", "tutte", "tutti", "tutto", "uguali",
    "ulteriore", "ultimo", "un", "una", "uno", "uomo", "va", "vale", "vari",
    "varia", "varie", "vario", "verso", "vi", "via", "vicino", "visto", "vita",
    "voi", "volta", "volte", "vostra", "vostre", "vostri", "vostro"
},

japanese={
    "あ", "あっ", "あまり", "あり", "ある", "あるいは", "あれ", "い", "いい", "いう", "いく", "いずれ",
    "いっ", "いつ", "いる", "いわ", "うち", "え", "お", "おい", "おけ", "および", "おら", "おり", "か",
    "かけ", "かつ", "かつて", "かなり", "から", "が", "き", "きっかけ", "くる", "くん", "こ", "こう",
    "ここ", "こと", "この", "これ", "ご", "ごと", "さ", "さらに", "さん", "し", "しか", "しかし",
    "しまう", "しまっ", "しよう", "す", "すぐ", "すべて", "する", "ず", "せ", "せい", "せる", "そう",
    "そこ", "そして", "その", "それ", "それぞれ", "た", "たい", "ただし", "たち", "ため", "たら", "たり",
    "だ", "だけ", "だっ", "ち", "ちゃん", "つ", "つい", "つけ", "つつ", "て", "で", "でき", "できる",
    "です", "と", "とき", "ところ", "とっ", "とも", "どう", "な", "ない", "なお", "なかっ", "ながら",
    "なく", "なけれ", "なし", "なっ", "など", "なら", "なり", "なる", "に", "にて", "ぬ", "ね", "の",
    "のち", "のみ", "は", "はじめ", "ば", "ひと", "ぶり", "へ", "べき", "ほか", "ほとんど", "ほど",
    "ほぼ", "ま", "ます", "また", "まで", "まま", "み", "も", "もう", "もっ", "もと", "もの", "や",
    "やっ", "よ", "よう", "よく", "よっ", "より", "よる", "よれ", "ら", "らしい", "られ", "られる",
    "る", "れ", "れる", "を", "ん", "一"
},

kazakh={
    "ах", "ох", "эх", "ай", "эй", "ой", "тағы", "тағыда", "әрине", "жоқ",
    "сондай", "осындай", "осылай", "солай", "мұндай", "бұндай", "мен",
    "сен", "ол", "біз", "біздер", "олар", "сіз", "сіздер", "маған",
    "оған", "саған", "біздің", "сіздің", "оның", "бізге", "сізге",
    "оларға", "біздерге", "сіздерге", "оларға", "менімен", "сенімен",
    "онымен", "бізбен", "сізбен", "олармен", "біздермен", "сіздермен",
    "менің", "сенің", "біздің", "сіздің", "оның", "біздердің",
    "сіздердің", "олардың", "маған", "саған", "оған", "менен", "сенен",
    "одан", "бізден", "сізден", "олардан", "біздерден", "сіздерден",
    "олардан", "айтпақшы", "сонымен", "сондықтан", "бұл", "осы", "сол",
    "анау", "мынау", "сонау", "осынау", "ана", "мына", "сона", "әні",
    "міне", "өй", "үйт", "бүйт", "біреу", "кейбіреу", "кейбір",
    "қайсыбір", "әрбір", "бірнеше", "бірдеме", "бірнеше", "әркім", "әрне",
    "әрқайсы", "әрқалай", "әлдекім", "әлдене", "әлдеқайдан", "әлденеше",
    "әлдеқалай", "әлдеқашан", "алдақашан", "еш", "ешкім", "ешбір",
    "ештеме", "дәнеңе", "ешқашан", "ешқандай", "ешқайсы", "емес", "бәрі",
    "барлық", "барша", "бар", "күллі", "бүкіл", "түгел", "өз", "өзім",
    "өзің", "өзінің", "өзіме", "өзіне", "өзімнің", "өзі", "өзге", "менде",
    "сенде", "онда", "менен", "сенен      онан", "одан", "ау", "па", "ей",
    "әй", "е", "уа", "уау", "уай", "я", "пай", "ә", "о", "оһо", "ой",
    "ие", "аһа", "ау", "беу", "мәссаған", "бәрекелді", "әттегенай",
    "жаракімалла", "масқарай", "астапыралла", "япырмай", "ойпырмай",
    "кәне", "кәнеки", "ал", "әйда", "кәні", "міне", "әні", "сорап",
    "қош-қош", "пфша", "пішә", "құрау-құрау", "шәйт", "шек", "моһ", "тәк",
    "құрау", "құр", "кә", "кәһ", "күшім", "күшім", "мышы", "пырс",
    "әукім", "алақай", "паһ-паһ", "бәрекелді", "ура", "әттең", "әттеген-ай",
    "қап", "түге", "пішту", "шіркін", "алатау", "пай-пай", "үшін",
    "сайын", "сияқты", "туралы", "арқылы", "бойы", "бойымен", "шамалы",
    "шақты", "қаралы", "ғұрлы", "ғұрлым", "шейін", "дейін", "қарай",
    "таман", "салым", "тарта", "жуық", "таяу", "гөрі", "бері", "кейін",
    "соң", "бұрын", "бетер", "қатар", "бірге", "қоса", "арс", "гүрс",
    "дүрс", "қорс", "тарс", "тырс", "ырс", "барқ", "борт", "күрт", "кірт",
    "морт", "сарт", "шырт", "дүңк", "күңк", "қыңқ", "мыңқ", "маңқ",
    "саңқ", "шаңқ", "шіңк", "сыңқ", "таңқ", "тыңқ", "ыңқ", "болп", "былп",
    "жалп", "желп", "қолп", "ірк", "ырқ", "сарт-сұрт", "тарс-тұрс", "арс-ұрс",
    "жалт-жалт", "жалт-жұлт", "қалт-қалт", "қалт-құлт", "қаңқ-қаңқ",
    "қаңқ-құңқ", "шаңқ-шаңқ", "шаңқ-шұңқ", "арбаң-арбаң", "бүгжең-бүгжең",
    "арсалаң-арсалаң", "ербелең-ербелең", "батыр-бұтыр", "далаң-далаң",
    "тарбаң-тарбаң", "қызараң-қызараң", "қаңғыр-күңгір", "қайқаң-құйқаң",
    "митың-митың", "салаң-сұлаң", "ыржың-тыржың", "бірақ", "алайда",
    "дегенмен", "әйтпесе", "әйткенмен", "себебі", "өйткені", "сондықтан",
    "үшін", "сайын", "сияқты", "туралы", "арқылы", "бойы", "бойымен",
    "шамалы", "шақты", "қаралы", "ғұрлы", "ғұрлым", "гөрі", "бері",
    "кейін", "соң", "бұрын", "бетер", "қатар", "бірге", "қоса", "шейін",
    "дейін", "қарай", "таман", "салым", "тарта", "жуық", "таяу", "арнайы",
    "осындай", "ғана", "қана", "тек", "әншейін"
},

nepali={
    "छ", "र", "पनि", "छन्", "लागि", "भएको", "गरेको", "भने", "गर्न",
    "गर्ने", "हो", "तथा", "यो", "रहेको", "उनले", "थियो", "हुने", "गरेका",
    "थिए", "गर्दै", "तर", "नै", "को", "मा", "हुन्", "भन्ने", "हुन", "गरी",
    "त", "हुन्छ", "अब", "के", "रहेका", "गरेर", "छैन", "दिए", "भए", "यस",
    "ले", "गर्नु", "औं", "सो", "त्यो", "कि", "जुन", "यी", "का", "गरि",
    "ती", "न", "छु", "छौं", "लाई", "नि", "उप", "अक्सर", "आदि", "कसरी",
    "क्रमशः", "चाले", "अगाडी", "अझै", "अनुसार", "अन्तर्गत", "अन्य",
    "अन्यत्र", "अन्यथा", "अरु", "अरुलाई", "अर्को", "अर्थात", "अर्थात्",
    "अलग", "आए", "आजको", "ओठ", "आत्म", "आफू", "आफूलाई", "आफ्नै", "आफ्नो",
    "आयो", "उदाहरण", "उनको", "उहालाई", "एउटै", "एक", "एकदम", "कतै", "कम सेकम",
    "कसै", "कसैले", "कहाँबाट", "कहिलेकाहीं", "का", "किन", "किनभने",
    "कुनै", "कुरा", "कृपया", "केही", "कोही", "गए", "गरौं", "गर्छ",
    "गर्छु", "गर्नुपर्छ", "गयौ", "गैर", "चार", "चाहनुहुन्छ", "चाहन्छु",
    "चाहिए", "छू", "जताततै", "जब", "जबकि", "जसको", "जसबाट", "जसमा",
    "जसलाई", "जसले", "जस्तै", "जस्तो", "जस्तोसुकै", "जहाँ", "जान",
    "जाहिर", "जे", "जो", "ठीक", "तत्काल", "तदनुसार", "तपाईको", "तपाई",
    "पर्याप्त", "पहिले", "पहिलो", "पहिल्यै", "पाँच", "पाँचौं", "तल",
    "तापनी", "तिनी", "तिनीहरू", "तिनीहरुको", "तिनिहरुलाई", "तिमी", "तिर",
    "तीन", "तुरुन्तै", "तेस्रो", "तेस्कारण", "पूर्व", "प्रति", "प्रतेक",
    "प्लस", "फेरी", "बने", "त्सपछि", "त्सैले", "त्यहाँ", "थिएन",
    "दिनुभएको", "दिनुहुन्छ", "दुई", "देखि", "बरु", "बारे", "बाहिर",
    "देखिन्छ", "देखियो", "देखे", "देखेको", "देखेर", "दोस्रो", "धेरै",
    "नजिकै", "नत्र", "नयाँ", "निम्ति", "बाहेक", "बीच", "बीचमा", "भन",
    "निम्न", "निम्नानुसार", "निर्दिष्ट", "नौ", "पक्का", "पक्कै", "पछि",
    "पछिल्लो", "पटक", "पर्छ", "पर्थ्यो", "भन्छन्", "भन्", "भन्छु",
    "भन्दा", "भन्नुभयो", "भर", "भित्र", "भित्री", "म", "मलाई", "मात्र",
    "माथि", "मुख्य", "मेरो", "यति", "यथोचित", "यदि", "यद्यपि", "यसको",
    "यसपछि", "यसबाहेक", "यसरी", "यसो", "यस्तो", "यहाँ", "यहाँसम्म", "या",
    "रही", "राखे", "राख्छ", "राम्रो", "रूप", "लगभग", "वरीपरी", "वास्तवमा",
    "बिरुद्ध", "बिशेष", "सायद", "शायद", "संग", "संगै", "सक्छ", "सट्टा",
    "सधै", "सबै", "सबैलाई", "समय", "सम्भव", "सम्म", "सही", "साँच्चै",
    "सात", "साथ", "साथै", "सारा", "सोही", "स्पष्ट", "हरे", "हरेक"
},

norwegian={
    "alle", "allerede", "alt", "and", "andre", "annen", "annet", "at", "av",
    "bak", "bare", "bedre", "beste", "blant", "ble", "bli", "blir", "blitt",
    "bris", "by", "både", "da", "dag", "de", "del", "dem", "den", "denne",
    "der", "dermed", "det", "dette", "disse", "drept", "du", "eller", "en",
    "enn", "er", "et", "ett", "etter", "fem", "fikk", "fire", "fjor", "flere",
    "folk", "for", "fortsatt", "fotball", "fra", "fram", "frankrike", "fredag",
    "funnet", "få", "får", "fått", "før", "først", "første", "gang", "gi",
    "gikk", "gjennom", "gjorde", "gjort", "gjør", "gjøre", "god", "godt",
    "grunn", "gå", "går", "ha", "hadde", "ham", "han", "hans", "har", "hele",
    "helt", "henne", "hennes", "her", "hun", "hva", "hvor", "hvordan",
    "hvorfor", "i", "ifølge", "igjen", "ikke", "ingen", "inn", "ja", "jeg",
    "kamp", "kampen", "kan", "kl", "klart", "kom", "komme", "kommer",
    "kontakt", "kort", "kroner", "kunne", "kveld", "kvinner", "la", "laget",
    "land", "landet", "langt", "leder", "ligger", "like", "litt", "løpet",
    "lørdag", "man", "mandag", "mange", "mannen", "mars", "med", "meg",
    "mellom", "men", "mener", "menn", "mennesker", "mens", "mer", "millioner",
    "minutter", "mot", "msci", "mye", "må", "mål", "måtte", "ned", "neste",
    "noe", "noen", "nok", "norge", "norsk", "norske", "ntb", "ny", "nye", "nå",
    "når", "og", "også", "om", "onsdag", "opp", "opplyser", "oslo", "oss",
    "over", "personer", "plass", "poeng", "politidistrikt", "politiet",
    "president", "prosent", "på", "regjeringen", "runde", "rundt", "russland",
    "sa", "saken", "samme", "sammen", "samtidig", "satt", "se", "seg", "seks",
    "selv", "senere", "september", "ser", "sett", "siden", "sier", "sin",
    "sine", "siste", "sitt", "skal", "skriver", "skulle", "slik", "som",
    "sted", "stedet", "stor", "store", "står", "sverige", "svært", "så",
    "søndag", "ta", "tatt", "tid", "tidligere", "til", "tilbake", "tillegg",
    "tirsdag", "to", "tok", "torsdag", "tre", "tror", "tyskland", "under",
    "usa", "ut", "uten", "utenfor", "vant", "var", "ved", "veldig", "vi",
    "videre", "viktig", "vil", "ville", "viser", "vår", "være", "vært", "å",
    "år", "ønsker"
},

persian={
    "آخرین", "آقای", "آمد", "آمده", "آمده‌است", "آن", "آنان", "آنجا", "آنها",
    "آنچه", "آنکه", "آورد", "آوری", "آیا", "ابتدا", "اثر", "اجرا", "اخیر",
    "از", "است", "اش", "اغلب", "افراد", "افرادی", "افزود", "البته", "اما",
    "امر", "امکان", "اند", "او", "اول", "اولین", "اکنون", "اگر", "ایشان",
    "این", "اینجا", "اینکه", "با", "بار", "باره", "باز", "باشد", "باشند",
    "باعث", "بالا", "باید", "بخش", "بخشی", "بدون", "بر", "برابر", "براساس",
    "برای", "برخی", "برداری", "بروز", "بزرگ", "بسیار", "بسیاری", "بعد", "بعضی",
    "بلکه", "بنابراین", "بندی", "به", "بهتر", "بهترین", "بود", "بودن", "بودند",
    "بوده", "بوده‌است", "بی", "بیان", "بیرون", "بیش", "بیشتر", "بیشتری", "بین",
    "تا", "تاکنون", "تبدیل", "تحت", "ترتیب", "تعداد", "تعیین", "تغییر", "تمام",
    "تمامی", "تنها", "تهیه", "تو", "جا", "جاری", "جای", "جایی", "جدی", "جدید",
    "جریان", "جز", "جمع", "جمعی", "حال", "حالا", "حالی", "حتی", "حد", "حداقل",
    "حدود", "حل", "خاص", "خاطرنشان", "خصوص", "خطر", "خواهد‌بود", "خواهد‌شد",
    "خواهد‌کرد", "خوب", "خوبی", "خود", "خودش", "خویش", "خیلی", "داد", "دادن",
    "دادند", "داده", "داده‌است", "دار", "دارای", "دارد", "دارند", "داریم",
    "داشت", "داشتن", "داشتند", "داشته", "داشته‌است", "داشته‌باشد",
    "داشته‌باشند", "دانست", "در", "درباره", "درون", "دسته", "دهد", "دهند",
    "دهه", "دو", "دوباره", "دور", "دوم", "دچار", "دیگر", "دیگران", "دیگری",
    "را", "راه", "رسید", "رسیدن", "رشد", "رفت", "رو", "روبه", "روش", "روند",
    "روی", "ریزی", "زاده", "زیاد", "زیادی", "زیر", "زیرا", "ساز", "سازی",
    "ساله", "سالهای", "سال‌های", "سایر", "سبب", "سراسر", "سعی", "سمت", "سه",
    "سهم", "سوم", "سوی", "سپس", "سی", "شامل", "شان", "شاید", "شخصی", "شد",
    "شدن", "شدند", "شده", "شده‌است", "شده‌اند", "شده‌بود", "شروع", "شش", "شما",
    "شمار", "شود", "شوند", "صرف", "ضمن", "طبق", "طرف", "طور", "طول", "طی", "ع",
    "عالی", "عدم", "علاوه", "علت", "علیه", "عهده", "عین", "غیر", "فرد", "فردی",
    "فقط", "فوق", "فکر", "قابل", "قبل", "لازم", "لحاظ", "لذا", "ما", "مانند",
    "متاسفانه", "متر", "متفاوت", "مثل", "محسوب", "مدت", "مربوط", "مشخص",
    "ممکن", "من", "مناسب", "منظور", "مهم", "مواجه", "موجب", "مورد", "می",
    "میان", "می‌آید", "می‌باشد", "می‌توان", "می‌تواند", "می‌توانند", "می‌دهد",
    "می‌دهند", "می‌رسد", "می‌رود", "می‌شد", "می‌شود", "می‌شوند", "می‌کرد",
    "می‌کردند", "می‌کند", "می‌کنم", "می‌کنند", "می‌کنیم", "می‌گوید",
    "می‌گویند", "می‌گیرد", "می‌یابد", "ناشی", "نباید", "نبود", "نحوه", "نخست",
    "نخستین", "ندارد", "ندارند", "نزدیک", "نسبت", "نشست", "نظر", "نظیر",
    "نمی‌شود", "نه", "نوع", "نوعی", "نیاز", "نیز", "نیست", "نیستند", "نیمه",
    "هایی", "هر", "هستند", "هستیم", "هم", "همان", "همه", "همواره", "همچنان",
    "همچنین", "همچون", "همیشه", "همین", "هنوز", "هنگام", "هیچ", "و", "وارد",
    "وجود", "وقتی", "ولی", "وگو", "وی", "ویژه", "پخش", "پر", "پس", "پشت",
    "پنج", "پی", "پیدا", "پیش", "چرا", "چند", "چنین", "چه", "چهار", "چهارم",
    "چون", "چگونه", "چیز", "چیزی", "کافی", "کامل", "کاملا", "کدام", "کرد",
    "کردم", "کردن", "کردند", "کرده", "کرده‌است", "کرده‌اند", "کرده‌بود",
    "کسانی", "کسی", "کل", "کلی", "کم", "کمی", "کنار", "کند", "کنم", "کنند",
    "کننده", "کنندگان", "کنید", "کنیم", "که", "کوچک", "گاه", "گذاری", "گردد",
    "گرفت", "گرفته", "گرفته‌است", "گروهی", "گفت", "گفته", "گونه", "گیرد",
    "گیری", "یا", "یابد", "یافت", "یافته", "یافته‌است", "یعنی", "یک", "یکدیگر",
    "یکی"
},

polish={
    "a", "aby", "ach", "acz", "aczkolwiek", "aj", "albo", "ale", "alez",
    "ależ", "ani", "az", "aż", "bardziej", "bardzo", "beda", "bede", "bedzie",
    "bez", "bo", "bowiem", "by", "byc", "byl", "byla", "byli", "bylo", "byly",
    "bym", "bynajmniej", "być", "był", "była", "było", "były", "będzie",
    "będą", "będę", "cala", "cali", "caly", "cała", "cały", "chce", "choć",
    "ci", "cie", "ciebie", "cię", "co", "cokolwiek", "coraz", "cos", "coś",
    "czasami", "czasem", "czemu", "czy", "czyli", "często", "daleko", "dla",
    "dlaczego", "dlatego", "do", "dobrze", "dokad", "dokąd", "dosc", "dość",
    "duzo", "dużo", "dwa", "dwaj", "dwie", "dwoje", "dzis", "dzisiaj", "dziś",
    "gdy", "gdyby", "gdyz", "gdyż", "gdzie", "gdziekolwiek", "gdzies",
    "gdzieś", "go", "godz", "i", "ich", "ile", "im", "inna", "inne", "inny",
    "innych", "iv", "ix", "iz", "iż", "ja", "jak", "jakas", "jakaś", "jakby",
    "jaki", "jakichs", "jakichś", "jakie", "jakis", "jakiz", "jakiś", "jakiż",
    "jakkolwiek", "jako", "jakos", "jakoś", "je", "jeden", "jedna", "jednak",
    "jednakze", "jednakże", "jedno", "jednym", "jedynie", "jego", "jej",
    "jemu", "jesli", "jest", "jestem", "jeszcze", "jezeli", "jeśli", "jeżeli",
    "juz", "już", "ją", "kazdy", "każdy", "kiedy", "kierunku", "kilka",
    "kilku", "kims", "kimś", "kto", "ktokolwiek", "ktora", "ktore", "ktorego",
    "ktorej", "ktory", "ktorych", "ktorym", "ktorzy", "ktos", "ktoś", "która",
    "które", "którego", "której", "który", "których", "którym", "którzy", "ku",
    "lecz", "lub", "ma", "mają", "mam", "mamy", "mało", "mi", "miał", "miedzy",
    "mimo", "między", "mna", "mnie", "mną", "moga", "mogą", "moi", "moim",
    "moj", "moja", "moje", "moze", "mozliwe", "mozna", "może", "możliwe",
    "można", "mu", "musi", "my", "mój", "na", "nad", "nam", "nami", "nas",
    "nasi", "nasz", "nasza", "nasze", "naszego", "naszych", "natomiast",
    "natychmiast", "nawet", "nia", "nic", "nich", "nie", "niech", "niego",
    "niej", "niemu", "nigdy", "nim", "nimi", "niz", "nią", "niż", "no", "o",
    "obok", "od", "ok", "około", "on", "ona", "one", "oni", "ono", "oraz",
    "oto", "owszem", "pan", "pana", "pani", "po", "pod", "podczas", "pomimo",
    "ponad", "poniewaz", "ponieważ", "powinien", "powinna", "powinni",
    "powinno", "poza", "prawie", "przeciez", "przecież", "przed", "przede",
    "przedtem", "przez", "przy", "raz", "razie", "roku", "rowniez", "również",
    "sam", "sama", "sie", "się", "skad", "skąd", "soba", "sobie", "sobą",
    "sposob", "sposób", "swoje", "są", "ta", "tak", "taka", "taki", "takich",
    "takie", "takze", "także", "tam", "te", "tego", "tej", "tel", "temu",
    "ten", "teraz", "też", "to", "toba", "tobie", "tobą", "totez", "toteż",
    "totobą", "trzeba", "tu", "tutaj", "twoi", "twoim", "twoj", "twoja",
    "twoje", "twym", "twój", "ty", "tych", "tylko", "tym", "tys", "tzw", "tę",
    "u", "vi", "vii", "viii", "w", "wam", "wami", "was", "wasi", "wasz",
    "wasza", "wasze", "we", "według", "wie", "wiele", "wielu", "więc",
    "więcej", "wlasnie", "wszyscy", "wszystkich", "wszystkie", "wszystkim",
    "wszystko", "wtedy", "wy", "właśnie", "wśród", "xi", "xii", "xiii", "xiv",
    "xv", "z", "za", "zaden", "zadna", "zadne", "zadnych", "zapewne", "zawsze",
    "zaś", "ze", "zeby", "znow", "znowu", "znów", "zostal", "został", "żaden",
    "żadna", "żadne", "żadnych", "że", "żeby"
},

portuguese={
    "acerca", "ademais", "adeus", "agora", "ainda", "algo", "algumas",
    "alguns", "ali", "além", "ambas", "ambos", "antes", "ao", "aos", "apenas",
    "apoia", "apoio", "apontar", "após", "aquela", "aquelas", "aquele",
    "aqueles", "aqui", "aquilo", "as", "assim", "através", "atrás", "até",
    "aí", "baixo", "bastante", "bem", "boa", "bom", "breve", "cada", "caminho",
    "catorze", "cedo", "cento", "certamente", "certeza", "cima", "cinco",
    "coisa", "com", "como", "comprida", "comprido", "conhecida", "conhecido",
    "conselho", "contra", "contudo", "corrente", "cuja", "cujo", "custa", "cá",
    "da", "daquela", "daquele", "dar", "das", "de", "debaixo", "demais",
    "dentro", "depois", "des", "desde", "dessa", "desse", "desta", "deste",
    "deve", "devem", "deverá", "dez", "dezanove", "dezasseis", "dezassete",
    "dezoito", "diante", "direita", "disso", "diz", "dizem", "dizer", "do",
    "dois", "dos", "doze", "duas", "dá", "dão", "ela", "elas", "ele", "eles",
    "em", "embora", "enquanto", "entre", "então", "era", "essa", "essas",
    "esse", "esses", "esta", "estado", "estar", "estará", "estas", "estava",
    "este", "estes", "esteve", "estive", "estivemos", "estiveram", "estiveste",
    "estivestes", "estou", "está", "estás", "estão", "eu", "eventual",
    "exemplo", "falta", "fará", "favor", "faz", "fazeis", "fazem", "fazemos",
    "fazer", "fazes", "fazia", "faço", "fez", "fim", "final", "foi", "fomos",
    "for", "fora", "foram", "forma", "foste", "fostes", "fui", "geral",
    "grande", "grandes", "grupo", "inclusive", "iniciar", "inicio", "ir",
    "irá", "isso", "isto", "já", "lado", "lhe", "ligado", "local", "logo",
    "longe", "lugar", "lá", "maior", "maioria", "maiorias", "mais", "mal",
    "mas", "me", "meio", "menor", "menos", "meses", "mesmo", "meu", "meus",
    "mil", "minha", "minhas", "momento", "muito", "muitos", "máximo", "mês",
    "na", "nada", "naquela", "naquele", "nas", "nem", "nenhuma", "nessa",
    "nesse", "nesta", "neste", "no", "nos", "nossa", "nossas", "nosso",
    "nossos", "nova", "novas", "nove", "novo", "novos", "num", "numa", "nunca",
    "nuns", "não", "nível", "nós", "número", "números", "obrigada", "obrigado",
    "oitava", "oitavo", "oito", "onde", "ontem", "onze", "ora", "os", "ou",
    "outra", "outras", "outros", "para", "parece", "parte", "partir", "pegar",
    "pela", "pelas", "pelo", "pelos", "perto", "pode", "podem", "poder",
    "poderá", "podia", "pois", "ponto", "pontos", "por", "porquanto", "porque",
    "porquê", "portanto", "porém", "posição", "possivelmente", "posso",
    "possível", "pouca", "pouco", "povo", "primeira", "primeiro", "próprio",
    "próxima", "próximo", "puderam", "pôde", "põe", "põem", "quais", "qual",
    "qualquer", "quando", "quanto", "quarta", "quarto", "quatro", "que",
    "quem", "quer", "querem", "quero", "questão", "quieta", "quieto", "quinta",
    "quinto", "quinze", "quê", "relação", "sabe", "saber", "se", "segunda",
    "segundo", "sei", "seis", "sem", "sempre", "ser", "seria", "sete", "seu",
    "seus", "sexta", "sexto", "sim", "sistema", "sob", "sobre", "sois",
    "somente", "somos", "sou", "sua", "suas", "são", "sétima", "sétimo", "só",
    "tais", "tal", "talvez", "também", "tanta", "tanto", "tarde", "te", "tem",
    "temos", "tempo", "tendes", "tenho", "tens", "tentar", "tentaram", "tente",
    "tentei", "ter", "terceira", "terceiro", "teu", "teus", "teve", "tipo",
    "tive", "tivemos", "tiveram", "tiveste", "tivestes", "toda", "todas",
    "todo", "todos", "treze", "três", "tu", "tua", "tuas", "tudo", "tão",
    "têm", "um", "uma", "umas", "uns", "usa", "usar", "vai", "vais", "valor",
    "veja", "vem", "vens", "ver", "vez", "vezes", "vinda", "vindo", "vinte",
    "você", "vocês", "vos", "vossa", "vossas", "vosso", "vossos", "vários",
    "vão", "vêm", "vós", "zero", "à", "às", "área", "é", "és", "último"
},

romanian={
    "a", "abia", "acea", "aceasta", "această", "aceea", "aceeasi", "acei",
    "aceia", "acel", "acela", "acelasi", "acele", "acelea", "acest", "acesta",
    "aceste", "acestea", "acestei", "acestia", "acestui", "aceşti", "aceştia",
    "acești", "aceștia", "acolo", "acord", "acum", "adica", "ai", "aia",
    "aibă", "aici", "aiurea", "al", "ala", "alaturi", "ale", "alea", "alt",
    "alta", "altceva", "altcineva", "alte", "altfel", "alti", "altii", "altul",
    "alături", "am", "anume", "apoi", "ar", "are", "as", "asa", "asemenea",
    "asta", "astazi", "astea", "astfel", "astăzi", "asupra", "atare", "atat",
    "atata", "atatea", "atatia", "ati", "atit", "atita", "atitea", "atitia",
    "atunci", "au", "avea", "avem", "aveţi", "aveți", "avut", "azi", "aş",
    "aşadar", "aţi", "aș", "așadar", "ați", "b", "ba", "bine", "bucur", "bună",
    "c", "ca", "cam", "cand", "capat", "care", "careia", "carora", "caruia",
    "cat", "catre", "caut", "ce", "cea", "ceea", "cei", "ceilalti", "cel",
    "cele", "celor", "ceva", "chiar", "ci", "cinci", "cind", "cine", "cineva",
    "cit", "cita", "cite", "citeva", "citi", "citiva", "conform", "contra",
    "cu", "cui", "cum", "cumva", "curând", "curînd", "când", "cât", "câte",
    "câtva", "câţi", "câți", "cînd", "cît", "cîte", "cîtva", "cîţi", "cîți",
    "că", "căci", "cărei", "căror", "cărui", "către", "d", "da", "daca",
    "dacă", "dar", "dat", "datorită", "dată", "dau", "de", "deasupra", "deci",
    "decit", "degraba", "deja", "deoarece", "departe", "desi", "despre",
    "deşi", "deși", "din", "dinaintea", "dintr", "dintr-", "dintre", "doar",
    "doi", "doilea", "două", "drept", "dupa", "după", "dă", "e", "ea", "ei",
    "el", "ele", "era", "eram", "este", "eu", "exact", "eşti", "ești", "f",
    "face", "fara", "fata", "fel", "fi", "fie", "fiecare", "fii", "fim", "fiu",
    "fiţi", "fiți", "foarte", "fost", "frumos", "fără", "g", "geaba", "graţie",
    "grație", "h", "halbă", "i", "ia", "iar", "ieri", "ii", "il", "imi", "in",
    "inainte", "inapoi", "inca", "incit", "insa", "intr", "intre", "isi",
    "iti", "j", "k", "l", "la", "le", "li", "lor", "lui", "lângă", "lîngă",
    "m", "ma", "mai", "mare", "mea", "mei", "mele", "mereu", "meu", "mi",
    "mie", "mine", "mod", "mult", "multa", "multe", "multi", "multă", "mulţi",
    "mulţumesc", "mulți", "mulțumesc", "mâine", "mîine", "mă", "n", "ne",
    "nevoie", "ni", "nici", "niciodata", "nicăieri", "nimeni", "nimeri",
    "nimic", "niste", "nişte", "niște", "noastre", "noastră", "noi", "noroc",
    "nostri", "nostru", "nou", "noua", "nouă", "noştri", "noștri", "nu",
    "numai", "o", "opt", "or", "ori", "oricare", "orice", "oricine", "oricum",
    "oricând", "oricât", "oricînd", "oricît", "oriunde", "p", "pai", "parca",
    "patra", "patru", "patrulea", "pe", "pentru", "peste", "pic", "pina",
    "plus", "poate", "pot", "prea", "prima", "primul", "prin", "printr-",
    "putini", "puţin", "puţina", "puţină", "puțin", "puțina", "puțină", "până",
    "pînă", "r", "rog", "s", "sa", "sa-mi", "sa-ti", "sai", "sale", "sau",
    "se", "si", "sint", "sintem", "spate", "spre", "sub", "sunt", "suntem",
    "sunteţi", "sunteți", "sus", "sută", "sînt", "sîntem", "sînteţi",
    "sînteți", "să", "săi", "său", "t", "ta", "tale", "te", "ti", "timp",
    "tine", "toata", "toate", "toată", "tocmai", "tot", "toti", "totul",
    "totusi", "totuşi", "totuși", "toţi", "toți", "trei", "treia", "treilea",
    "tu", "tuturor", "tăi", "tău", "u", "ul", "ului", "un", "una", "unde",
    "undeva", "unei", "uneia", "unele", "uneori", "unii", "unor", "unora",
    "unu", "unui", "unuia", "unul", "v", "va", "vi", "voastre", "voastră",
    "voi", "vom", "vor", "vostru", "vouă", "voştri", "voștri", "vreme", "vreo",
    "vreun", "vă", "x", "z", "zece", "zero", "zi", "zice", "îi", "îl", "îmi",
    "împotriva", "în", "înainte", "înaintea", "încotro", "încât", "încît",
    "între", "întrucât", "întrucît", "îţi", "îți", "ăla", "ălea", "ăsta",
    "ăstea", "ăştia", "ăștia", "şapte", "şase", "şi", "ştiu", "ţi", "ţie",
    "șapte", "șase", "și", "știu", "ți", "ție"
},

russian={
    "а", "будем", "будет", "будете", "будешь", "буду", "будут", "будучи",
    "будь", "будьте", "бы", "был", "была", "были", "было", "быть", "в", "вам",
    "вами", "вас", "весь", "во", "вот", "все", "всего", "всей", "всем",
    "всеми", "всему", "всех", "всею", "всея", "всю", "вся", "всё", "всём",
    "вы", "да", "для", "до", "его", "едим", "едят", "ее", "ей", "ел", "ела",
    "ем", "ему", "емъ", "если", "ест", "есть", "ешь", "еще", "ещё", "ею", "её",
    "же", "за", "и", "из", "или", "им", "ими", "имъ", "их", "к", "как", "кем",
    "ко", "когда", "кого", "ком", "кому", "комья", "которая", "которого",
    "которое", "которой", "котором", "которому", "которою", "которую",
    "которые", "который", "которым", "которыми", "которых", "кто", "меня",
    "мне", "мной", "мною", "мог", "моги", "могите", "могла", "могли", "могло",
    "могу", "могут", "мое", "моего", "моей", "моем", "моему", "моею", "можем",
    "может", "можете", "можешь", "мои", "моим", "моими", "моих", "мой", "мочь",
    "мою", "моя", "моё", "моём", "мы", "на", "нам", "нами", "нас", "наса",
    "наш", "наша", "наше", "нашего", "нашей", "нашем", "нашему", "нашею",
    "наши", "нашим", "нашими", "наших", "нашу", "не", "него", "нее", "ней",
    "нем", "нему", "нет", "нею", "неё", "ним", "ними", "них", "но", "нём", "о",
    "об", "один", "одна", "одни", "одним", "одними", "одних", "одно", "одного",
    "одной", "одном", "одному", "одною", "одну", "он", "она", "оне", "они",
    "оно", "от", "по", "при", "с", "сам", "сама", "сами", "самим", "самими",
    "самих", "само", "самого", "самом", "самому", "саму", "свое", "своего",
    "своей", "своем", "своему", "своею", "свои", "своим", "своими", "своих",
    "свой", "свою", "своя", "своё", "своём", "себе", "себя", "собой", "собою",
    "та", "так", "такая", "такие", "таким", "такими", "таких", "такого",
    "такое", "такой", "таком", "такому", "такою", "такую", "те", "тебе",
    "тебя", "тем", "теми", "тех", "то", "тобой", "тобою", "того", "той",
    "только", "том", "томах", "тому", "тот", "тою", "ту", "ты", "у", "уже",
    "чего", "чем", "чему", "что", "чтобы", "чём", "эта", "эти", "этим",
    "этими", "этих", "это", "этого", "этой", "этом", "этому", "этот", "этою",
    "эту", "я"
},

sinhala={
    "අතර", "එච්චර", "එපමණ", "එලෙස", "එවිට", "ඒ", "කට", "කදී", "ක්", "කින්",
    "ට", "ත්", "තුර", "ද", "නමුත්", "නොහොත්", "පමණ", "පමණි", "ම", "මෙච්චර",
    "මෙපමණ", "මෙලෙස", "මෙවිට", "මේ", "ය", "යි", "ලදී", "ලෙස", "වගේ", "වන",
    "විට", "විටෙක", "විතර", "විය", "වුව", "වුවත්", "වුවද", "වූ", "සමඟ", "සහ",
    "හා", "හෙවත්", "හෝ"
},

spanish={
    "actualmente", "acuerdo", "adelante", "ademas", "además", "adrede",
    "afirmó", "agregó", "ahi", "ahora", "ahí", "al", "algo", "alguna",
    "algunas", "alguno", "algunos", "algún", "alli", "allí", "alrededor",
    "ambos", "ampleamos", "antano", "antaño", "ante", "anterior", "antes",
    "apenas", "aproximadamente", "aquel", "aquella", "aquellas", "aquello",
    "aquellos", "aqui", "aquél", "aquélla", "aquéllas", "aquéllos", "aquí",
    "arriba", "arribaabajo", "aseguró", "asi", "así", "atras", "aun", "aunque",
    "ayer", "añadió", "aún", "bajo", "bastante", "bien", "breve", "buen",
    "buena", "buenas", "bueno", "buenos", "cada", "casi", "cerca", "cierta",
    "ciertas", "cierto", "ciertos", "cinco", "claro", "comentó", "como", "con",
    "conmigo", "conocer", "conseguimos", "conseguir", "considera", "consideró",
    "consigo", "consigue", "consiguen", "consigues", "contigo", "contra",
    "cosas", "creo", "cual", "cuales", "cualquier", "cuando", "cuanta",
    "cuantas", "cuanto", "cuantos", "cuatro", "cuenta", "cuál", "cuáles",
    "cuándo", "cuánta", "cuántas", "cuánto", "cuántos", "cómo", "da", "dado",
    "dan", "dar", "de", "debajo", "debe", "deben", "debido", "decir", "dejó",
    "del", "delante", "demasiado", "demás", "dentro", "deprisa", "desde",
    "despacio", "despues", "después", "detras", "detrás", "dia", "dias",
    "dice", "dicen", "dicho", "dieron", "diferente", "diferentes", "dijeron",
    "dijo", "dio", "donde", "dos", "durante", "día", "días", "dónde",
    "ejemplo", "el", "ella", "ellas", "ello", "ellos", "embargo", "empleais",
    "emplean", "emplear", "empleas", "empleo", "en", "encima", "encuentra",
    "enfrente", "enseguida", "entonces", "entre", "era", "eramos", "eran",
    "eras", "eres", "es", "esa", "esas", "ese", "eso", "esos", "esta",
    "estaba", "estaban", "estado", "estados", "estais", "estamos", "estan",
    "estar", "estará", "estas", "este", "esto", "estos", "estoy", "estuvo",
    "está", "están", "ex", "excepto", "existe", "existen", "explicó",
    "expresó", "fin", "final", "fue", "fuera", "fueron", "fui", "fuimos",
    "general", "gran", "grandes", "gueno", "ha", "haber", "habia", "habla",
    "hablan", "habrá", "había", "habían", "hace", "haceis", "hacemos", "hacen",
    "hacer", "hacerlo", "haces", "hacia", "haciendo", "hago", "han", "hasta",
    "hay", "haya", "he", "hecho", "hemos", "hicieron", "hizo", "horas", "hoy",
    "hubo", "igual", "incluso", "indicó", "informo", "informó", "intenta",
    "intentais", "intentamos", "intentan", "intentar", "intentas", "intento",
    "ir", "junto", "la", "lado", "largo", "las", "le", "lejos", "les", "llegó",
    "lleva", "llevar", "lo", "los", "luego", "lugar", "mal", "manera",
    "manifestó", "mas", "mayor", "me", "mediante", "medio", "mejor",
    "mencionó", "menos", "menudo", "mi", "mia", "mias", "mientras", "mio",
    "mios", "mis", "misma", "mismas", "mismo", "mismos", "modo", "momento",
    "mucha", "muchas", "mucho", "muchos", "muy", "más", "mí", "mía", "mías",
    "mío", "míos", "nada", "nadie", "ni", "ninguna", "ningunas", "ninguno",
    "ningunos", "ningún", "no", "nos", "nosotras", "nosotros", "nuestra",
    "nuestras", "nuestro", "nuestros", "nueva", "nuevas", "nuevo", "nuevos",
    "nunca", "ocho", "os", "otra", "otras", "otro", "otros", "pais", "para",
    "parece", "parte", "partir", "pasada", "pasado", "paìs", "peor", "pero",
    "pesar", "poca", "pocas", "poco", "pocos", "podeis", "podemos", "poder",
    "podria", "podriais", "podriamos", "podrian", "podrias", "podrá", "podrán",
    "podría", "podrían", "poner", "por", "porque", "posible", "primer",
    "primera", "primero", "primeros", "principalmente", "pronto", "propia",
    "propias", "propio", "propios", "proximo", "próximo", "próximos", "pudo",
    "pueda", "puede", "pueden", "puedo", "pues", "qeu", "que", "quedó",
    "queremos", "quien", "quienes", "quiere", "quiza", "quizas", "quizá",
    "quizás", "quién", "quiénes", "qué", "raras", "realizado", "realizar",
    "realizó", "repente", "respecto", "sabe", "sabeis", "sabemos", "saben",
    "saber", "sabes", "salvo", "se", "sea", "sean", "segun", "segunda",
    "segundo", "según", "seis", "ser", "sera", "será", "serán", "sería",
    "señaló", "si", "sido", "siempre", "siendo", "siete", "sigue", "siguiente",
    "sin", "sino", "sobre", "sois", "sola", "solamente", "solas", "solo",
    "solos", "somos", "son", "soy", "soyos", "su", "supuesto", "sus", "suya",
    "suyas", "suyo", "sé", "sí", "sólo", "tal", "tambien", "también",
    "tampoco", "tan", "tanto", "tarde", "te", "temprano", "tendrá", "tendrán",
    "teneis", "tenemos", "tener", "tenga", "tengo", "tenido", "tenía",
    "tercera", "ti", "tiempo", "tiene", "tienen", "toda", "todas", "todavia",
    "todavía", "todo", "todos", "total", "trabaja", "trabajais", "trabajamos",
    "trabajan", "trabajar", "trabajas", "trabajo", "tras", "trata", "través",
    "tres", "tu", "tus", "tuvo", "tuya", "tuyas", "tuyo", "tuyos", "tú",
    "ultimo", "un", "una", "unas", "uno", "unos", "usa", "usais", "usamos",
    "usan", "usar", "usas", "uso", "usted", "ustedes", "va", "vais", "valor",
    "vamos", "van", "varias", "varios", "vaya", "veces", "ver", "verdad",
    "verdadera", "verdadero", "vez", "vosotras", "vosotros", "voy", "vuestra",
    "vuestras", "vuestro", "vuestros", "ya", "yo", "él", "ésa", "ésas", "ése",
    "ésos", "ésta", "éstas", "éste", "éstos", "última", "últimas", "último",
    "últimos"
},

swedish={
    "aderton", "adertonde", "adjö", "aldrig", "alla", "allas", "allt",
    "alltid", "alltså", "andra", "andras", "annan", "annat", "arton",
    "artonde", "att", "av", "bakom", "bara", "behöva", "behövas", "behövde",
    "behövt", "beslut", "beslutat", "beslutit", "bland", "blev", "bli", "blir",
    "blivit", "bort", "borta", "bra", "bäst", "bättre", "båda", "bådas", "dag",
    "dagar", "dagarna", "dagen", "de", "del", "delen", "dem", "den", "deras",
    "dess", "det", "detta", "dig", "din", "dina", "dit", "ditt", "dock", "du",
    "där", "därför", "då", "efter", "eftersom", "elfte", "eller", "elva", "en",
    "enkel", "enkelt", "enkla", "enligt", "er", "era", "ert", "ett", "ettusen",
    "fanns", "fem", "femte", "femtio", "femtionde", "femton", "femtonde",
    "fick", "fin", "finnas", "finns", "fjorton", "fjortonde", "fjärde", "fler",
    "flera", "flesta", "fram", "framför", "från", "fyra", "fyrtio",
    "fyrtionde", "få", "får", "fått", "följande", "för", "före", "förlåt",
    "förra", "första", "genast", "genom", "gick", "gjorde", "gjort", "god",
    "goda", "godare", "godast", "gott", "gälla", "gäller", "gällt", "gärna",
    "gå", "går", "gått", "gör", "göra", "ha", "hade", "haft", "han", "hans",
    "har", "heller", "hellre", "helst", "helt", "henne", "hennes", "hit",
    "hon", "honom", "hundra", "hundraen", "hundraett", "hur", "här", "hög",
    "höger", "högre", "högst", "i", "ibland", "idag", "igen", "igår",
    "imorgon", "in", "inför", "inga", "ingen", "ingenting", "inget", "innan",
    "inne", "inom", "inte", "inuti", "ja", "jag", "jämfört", "kan", "kanske",
    "knappast", "kom", "komma", "kommer", "kommit", "kr", "kunde", "kunna",
    "kunnat", "kvar", "legat", "ligga", "ligger", "lika", "likställd",
    "likställda", "lilla", "lite", "liten", "litet", "länge", "längre",
    "längst", "lätt", "lättare", "lättast", "långsam", "långsammare",
    "långsammast", "långsamt", "långt", "man", "med", "mellan", "men", "mer",
    "mera", "mest", "mig", "min", "mina", "mindre", "minst", "mitt",
    "mittemot", "mot", "mycket", "många", "måste", "möjlig", "möjligen",
    "möjligt", "möjligtvis", "ned", "nederst", "nedersta", "nedre", "nej",
    "ner", "ni", "nio", "nionde", "nittio", "nittionde", "nitton", "nittonde",
    "nog", "noll", "nr", "nu", "nummer", "när", "nästa", "någon", "någonting",
    "något", "några", "nödvändig", "nödvändiga", "nödvändigt", "nödvändigtvis",
    "och", "också", "ofta", "oftast", "olika", "olikt", "om", "oss", "på",
    "rakt", "redan", "rätt", "sade", "sagt", "samma", "sedan", "senare",
    "senast", "sent", "sex", "sextio", "sextionde", "sexton", "sextonde",
    "sig", "sin", "sina", "sist", "sista", "siste", "sitt", "sju", "sjunde",
    "sjuttio", "sjuttionde", "sjutton", "sjuttonde", "sjätte", "ska", "skall",
    "skulle", "slutligen", "små", "smått", "snart", "som", "stor", "stora",
    "stort", "större", "störst", "säga", "säger", "sämre", "sämst", "så",
    "tack", "tidig", "tidigare", "tidigast", "tidigt", "till", "tills",
    "tillsammans", "tio", "tionde", "tjugo", "tjugoen", "tjugoett", "tjugonde",
    "tjugotre", "tjugotvå", "tjungo", "tolfte", "tolv", "tre", "tredje",
    "trettio", "trettionde", "tretton", "trettonde", "två", "tvåhundra",
    "under", "upp", "ur", "ursäkt", "ut", "utan", "utanför", "ute", "vad",
    "var", "vara", "varför", "varifrån", "varit", "varken", "varsågod", "vart",
    "vem", "vems", "verkligen", "vi", "vid", "vidare", "viktig", "viktigare",
    "viktigast", "viktigt", "vilka", "vilken", "vilket", "vill", "vänster",
    "vänstra", "värre", "vår", "våra", "vårt", "än", "ännu", "även",
    "åtminstone", "åtta", "åttio", "åttionde", "åttonde", "över", "övermorgon",
    "överst", "övre"
},

tagalog={
    "akin", "aking", "ako", "alin", "am", "amin", "aming", "ang", "ano",
    "anumang", "apat", "at", "atin", "ating", "ay", "bababa", "bago", "bakit",
    "bawat", "bilang", "dahil", "dalawa", "dapat", "din", "dito", "doon",
    "gagawin", "gayunman", "ginagawa", "ginawa", "ginawang", "gumawa", "gusto",
    "habang", "hanggang", "hindi", "huwag", "iba", "ibaba", "ibabaw", "ibig",
    "ikaw", "ilagay", "ilalim", "ilan", "inyong", "isa", "isang", "itaas",
    "ito", "iyo", "iyon", "iyong", "ka", "kahit", "kailangan", "kailanman",
    "kami", "kanila", "kanilang", "kanino", "kanya", "kanyang", "kapag",
    "kapwa", "karamihan", "katiyakan", "katulad", "kaya", "kaysa", "ko",
    "kong", "kulang", "kumuha", "kung", "laban", "lahat", "lamang", "likod",
    "lima", "maaari", "maaaring", "maging", "mahusay", "makita", "marami",
    "marapat", "masyado", "may", "mayroon", "mga", "minsan", "mismo", "mula",
    "muli", "na", "nabanggit", "naging", "nagkaroon", "nais", "nakita",
    "namin", "napaka", "narito", "nasaan", "ng", "ngayon", "ni", "nila",
    "nilang", "nito", "niya", "niyang", "noon", "o", "pa", "paano", "pababa",
    "paggawa", "pagitan", "pagkakaroon", "pagkatapos", "palabas",
    "pamamagitan", "panahon", "pangalawa", "para", "paraan", "pareho",
    "pataas", "pero", "pumunta", "pumupunta", "sa", "saan", "sabi", "sabihin",
    "sarili", "sila", "sino", "siya", "tatlo", "tayo", "tulad", "tungkol",
    "una", "walang"
},

tamil={
    "அங்கு", "அங்கே", "அடுத்த", "அதனால்", "அதன்", "அதற்கு", "அதிக", "அதில்",
    "அது", "அதே", "அதை", "அந்த", "அந்தக்", "அந்தப்", "அன்று", "அல்லது", "அவன்",
    "அவரது", "அவர்", "அவர்கள்", "அவள்", "அவை", "ஆகிய", "ஆகியோர்", "ஆகும்",
    "இங்கு", "இங்கே", "இடத்தில்", "இடம்", "இதனால்", "இதனை", "இதன்", "இதற்கு",
    "இதில்", "இது", "இதை", "இந்த", "இந்தக்", "இந்தத்", "இந்தப்", "இன்னும்",
    "இப்போது", "இரு", "இருக்கும்", "இருந்த", "இருந்தது", "இருந்து", "இவர்",
    "இவை", "உன்", "உள்ள", "உள்ளது", "உள்ளன", "எந்த", "என", "எனக்", "எனக்கு",
    "எனப்படும்", "எனவும்", "எனவே", "எனினும்", "எனும்", "என்", "என்ன",
    "என்னும்", "என்பது", "என்பதை", "என்ற", "என்று", "என்றும்", "எல்லாம்",
    "ஏன்", "ஒரு", "ஒரே", "ஓர்", "கொண்ட", "கொண்டு", "கொள்ள", "சற்று", "சிறு",
    "சில", "சேர்ந்த", "தனது", "தன்", "தவிர", "தான்", "நான்", "நாம்", "நீ",
    "பற்றி", "பற்றிய", "பல", "பலரும்", "பல்வேறு", "பின்", "பின்னர்", "பிற",
    "பிறகு", "பெரும்", "பேர்", "போது", "போன்ற", "போல", "போல்", "மட்டுமே",
    "மட்டும்", "மற்ற", "மற்றும்", "மிக", "மிகவும்", "மீது", "முதல்", "முறை",
    "மேலும்", "மேல்", "யார்", "வந்த", "வந்து", "வரும்", "வரை", "வரையில்",
    "விட", "விட்டு", "வேண்டும்", "வேறு"
},

tatar={
    "алай", "алайса", "алар", "аларга", "аларда", "алардан", "аларны",
    "аларның", "аларча", "алары", "аларын", "аларынга", "аларында",
    "аларыннан", "аларының", "алтмыш", "алтмышынчы", "алтмышынчыга",
    "алтмышынчыда", "алтмышынчыдан", "алтмышынчылар", "алтмышынчыларга",
    "алтмышынчыларда", "алтмышынчылардан", "алтмышынчыларны",
    "алтмышынчыларның", "алтмышынчыны", "алтмышынчының", "алты", "алтылап",
    "алтынчы", "алтынчыга", "алтынчыда", "алтынчыдан", "алтынчылар",
    "алтынчыларга", "алтынчыларда", "алтынчылардан", "алтынчыларны",
    "алтынчыларның", "алтынчыны", "алтынчының", "алтышар", "анда", "андагы",
    "андай", "андый", "андыйга", "андыйда", "андыйдан", "андыйны", "андыйның",
    "аннан", "ансы", "анча", "аны", "аныкы", "аныкын", "аныкынга", "аныкында",
    "аныкыннан", "аныкының", "анысы", "анысын", "анысынга", "анысында",
    "анысыннан", "анысының", "аның", "аныңча", "аркылы", "ары", "аша", "аңа",
    "аңар", "аңарга", "аңарда", "аңардагы", "аңардан", "бар", "бара", "барлык",
    "барча", "барчасы", "барчасын", "барчасына", "барчасында", "барчасыннан",
    "барчасының", "бары", "башка", "башкача", "бе­лән", "без", "безгә",
    "бездә", "бездән", "безне", "безнең", "безнеңчә", "белдерүенчә", "белән",
    "бер", "бергә", "беренче", "беренчегә", "беренчедә", "беренчедән",
    "беренчеләр", "беренчеләргә", "беренчеләрдә", "беренчеләрдән",
    "беренчеләрне", "беренчеләрнең", "беренчене", "беренченең", "беркайда",
    "беркайсы", "беркая", "беркаян", "беркем", "беркемгә", "беркемдә",
    "беркемне", "беркемнең", "беркемнән", "берлән", "берни", "бернигә",
    "бернидә", "бернидән", "бернинди", "бернине", "бернинең", "берничек",
    "берничә", "бернәрсә", "бернәрсәгә", "бернәрсәдә", "бернәрсәдән",
    "бернәрсәне", "бернәрсәнең", "беррәттән", "берсе", "берсен", "берсенгә",
    "берсендә", "берсенең", "берсеннән", "берәр", "берәрсе", "берәрсен",
    "берәрсендә", "берәрсенең", "берәрсеннән", "берәрсенә", "берәү", "бигрәк",
    "бик", "бирле", "бит", "биш", "бишенче", "бишенчегә", "бишенчедә",
    "бишенчедән", "бишенчеләр", "бишенчеләргә", "бишенчеләрдә",
    "бишенчеләрдән", "бишенчеләрне", "бишенчеләрнең", "бишенчене",
    "бишенченең", "бишләп", "болай", "болар", "боларга", "боларда", "болардан",
    "боларны", "боларның", "болары", "боларын", "боларынга", "боларында",
    "боларыннан", "боларының", "бу", "буе", "буена", "буенда", "буенча",
    "буйлап", "буларак", "булачак", "булды", "булмый", "булса", "булып",
    "булыр", "булырга", "бусы", "бүтән", "бәлки", "бән", "бәрабәренә", "бөтен",
    "бөтенесе", "бөтенесен", "бөтенесендә", "бөтенесенең", "бөтенесеннән",
    "бөтенесенә", "вә", "гел", "генә", "гына", "гүя", "гүяки", "гәрчә", "да",
    "ди", "дигән", "диде", "дип", "дистәләгән", "дистәләрчә", "дүрт",
    "дүртенче", "дүртенчегә", "дүртенчедә", "дүртенчедән", "дүртенчеләр",
    "дүртенчеләргә", "дүртенчеләрдә", "дүртенчеләрдән", "дүртенчеләрне",
    "дүртенчеләрнең", "дүртенчене", "дүртенченең", "дүртләп", "дә", "егерме",
    "егерменче", "егерменчегә", "егерменчедә", "егерменчедән", "егерменчеләр",
    "егерменчеләргә", "егерменчеләрдә", "егерменчеләрдән", "егерменчеләрне",
    "егерменчеләрнең", "егерменчене", "егерменченең", "ел", "елда", "иде",
    "идек", "идем", "ике", "икенче", "икенчегә", "икенчедә", "икенчедән",
    "икенчеләр", "икенчеләргә", "икенчеләрдә", "икенчеләрдән", "икенчеләрне",
    "икенчеләрнең", "икенчене", "икенченең", "икешәр", "икән", "илле",
    "илленче", "илленчегә", "илленчедә", "илленчедән", "илленчеләр",
    "илленчеләргә", "илленчеләрдә", "илленчеләрдән", "илленчеләрне",
    "илленчеләрнең", "илленчене", "илленченең", "илә", "илән", "инде", "исә",
    "итеп", "иткән", "итте", "итү", "итә", "итәргә", "иң", "йөз", "йөзенче",
    "йөзенчегә", "йөзенчедә", "йөзенчедән", "йөзенчеләр", "йөзенчеләргә",
    "йөзенчеләрдә", "йөзенчеләрдән", "йөзенчеләрне", "йөзенчеләрнең",
    "йөзенчене", "йөзенченең", "йөзләгән", "йөзләрчә", "йөзәрләгән", "кадәр",
    "кай", "кайбер", "кайберләре", "кайберсе", "кайберәү", "кайберәүгә",
    "кайберәүдә", "кайберәүдән", "кайберәүне", "кайберәүнең", "кайдагы",
    "кайсы", "кайсыбер", "кайсын", "кайсына", "кайсында", "кайсыннан",
    "кайсының", "кайчангы", "кайчандагы", "кайчаннан", "караганда",
    "карамастан", "карамый", "карата", "каршы", "каршына", "каршында",
    "каршындагы", "кебек", "кем", "кемгә", "кемдә", "кемне", "кемнең",
    "кемнән", "кенә", "ки", "килеп", "килә", "кирәк", "кына", "кырыгынчы",
    "кырыгынчыга", "кырыгынчыда", "кырыгынчыдан", "кырыгынчылар",
    "кырыгынчыларга", "кырыгынчыларда", "кырыгынчылардан", "кырыгынчыларны",
    "кырыгынчыларның", "кырыгынчыны", "кырыгынчының", "кырык", "күк",
    "күпләгән", "күпме", "күпмеләп", "күпмешәр", "күпмешәрләп", "күптән",
    "күрә", "ләкин", "максатында", "менә", "мең", "меңенче", "меңенчегә",
    "меңенчедә", "меңенчедән", "меңенчеләр", "меңенчеләргә", "меңенчеләрдә",
    "меңенчеләрдән", "меңенчеләрне", "меңенчеләрнең", "меңенчене",
    "меңенченең", "меңләгән", "меңләп", "меңнәрчә", "меңәрләгән", "меңәрләп",
    "миллиард", "миллиардлаган", "миллиардларча", "миллион", "миллионлаган",
    "миллионнарча", "миллионынчы", "миллионынчыга", "миллионынчыда",
    "миллионынчыдан", "миллионынчылар", "миллионынчыларга", "миллионынчыларда",
    "миллионынчылардан", "миллионынчыларны", "миллионынчыларның",
    "миллионынчыны", "миллионынчының", "мин", "миндә", "мине", "минем",
    "минемчә", "миннән", "миңа", "монда", "мондагы", "мондые", "мондыен",
    "мондыенгә", "мондыендә", "мондыеннән", "мондыеның", "мондый", "мондыйга",
    "мондыйда", "мондыйдан", "мондыйлар", "мондыйларга", "мондыйларда",
    "мондыйлардан", "мондыйларны", "мондыйларның", "мондыйлары", "мондыйларын",
    "мондыйларынга", "мондыйларында", "мондыйларыннан", "мондыйларының",
    "мондыйны", "мондыйның", "моннан", "монсыз", "монча", "моны", "моныкы",
    "моныкын", "моныкынга", "моныкында", "моныкыннан", "моныкының", "монысы",
    "монысын", "монысынга", "монысында", "монысыннан", "монысының", "моның",
    "моңа", "моңар", "моңарга", "мәгълүматынча", "мәгәр", "мән", "мөмкин",
    "ни", "нибарысы", "никадәре", "нинди", "ниндие", "ниндиен", "ниндиенгә",
    "ниндиендә", "ниндиенең", "ниндиеннән", "ниндиләр", "ниндиләргә",
    "ниндиләрдә", "ниндиләрдән", "ниндиләрен", "ниндиләренн", "ниндиләреннгә",
    "ниндиләренндә", "ниндиләреннең", "ниндиләренннән", "ниндиләрне",
    "ниндиләрнең", "ниндирәк", "нихәтле", "ничаклы", "ничек", "ничәшәр",
    "ничәшәрләп", "нуль", "нче", "нчы", "нәрсә", "нәрсәгә", "нәрсәдә",
    "нәрсәдән", "нәрсәне", "нәрсәнең", "саен", "сез", "сезгә", "сездә",
    "сездән", "сезне", "сезнең", "сезнеңчә", "сигез", "сигезенче",
    "сигезенчегә", "сигезенчедә", "сигезенчедән", "сигезенчеләр",
    "сигезенчеләргә", "сигезенчеләрдә", "сигезенчеләрдән", "сигезенчеләрне",
    "сигезенчеләрнең", "сигезенчене", "сигезенченең", "сиксән", "син", "синдә",
    "сине", "синең", "синеңчә", "синнән", "сиңа", "соң", "сыман", "сүзенчә",
    "сүзләренчә", "та", "таба", "теге", "тегеләй", "тегеләр", "тегеләргә",
    "тегеләрдә", "тегеләрдән", "тегеләре", "тегеләрен", "тегеләренгә",
    "тегеләрендә", "тегеләренең", "тегеләреннән", "тегеләрне", "тегеләрнең",
    "тегенди", "тегендигә", "тегендидә", "тегендидән", "тегендине",
    "тегендинең", "тегендә", "тегендәге", "тегене", "тегенеке", "тегенекен",
    "тегенекенгә", "тегенекендә", "тегенекенең", "тегенекеннән", "тегенең",
    "тегеннән", "тегесе", "тегесен", "тегесенгә", "тегесендә", "тегесенең",
    "тегесеннән", "тегеңә", "тиеш", "тик", "тикле", "тора", "триллиард",
    "триллион", "тугыз", "тугызлап", "тугызлашып", "тугызынчы", "тугызынчыга",
    "тугызынчыда", "тугызынчыдан", "тугызынчылар", "тугызынчыларга",
    "тугызынчыларда", "тугызынчылардан", "тугызынчыларны", "тугызынчыларның",
    "тугызынчыны", "тугызынчының", "туксан", "туксанынчы", "туксанынчыга",
    "туксанынчыда", "туксанынчыдан", "туксанынчылар", "туксанынчыларга",
    "туксанынчыларда", "туксанынчылардан", "туксанынчыларны",
    "туксанынчыларның", "туксанынчыны", "туксанынчының", "турында", "тыш",
    "түгел", "тә", "тәгаенләнгән", "төмән", "уенча", "уйлавынча", "ук", "ул",
    "ун", "уналты", "уналтынчы", "уналтынчыга", "уналтынчыда", "уналтынчыдан",
    "уналтынчылар", "уналтынчыларга", "уналтынчыларда", "уналтынчылардан",
    "уналтынчыларны", "уналтынчыларның", "уналтынчыны", "уналтынчының",
    "унарлаган", "унарлап", "унаула", "унаулап", "унбер", "унберенче",
    "унберенчегә", "унберенчедә", "унберенчедән", "унберенчеләр",
    "унберенчеләргә", "унберенчеләрдә", "унберенчеләрдән", "унберенчеләрне",
    "унберенчеләрнең", "унберенчене", "унберенченең", "унбиш", "унбишенче",
    "унбишенчегә", "унбишенчедә", "унбишенчедән", "унбишенчеләр",
    "унбишенчеләргә", "унбишенчеләрдә", "унбишенчеләрдән", "унбишенчеләрне",
    "унбишенчеләрнең", "унбишенчене", "унбишенченең", "ундүрт", "ундүртенче",
    "ундүртенчегә", "ундүртенчедә", "ундүртенчедән", "ундүртенчеләр",
    "ундүртенчеләргә", "ундүртенчеләрдә", "ундүртенчеләрдән",
    "ундүртенчеләрне", "ундүртенчеләрнең", "ундүртенчене", "ундүртенченең",
    "унике", "уникенче", "уникенчегә", "уникенчедә", "уникенчедән",
    "уникенчеләр", "уникенчеләргә", "уникенчеләрдә", "уникенчеләрдән",
    "уникенчеләрне", "уникенчеләрнең", "уникенчене", "уникенченең", "унлаган",
    "унлап", "уннарча", "унсигез", "унсигезенче", "унсигезенчегә",
    "унсигезенчедә", "унсигезенчедән", "унсигезенчеләр", "унсигезенчеләргә",
    "унсигезенчеләрдә", "унсигезенчеләрдән", "унсигезенчеләрне",
    "унсигезенчеләрнең", "унсигезенчене", "унсигезенченең", "унтугыз",
    "унтугызынчы", "унтугызынчыга", "унтугызынчыда", "унтугызынчыдан",
    "унтугызынчылар", "унтугызынчыларга", "унтугызынчыларда",
    "унтугызынчылардан", "унтугызынчыларны", "унтугызынчыларның",
    "унтугызынчыны", "унтугызынчының", "унынчы", "унынчыга", "унынчыда",
    "унынчыдан", "унынчылар", "унынчыларга", "унынчыларда", "унынчылардан",
    "унынчыларны", "унынчыларның", "унынчыны", "унынчының", "унҗиде",
    "унҗиденче", "унҗиденчегә", "унҗиденчедә", "унҗиденчедән", "унҗиденчеләр",
    "унҗиденчеләргә", "унҗиденчеләрдә", "унҗиденчеләрдән", "унҗиденчеләрне",
    "унҗиденчеләрнең", "унҗиденчене", "унҗиденченең", "унөч", "унөченче",
    "унөченчегә", "унөченчедә", "унөченчедән", "унөченчеләр", "унөченчеләргә",
    "унөченчеләрдә", "унөченчеләрдән", "унөченчеләрне", "унөченчеләрнең",
    "унөченчене", "унөченченең", "утыз", "утызынчы", "утызынчыга",
    "утызынчыда", "утызынчыдан", "утызынчылар", "утызынчыларга",
    "утызынчыларда", "утызынчылардан", "утызынчыларны", "утызынчыларның",
    "утызынчыны", "утызынчының", "фикеренчә", "фәкать", "хакында", "хәбәр",
    "хәлбуки", "хәтле", "хәтта", "чаклы", "чакта", "чөнки", "шикелле", "шул",
    "шулай", "шулар", "шуларга", "шуларда", "шулардан", "шуларны", "шуларның",
    "шулары", "шуларын", "шуларынга", "шуларында", "шуларыннан", "шуларының",
    "шулкадәр", "шултикле", "шултиклем", "шулхәтле", "шулчаклы", "шунда",
    "шундагы", "шундый", "шундыйга", "шундыйда", "шундыйдан", "шундыйны",
    "шундыйның", "шунлыктан", "шуннан", "шунсы", "шунча", "шуны", "шуныкы",
    "шуныкын", "шуныкынга", "шуныкында", "шуныкыннан", "шуныкының", "шунысы",
    "шунысын", "шунысынга", "шунысында", "шунысыннан", "шунысының", "шуның",
    "шушы", "шушында", "шушыннан", "шушыны", "шушының", "шушыңа", "шуңа",
    "шуңар", "шуңарга", "элек", "югыйсә", "юк", "юкса", "я", "ягъни",
    "язуынча", "яисә", "яки", "яктан", "якын", "ярашлы", "яхут", "яшь",
    "яшьлек", "җиде", "җиделәп", "җиденче", "җиденчегә", "җиденчедә",
    "җиденчедән", "җиденчеләр", "җиденчеләргә", "җиденчеләрдә",
    "җиденчеләрдән", "җиденчеләрне", "җиденчеләрнең", "җиденчене",
    "җиденченең", "җидешәр", "җитмеш", "җитмешенче", "җитмешенчегә",
    "җитмешенчедә", "җитмешенчедән", "җитмешенчеләр", "җитмешенчеләргә",
    "җитмешенчеләрдә", "җитмешенчеләрдән", "җитмешенчеләрне",
    "җитмешенчеләрнең", "җитмешенчене", "җитмешенченең", "җыенысы", "үз",
    "үзе", "үзем", "үземдә", "үземне", "үземнең", "үземнән", "үземә", "үзен",
    "үзендә", "үзенең", "үзеннән", "үзенә", "үк", "һичбер", "һичбере",
    "һичберен", "һичберендә", "һичберенең", "һичбереннән", "һичберенә",
    "һичберсе", "һичберсен", "һичберсендә", "һичберсенең", "һичберсеннән",
    "һичберсенә", "һичберәү", "һичберәүгә", "һичберәүдә", "һичберәүдән",
    "һичберәүне", "һичберәүнең", "һичкайсы", "һичкайсыга", "һичкайсыда",
    "һичкайсыдан", "һичкайсыны", "һичкайсының", "һичкем", "һичкемгә",
    "һичкемдә", "һичкемне", "һичкемнең", "һичкемнән", "һични", "һичнигә",
    "һичнидә", "һичнидән", "һичнинди", "һичнине", "һичнинең", "һичнәрсә",
    "һичнәрсәгә", "һичнәрсәдә", "һичнәрсәдән", "һичнәрсәне", "һичнәрсәнең",
    "һәм", "һәммә", "һәммәсе", "һәммәсен", "һәммәсендә", "һәммәсенең",
    "һәммәсеннән", "һәммәсенә", "һәр", "һәрбер", "һәрбере", "һәрберсе",
    "һәркайсы", "һәркайсыга", "һәркайсыда", "һәркайсыдан", "һәркайсыны",
    "һәркайсының", "һәркем", "һәркемгә", "һәркемдә", "һәркемне", "һәркемнең",
    "һәркемнән", "һәрни", "һәрнәрсә", "һәрнәрсәгә", "һәрнәрсәдә",
    "һәрнәрсәдән", "һәрнәрсәне", "һәрнәрсәнең", "һәртөрле", "ә", "әгәр",
    "әйтүенчә", "әйтүләренчә", "әлбәттә", "әле", "әлеге", "әллә", "әмма",
    "әнә", "өстәп", "өч", "өчен", "өченче", "өченчегә", "өченчедә",
    "өченчедән", "өченчеләр", "өченчеләргә", "өченчеләрдә", "өченчеләрдән",
    "өченчеләрне", "өченчеләрнең", "өченчене", "өченченең", "өчләп", "өчәрләп"
},

telugu={
    "అందరూ", "అందుబాటులో", "అడగండి", "అడగడం", "అడ్డంగా", "అనుగుణంగా",
    "అనుమతించు", "అనుమతిస్తుంది", "అయితే", "ఇప్పటికే", "ఉన్నారు", "ఎక్కడైనా",
    "ఎప్పుడు", "ఎవరైనా", "ఎవరో", "ఏ", "ఏదైనా", "ఏమైనప్పటికి", "ఒక", "ఒకరు",
    "కనిపిస్తాయి", "కాదు", "కూడా", "గా", "గురించి", "చుట్టూ", "చేయగలిగింది",
    "తగిన", "తర్వాత", "దాదాపు", "దూరంగా", "నిజంగా", "పై", "ప్రకారం", "ప్రక్కన",
    "మధ్య", "మరియు", "మరొక", "మళ్ళీ", "మాత్రమే", "మెచ్చుకో", "వద్ద", "వెంట",
    "వేరుగా", "వ్యతిరేకంగా", "సంబంధం"
},

thai={
    "กระทั่ง", "กระทำ", "กระนั้น", "กระผม", "กลับ", "กลุ่ม", "กลุ่มก้อน",
    "กลุ่มๆ", "กล่าว", "กล่าวคือ", "กว่า", "กว้าง", "กว้างขวาง", "กว้างๆ",
    "กัน", "กันดีกว่า", "กันดีไหม", "กันนะ", "กันเถอะ", "กันเอง", "กันและกัน",
    "กันไหม", "กับ", "การ", "กำลัง", "กำลังจะ", "กำหนด", "กู", "ก็", "ก็คือ",
    "ก็จะ", "ก็ดี", "ก็ตาม", "ก็ตามที", "ก็ตามแต่", "ก็ต่อเมื่อ", "ก็แค่",
    "ก็แล้วแต่", "ก็ได้", "ก่อน", "ก่อนหน้า", "ก่อนหน้านี้", "ก่อนๆ", "ขณะ",
    "ขณะที่", "ขณะนั้น", "ขณะนี้", "ขณะหนึ่ง", "ขณะเดียวกัน", "ขณะใด",
    "ขณะใดๆ", "ขวาง", "ขวางๆ", "ขอ", "ของ", "ขั้น", "ขาด", "ขึ้น", "ข้า",
    "ข้าง", "ข้างต้น", "ข้างบน", "ข้างล่าง", "ข้างเคียง", "ข้างๆ", "ข้าพเจ้า",
    "ข้าฯ", "คง", "คงจะ", "คงอยู่", "ครบ", "ครบครัน", "ครบถ้วน", "ครัน",
    "ครับ", "ครั้ง", "ครั้งกระนั้น", "ครั้งก่อน", "ครั้งครา", "ครั้งคราว",
    "ครั้งที่", "ครั้งนั้น", "ครั้งนี้", "ครั้งละ", "ครั้งหนึ่ง", "ครั้งหลัง",
    "ครั้งหลังสุด", "ครั้งใด", "ครั้งไหน", "ครั้งๆ", "ครา", "คราที่",
    "ครานั้น", "ครานี้", "คราว", "คราวก่อน", "คราวที่", "คราวนั้น", "คราวนี้",
    "คราวละ", "คราวหนึ่ง", "คราวหน้า", "คราวหลัง", "คราวโน้น", "คราวใด",
    "คราวไหน", "คราวๆ", "คราหนึ่ง", "คราใด", "คราไหน", "คล้าย", "คล้ายกัน",
    "คล้ายกันกับ", "คล้ายกับ", "คล้ายกับว่า", "คล้ายว่า", "ควร", "ความ", "คะ",
    "คำ", "คิด", "คิดว่า", "คือ", "คุณ", "คุณๆ", "ค่อน", "ค่อนข้าง",
    "ค่อนข้างจะ", "ค่อนมาทาง", "ค่อย", "ค่อยไปทาง", "ค่อยๆ", "ค่ะ", "ง่าย",
    "ง่ายๆ", "จง", "จด", "จน", "จนกระทั่ง", "จนกว่า", "จนขณะนี้", "จนตลอด",
    "จนถึง", "จนทั่ว", "จนบัดนี้", "จนเมื่อ", "จนแม้", "จนแม้น", "จรด",
    "จรดกับ", "จริง", "จริงจัง", "จริงๆ", "จริงๆจังๆ", "จวน", "จวนจะ",
    "จวนเจียน", "จวบ", "จวบกับ", "จวบจน", "จะ", "จะได้", "จัง", "จังๆ", "จัด",
    "จัดการ", "จัดงาน", "จัดตั้ง", "จัดทำ", "จัดหา", "จัดแจง", "จัดให้", "จับ",
    "จาก", "จากนั้น", "จากนี้", "จากนี้ไป", "จำ", "จำพวก", "จำเป็น", "จึง",
    "จึงจะ", "จึงเป็น", "จู่ๆ", "จ้ะ", "จ้า", "จ๊ะ", "จ๋า", "ฉะนั้น", "ฉะนี้",
    "ฉัน", "ชาว", "ช่วง", "ช่วงก่อน", "ช่วงต่อไป", "ช่วงถัดไป", "ช่วงที่",
    "ช่วงท้าย", "ช่วงนั้น", "ช่วงนี้", "ช่วงระหว่าง", "ช่วงหน้า", "ช่วงหลัง",
    "ช่วงแรก", "ช่วงๆ", "ช่วย", "ช้า", "ช้านาน", "ช้าๆ", "ซะ", "ซะก่อน",
    "ซะจน", "ซะจนกระทั่ง", "ซะจนถึง", "ซึ่ง", "ซึ่งกัน", "ซึ่งกันและกัน",
    "ซึ่งก็", "ซึ่งก็คือ", "ซึ่งได้แก่", "ซึ่งๆ", "ณ", "ดัง", "ดังกล่าว",
    "ดังกับ", "ดังกับว่า", "ดังเก่า", "ดังเคย", "ดั่ง", "ดั่งกับ",
    "ดั่งกับว่า", "ดั่งเก่า", "ดั่งเคย", "ด้วย", "ด้วยกัน", "ด้วยที่",
    "ด้วยประการฉะนี้", "ด้วยว่า", "ด้วยเช่นกัน", "ด้วยเพราะ", "ด้วยเหตุที่",
    "ด้วยเหตุนั้น", "ด้วยเหตุนี้", "ด้วยเหตุว่า", "ด้วยเหตุเพราะ",
    "ด้วยเหมือนกัน", "ด้าน", "ตน", "ตนฯ", "ตนเอง", "ตรง", "ตรงๆ", "ตลอด",
    "ตลอดกาล", "ตลอดกาลนาน", "ตลอดจน", "ตลอดถึง", "ตลอดทั่ว", "ตลอดทั่วถึง",
    "ตลอดทั่วทั้ง", "ตลอดทั้ง", "ตลอดปี", "ตลอดมา", "ตลอดระยะเวลา", "ตลอดวัน",
    "ตลอดศก", "ตลอดเวลา", "ตลอดไป", "ตั้ง", "ตั้งแต่", "ตาม", "ตามด้วย",
    "ตามที่", "ตามแต่", "ตามๆ", "ต่อ", "ต่อกัน", "ต่าง", "ต่างก็", "ต่างหาก",
    "ต่างๆ", "ต้อง", "ถึง", "ถึงจะ", "ถึงบัดนั้น", "ถึงบัดนี้", "ถึงอย่างไร",
    "ถึงเมื่อ", "ถึงเมื่อใด", "ถึงเมื่อไร", "ถึงแก่", "ถึงแม้", "ถึงแม้จะ",
    "ถึงแม้ว่า", "ถือ", "ถือว่า", "ถูก", "ถูกต้อง", "ถูกๆ", "ถ้า", "ถ้าจะ",
    "ถ้าหาก", "ทรง", "ทว่า", "ทัน", "ทันที", "ทันทีทันใด", "ทันใดนั้น", "ทั่ว",
    "ทั้ง", "ทั้งคน", "ทั้งตัว", "ทั้งที", "ทั้งที่", "ทั้งนั้น",
    "ทั้งนั้นด้วย", "ทั้งนั้นเพราะ", "ทั้งนี้", "ทั้งปวง", "ทั้งมวล",
    "ทั้งสิ้น", "ทั้งหมด", "ทั้งหลาย", "ทั้งเป็น", "ทั้งๆ", "ทาง", "ทำให้",
    "ทำไม", "ทำไร", "ทำๆ", "ที", "ทีละ", "ทีเดียว", "ทีเถอะ", "ทีใด", "ทีไร",
    "ทีๆ", "ที่", "ที่จริง", "ที่ซึ่ง", "ที่นั้น", "ที่นี้", "ที่ละ", "ที่ว่า",
    "ที่สุด", "ที่แท้", "ที่แท้จริง", "ที่แล้ว", "ที่แห่งนั้น", "ที่ใด",
    "ที่ได้", "ที่ไหน", "ที่ๆ", "ทุก", "ทุกคน", "ทุกครั้ง", "ทุกครา",
    "ทุกคราว", "ทุกชิ้น", "ทุกตัว", "ทุกทาง", "ทุกที", "ทุกที่", "ทุกวัน",
    "ทุกวันนี้", "ทุกสิ่ง", "ทุกหน", "ทุกอย่าง", "ทุกอัน", "ทุกเมื่อ",
    "ทุกแห่ง", "ทุกๆ", "ทํา", "ทําให้", "นอก", "นอกจาก", "นอกจากที่",
    "นอกจากนั้น", "นอกจากนี้", "นอกจากว่า", "นอกนั้น", "นอกเหนือ",
    "นอกเหนือจาก", "นะ", "นัก", "นักๆ", "นับ", "นับจากนั้น", "นับจากนี้",
    "นับตั้งแต่", "นับแต่", "นับแต่ที่", "นับแต่นั้น", "นับแต่นี้", "นั่น",
    "นั่นเป็น", "นั่นเอง", "นั่นแหละ", "นั่นไง", "นั้น", "นั้นไว", "นั้นๆ",
    "นาง", "นางสาว", "นาน", "นานๆ", "นาย", "นำ", "นำพา", "นำมา", "นิด",
    "นิดหน่อย", "นิดๆ", "นี่", "นี่นา", "นี่เอง", "นี่แน่ะ", "นี่แหละ",
    "นี่ไง", "นี้", "นี้เอง", "นี้แหล่", "นู่น", "นู้น", "น่ะ", "น่า", "น่าจะ",
    "น้อย", "น้อยกว่า", "น้อยๆ", "นํา", "บน", "บอก", "บอกว่า", "บอกแล้ว",
    "บัดดล", "บัดนั้น", "บัดนี้", "บัดเดี๋ยวนี้", "บาง", "บางกว่า", "บางขณะ",
    "บางครั้ง", "บางครา", "บางคราว", "บางที", "บางที่", "บางแห่ง", "บางๆ",
    "บ่อย", "บ่อยกว่า", "บ่อยครั้ง", "บ่อยๆ", "บ้าง", "ปฏิบัติ", "ประกอบ",
    "ประการ", "ประการฉะนี้", "ประการหนึ่ง", "ประการใด", "ประมาณ", "ประสบ",
    "ปรับ", "ปรากฏ", "ปรากฏว่า", "ปัจจุบัน", "ปิด", "ผล", "ผิด", "ผิดๆ", "ผู้",
    "ผู้ใด", "ผ่าน", "ผ่านๆ", "ฝ่าย", "ฝ่ายใด", "พบ", "พบว่า", "พยายาม",
    "พร้อม", "พร้อมกัน", "พร้อมกับ", "พร้อมด้วย", "พร้อมทั้ง", "พร้อมที่",
    "พร้อมเพียง", "พวก", "พวกกัน", "พวกกู", "พวกคุณ", "พวกฉัน", "พวกที่",
    "พวกท่าน", "พวกนั้น", "พวกนี้", "พวกนู้น", "พวกมัน", "พวกมึง", "พวกเขา",
    "พวกเธอ", "พวกแก", "พวกโน้น", "พอ", "พอกัน", "พอควร", "พอจะ", "พอดี",
    "พอตัว", "พอที", "พอที่", "พอสม", "พอสมควร", "พอเพียง", "พอเหมาะ",
    "พอแล้ว", "พอๆ", "พา", "พึง", "พึ่ง", "พื้นๆ", "พูด", "ภาค", "ภาคฯ", "ภาย",
    "ภายนอก", "ภายภาค", "ภายภาคหน้า", "ภายหน้า", "ภายหลัง", "ภายใต้", "ภายใน",
    "มอง", "มองว่า", "มัก", "มักจะ", "มัน", "มันๆ", "มั้ย", "มั้ยนะ",
    "มั้ยนั่น", "มั้ยล่ะ", "มั้ยเนี่ย", "มั๊ย", "มา", "มาก", "มากกว่า",
    "มากมาย", "มิ", "มิฉะนั้น", "มิใช่", "มิได้", "มี", "มีแต่", "มึง", "มุ่ง",
    "มุ่งหมาย", "มุ่งเน้น", "ยก", "ยกให้", "ยอม", "ยอมรับ", "ยัง", "ยังคง",
    "ยังงั้น", "ยังงี้", "ยังจะ", "ยังแต่", "ยังโง้น", "ยังไง", "ยาก", "ยาว",
    "ยาวนาน", "ยิ่ง", "ยิ่งกว่า", "ยิ่งขึ้น", "ยิ่งขึ้นไป", "ยิ่งจน", "ยิ่งจะ",
    "ยิ่งนัก", "ยิ่งเมื่อ", "ยิ่งแล้ว", "ยิ่งใหญ่", "ยืนนาน", "ยืนยง",
    "ยืนยัน", "ยืนยาว", "ย่อม", "ย่อย", "รวด", "รวดเร็ว", "รวม", "รวมกัน",
    "รวมด้วย", "รวมถึง", "รวมทั้ง", "รวมๆ", "ระยะ", "ระยะๆ", "ระหว่าง", "รับ",
    "รับรอง", "ราย", "รึ", "รึว่า", "รือ", "รือว่า", "ร่วม", "ร่วมกัน",
    "ร่วมด้วย", "ร่วมมือ", "ลง", "ละ", "ล่าสุด", "ล้วน", "ล้วนจน", "ล้วนแต่",
    "วัน", "วันนั้น", "วันนี้", "วันใด", "วันไหน", "ว่า", "สบาย", "สมัย",
    "สมัยก่อน", "สมัยนั้น", "สมัยนี้", "สมัยโน้น", "สั้น", "สั้นๆ", "สามารถ",
    "สำคัญ", "สิ่ง", "สิ่งนั้น", "สิ่งนี้", "สิ่งใด", "สิ่งไหน", "สิ้น",
    "สิ้นกาลนาน", "สืบเนื่อง", "สุด", "สุดๆ", "สูง", "สูงกว่า", "สูงสุด",
    "สูงส่ง", "สูงๆ", "สู่", "ส่ง", "ส่วน", "ส่วนดี", "ส่วนด้อย", "ส่วนที่",
    "ส่วนนั้น", "ส่วนน้อย", "ส่วนมาก", "ส่วนเกิน", "ส่วนใด", "ส่วนใหญ่",
    "สําหรับ", "หน", "หนอ", "หนอย", "หนึ่ง", "หน่อย", "หมด", "หมดกัน",
    "หมดสิ้น", "หรือ", "หรือยัง", "หรือเปล่า", "หรือไง", "หรือไม่", "หรือไร",
    "หลัง", "หลังจาก", "หลาย", "หาก", "หากว่า", "หากแม้", "หากแม้น",
    "หากแม้นว่า", "หาความ", "หารือ", "หาใช่", "อดีต", "อนึ่ง", "อยาก", "อยู่",
    "อย่าง", "อย่างดี", "อย่างที่", "อย่างนั้น", "อย่างนี้", "อย่างน้อย",
    "อย่างมาก", "อย่างยิ่ง", "อย่างละ", "อย่างหนึ่ง", "อย่างเช่น",
    "อย่างเดียว", "อย่างโน้น", "อย่างใด", "อย่างไร", "อย่างไรก็",
    "อย่างไรก็ได้", "อย่างไรเสีย", "อย่างไหน", "อย่างๆ", "ออก", "อะไร", "อัน",
    "อันจะ", "อันที่", "อันที่จริง", "อันที่จะ", "อันละ", "อันเนื่องมาจาก",
    "อันใด", "อันได้แก่", "อันไหน", "อันๆ", "อาจ", "อาจจะ", "อาจเป็น",
    "อาจเป็นด้วย", "อีก", "อื่น", "อื่นๆ", "ฯ", "ฯล", "ฯลฯ", "เกิด", "เกิน",
    "เกินๆ", "เกี่ยวกัน", "เกี่ยวกับ", "เกี่ยวข้อง", "เกี่ยวเนื่อง", "เกี่ยวๆ",
    "เกือบ", "เกือบจะ", "เกือบๆ", "เก็บ", "เขา", "เขียน", "เข้า", "เข้าใจ",
    "เคย", "เคยๆ", "เฉกเช่น", "เฉพาะ", "เฉย", "เฉยๆ", "เชื่อ", "เชื่อถือ",
    "เชื่อมั่น", "เชื่อว่า", "เช่น", "เช่นกัน", "เช่นก่อน", "เช่นดัง",
    "เช่นดังก่อน", "เช่นดังที่", "เช่นดังว่า", "เช่นดังเก่า", "เช่นที่",
    "เช่นที่ว่า", "เช่นที่เคย", "เช่นนั้น", "เช่นนั้นเอง", "เช่นนี้",
    "เช่นเคย", "เช่นเดียวกัน", "เช่นเดียวกับ", "เช่นเมื่อ", "เช่นใด", "เช่นไร",
    "เดียว", "เดียวกัน", "เต็มไปด้วย", "เต็มไปหมด", "เต็มๆ", "เถอะ", "เถิด",
    "เท่า", "เท่ากัน", "เท่ากับ", "เท่าที่", "เท่านั้น", "เท่านี้", "เท่าใด",
    "เท่าไร", "เท่าไหร่", "เธอ", "เนี่ย", "เนี่ยเอง", "เนื่องจาก", "เน้น",
    "เปลี่ยน", "เปลี่ยนแปลง", "เปิด", "เปิดเผย", "เป็น", "เป็นการ", "เป็นดัง",
    "เป็นด้วย", "เป็นต้น", "เป็นต้นมา", "เป็นต้นไป", "เป็นที", "เป็นที่",
    "เป็นที่สุด", "เป็นอัน", "เป็นอันมาก", "เป็นอันว่า", "เป็นอันๆ",
    "เป็นอาทิ", "เป็นเพราะ", "เป็นเพราะว่า", "เป็นเพียง", "เป็นเพียงว่า",
    "เป็นเพื่อ", "เป็นแต่", "เป็นแต่เพียง", "เป็นๆ", "เผื่อ", "เผื่อจะ",
    "เผื่อที่", "เผื่อว่า", "เพราะ", "เพราะฉะนั้น", "เพราะว่า", "เพิ่ง",
    "เพิ่งจะ", "เพิ่ม", "เพิ่มเติม", "เพียง", "เพียงพอ", "เพียงเพราะ",
    "เพียงเพื่อ", "เพียงแค่", "เพียงแต่", "เพียงใด", "เพียงไร", "เพียงไหน",
    "เพื่อ", "เพื่อที่", "เพื่อที่จะ", "เพื่อว่า", "เพื่อให้", "เมื่อ",
    "เมื่อก่อน", "เมื่อครั้ง", "เมื่อครั้งก่อน", "เมื่อคราว", "เมื่อคราวก่อน",
    "เมื่อคราวที่", "เมื่อคืน", "เมื่อนั้น", "เมื่อนี้", "เมื่อวันวาน",
    "เมื่อวาน", "เมื่อเช้า", "เมื่อเย็น", "เมื่อใด", "เมื่อไร", "เมื่อไหร่",
    "เยอะ", "เยอะแยะ", "เยอะๆ", "เรา", "เราๆ", "เริ่ม", "เรียก", "เรียบ",
    "เรื่อย", "เรื่อยๆ", "เร็ว", "เร็วๆ", "เลย", "เล็ก", "เล็กน้อย", "เล็กๆ",
    "เล่าว่า", "เสมือนกับ", "เสมือนว่า", "เสร็จ", "เสร็จกัน", "เสร็จสมบูรณ์",
    "เสร็จสิ้น", "เสร็จแล้ว", "เสีย", "เสียก่อน", "เสียจน", "เสียจนกระทั่ง",
    "เสียจนถึง", "เสียด้วย", "เสียนั่น", "เสียนั่นเอง", "เสียนี่",
    "เสียนี่กระไร", "เสียยิ่ง", "เสียยิ่งนัก", "เสียแล้ว", "เหตุ", "เหตุนั้น",
    "เหตุนี้", "เหตุผล", "เหตุไร", "เหลือ", "เหลือเกิน", "เหล่า", "เหล่านั้น",
    "เหล่านี้", "เห็น", "เห็นควร", "เห็นจะ", "เห็นว่า", "เห็นแก่", "เอง",
    "เอา", "เอ็ง", "แก", "แก่", "แก้ไข", "แค่", "แค่จะ", "แค่นั้น", "แค่นี้",
    "แค่ว่า", "แค่เพียง", "แค่ไหน", "แต่", "แต่ก็", "แต่ก่อน", "แต่จะ",
    "แต่ต้อง", "แต่ถ้า", "แต่ทว่า", "แต่ที่", "แต่นั้น", "แต่ละ", "แต่ว่า",
    "แต่อย่างใด", "แต่เดิม", "แต่เพียง", "แต่เมื่อ", "แต่ไร", "แต่ไหน", "แท้",
    "แท้จริง", "แบบ", "แม้", "แม้กระทั่ง", "แม้นว่า", "แม้ว่า", "แม้แต่",
    "แยะ", "แยะๆ", "แรก", "และ", "แล้ว", "แล้วกัน", "แล้วเสร็จ", "แล้วแต่",
    "แสดง", "แสดงว่า", "แหละ", "แห่ง", "แห่งนั้น", "แห่งนี้", "แห่งโน้น",
    "แห่งใด", "แห่งไหน", "โดย", "โต", "โตๆ", "ใกล้", "ใกล้ๆ", "ใคร", "ใครๆ",
    "ใคร่", "ใคร่จะ", "ใช่", "ใช่ไหม", "ใช้", "ใดๆ", "ใต้", "ใน", "ในช่วง",
    "ในที่", "ในระหว่าง", "ในเมื่อ", "ใหญ่", "ใหญ่โต", "ใหญ่ๆ", "ใหม่",
    "ใหม่ๆ", "ให้", "ให้ดี", "ให้มา", "ให้แก่", "ให้แด่", "ให้ไป", "ไกล",
    "ไกลๆ", "ไง", "ไฉน", "ได้", "ได้ที่", "ได้มา", "ได้รับ", "ได้แก่",
    "ได้แต่", "ไป", "ไป่", "ไม่", "ไม่ค่อย", "ไม่ค่อยจะ", "ไม่ค่อยเป็น",
    "ไม่ว่า", "ไม่เป็นไร", "ไม่ใช่", "ไร", "ไว้", "ไหน", "ไหนๆ", "ๆ"
},

turkish={
    "acaba", "acep", "adamakıllı", "adeta", "ait", "ama", "amma", "anca",
    "ancak", "arada", "artık", "aslında", "aynen", "ayrıca", "az", "açıkça",
    "açıkçası", "bana", "bari", "bazen", "bazı", "bazısı", "bazısına",
    "bazısında", "bazısından", "bazısını", "bazısının", "başka", "başkası",
    "başkasına", "başkasında", "başkasından", "başkasını", "başkasının",
    "belki", "ben", "bende", "benden", "beni", "benim", "beri", "beriki",
    "berikinin", "berikiyi", "berisi", "bilcümle", "bile", "binaen",
    "binaenaleyh", "biraz", "birazdan", "birbiri", "birbirinde", "birbirinden",
    "birbirine", "birbirini", "birbirinin", "birden", "birdenbire", "biri",
    "birice", "birileri", "birilerinde", "birilerinden", "birilerine",
    "birilerini", "birilerinin", "birinde", "birinden", "birine", "birini",
    "birinin", "birisi", "birisinde", "birisinden", "birisine", "birisini",
    "birisinin", "birkaç", "birkaçı", "birkaçına", "birkaçında", "birkaçından",
    "birkaçını", "birkaçının", "birkez", "birlikte", "birçok", "birçoğu",
    "birçoğuna", "birçoğunda", "birçoğundan", "birçoğunu", "birçoğunun",
    "birşey", "birşeyi", "bitevi", "biteviye", "bittabi", "biz", "bizatihi",
    "bizce", "bizcileyin", "bizden", "bize", "bizi", "bizim", "bizimki",
    "bizzat", "boşuna", "bu", "buna", "bunda", "bundan", "bunlar", "bunları",
    "bunların", "bunu", "bunun", "buracıkta", "burada", "buradan", "burası",
    "burasına", "burasında", "burasından", "burasını", "burasının", "böyle",
    "böylece", "böylecene", "böylelikle", "böylemesine", "böylesine",
    "büsbütün", "bütün", "cuk", "cümlemizden", "cümlemize", "cümlemizi",
    "cümlesi", "cümlesinden", "cümlesine", "cümlesini", "cümlesinin", "da",
    "daha", "dahası", "dahi", "dahil", "dahilen", "daima", "dair", "dayanarak",
    "de", "defa", "dek", "demin", "demincek", "deminden", "denli", "derakap",
    "derhal", "derken", "değil", "değin", "diye", "diğer", "diğeri",
    "diğerinden", "diğerine", "diğerini", "dolayı", "dolayısıyla", "doğru",
    "edecek", "eden", "ederek", "edilecek", "ediliyor", "edilmesi", "ediyor",
    "elbet", "elbette", "emme", "en", "enikonu", "epey", "epeyce", "epeyi",
    "esasen", "esnasında", "etmesi", "etraflı", "etraflıca", "etti", "ettiği",
    "ettiğini", "evleviyetle", "evvel", "evvela", "evvelce", "evvelden",
    "evvelemirde", "evveli", "eğer", "fakat", "filanca", "filancanın", "gah",
    "gayet", "gayetle", "gayri", "gayrı", "gelgelelim", "gene", "gerek",
    "gerçi", "geçende", "geçenlerde", "gibi", "gibilerden", "gibisinden",
    "gine", "göre", "gırla", "hakeza", "halbuki", "halen", "halihazırda",
    "haliyle", "handiyse", "hangi", "hangisi", "hangisinde", "hangisinden",
    "hangisine", "hani", "hariç", "hasebiyle", "hasılı", "hatta", "hele",
    "hem", "henüz", "hep", "hepsi", "hepsinde", "hepsinden", "hepsini",
    "hepsinin", "her", "herhangi", "herkes", "herkesi", "herkesin",
    "herkesten", "hiç", "hiçbir", "hiçbiri", "hiçbirinde", "hiçbirinden",
    "hiçbirine", "hiçbirini", "hiçbirinin", "hoş", "hulasaten", "iken", "ila",
    "ile", "ilen", "ilgili", "ilk", "illa", "illaki", "imdi", "indinde",
    "inen", "insermi", "ise", "ister", "itibaren", "itibariyle", "itibarıyla",
    "iyi", "iyice", "iyicene", "için", "iş", "işte", "kadar", "kaffesi", "kah",
    "kala", "kanımca", "karşın", "kaynak", "kaçı", "kaçına", "kaçında",
    "kaçından", "kaçını", "kaçının", "kelli", "kendi", "kendilerinde",
    "kendilerinden", "kendilerine", "kendilerini", "kendilerinin", "kendini",
    "kendisi", "kendisinde", "kendisinden", "kendisine", "kendisini",
    "kendisinin", "kere", "kez", "keza", "kezalik", "keşke", "ki", "kim",
    "kimden", "kime", "kimi", "kiminin", "kimisi", "kimisinde", "kimisinden",
    "kimisine", "kimisinin", "kimse", "kimsecik", "kimsecikler", "külliyen",
    "kısaca", "kısacası", "lakin", "leh", "lütfen", "maada", "madem",
    "mademki", "mamafih", "mebni", "međer", "meğer", "meğerki", "meğerse",
    "mi", "mu", "mü", "mı", "nasıl", "nasılsa", "nazaran", "naşi", "ne",
    "neden", "nedeniyle", "nedenle", "nedenler", "nedenlerden", "nedense",
    "nerde", "nerden", "nerdeyse", "nere", "nerede", "nereden", "neredeyse",
    "neresi", "nereye", "netekim", "neye", "neyi", "neyse", "nice", "nihayet",
    "nihayetinde", "nitekim", "niye", "niçin", "o", "olan", "olarak", "oldu",
    "olduklarını", "oldukça", "olduğu", "olduğunu", "olmak", "olması", "olsa",
    "olsun", "olup", "olur", "olursa", "oluyor", "ona", "onca", "onculayın",
    "onda", "ondan", "onlar", "onlara", "onlardan", "onları", "onların", "onu",
    "onun", "ora", "oracık", "oracıkta", "orada", "oradan", "oranca", "oranla",
    "oraya", "oysa", "oysaki", "pek", "pekala", "peki", "pekçe", "peyderpey",
    "rağmen", "sadece", "sahi", "sahiden", "sana", "sanki", "sen", "senden",
    "seni", "senin", "siz", "sizden", "sizi", "sizin", "sonra", "sonradan",
    "sonraları", "sonunda", "tabii", "tam", "tamam", "tamamen", "tamamıyla",
    "tarafından", "tek", "tüm", "var", "vardı", "vasıtasıyla", "ve", "velev",
    "velhasıl", "velhasılıkelam", "veya", "veyahut", "ya", "yahut", "yakinen",
    "yakında", "yakından", "yakınlarda", "yalnız", "yalnızca", "yani",
    "yapacak", "yapmak", "yaptı", "yaptıkları", "yaptığı", "yaptığını",
    "yapılan", "yapılması", "yapıyor", "yeniden", "yenilerde", "yerine",
    "yine", "yok", "yoksa", "yoluyla", "yüzünden", "zarfında", "zaten", "zati",
    "zira", "çabuk", "çabukça", "çeşitli", "çok", "çokları", "çoklarınca",
    "çokluk", "çoklukla", "çokça", "çoğu", "çoğun", "çoğunca", "çoğunda",
    "çoğundan", "çoğunlukla", "çoğunu", "çoğunun", "çünkü", "öbür", "öbürkü",
    "öbürü", "öbüründe", "öbüründen", "öbürüne", "öbürünü", "önce", "önceden",
    "önceleri", "öncelikle", "öteki", "ötekisi", "öyle", "öylece", "öylelikle",
    "öylemesine", "öz", "üzere", "şayet", "şey", "şeyden", "şeyi", "şeyler",
    "şimdi", "şu", "şuna", "şuncacık", "şunda", "şundan", "şunlar", "şunları",
    "şunların", "şunu", "şunun", "şura", "şuracık", "şuracıkta", "şurası",
    "şöyle"
},

ukrainian={
    "а", "або", "адже", "але", "алло", "багато", "без", "безперервно", "би",
    "близько", "був", "буває", "буде", "будемо", "будете", "будеш", "буду",
    "будуть", "будь", "була", "були", "було", "бути", "бывь", "більш",
    "більше", "біля", "в", "важлива", "важливе", "важливий", "важливі", "вам",
    "вами", "вас", "ваш", "ваша", "ваше", "ваші", "вгору", "вгорі", "вдалині",
    "вже", "ви", "вниз", "внизу", "вона", "вони", "воно", "восьмий", "всього",
    "втім", "від", "відсотків", "він", "вісім", "вісімнадцятий",
    "вісімнадцять", "г", "геть", "говорив", "говорить", "давно", "далеко",
    "далі", "дарма", "два", "двадцятий", "двадцять", "дванадцятий",
    "дванадцять", "двох", "дві", "де", "дев'ятий", "дев'ятнадцятий",
    "дев'ятнадцять", "дев'ять", "декілька", "день", "десятий", "десять", "для",
    "дня", "до", "добре", "довго", "доки", "досить", "другий", "дуже",
    "дійсно", "же", "життя", "з", "за", "завжди", "зазвичай", "зайнята",
    "зайнятий", "зайнято", "зайняті", "занадто", "зараз", "зате", "звичайно",
    "звідси", "звідусіль", "здається", "значить", "знову", "зовсім", "його",
    "йому", "ким", "кого", "кожен", "кожна", "кожне", "кожні", "коли", "кому",
    "краще", "крейдуючи", "кругом", "куди", "ласка", "лише", "люди", "людина",
    "майже", "мало", "мати", "мене", "менш", "менше", "мені", "ми", "мимо",
    "мною", "могти", "мож", "може", "можна", "можно", "можуть", "можхо", "мор",
    "моя", "моє", "мої", "міг", "між", "мій", "мільйонів", "на", "навкруги",
    "навіть", "навіщо", "нагорі", "над", "назад", "найбільш", "нам", "нами",
    "нарешті", "нас", "наш", "наша", "наше", "наші", "не", "небагато",
    "недалеко", "немає", "нерідко", "нещодавно", "нею", "нибудь", "нижче",
    "низько", "ним", "ними", "них", "ну", "нх", "нього", "ні", "ніби",
    "ніколи", "нікуди", "нічого", "о", "обоє", "один", "одинадцятий",
    "одинадцять", "одного", "одній", "однієї", "означає", "окрім", "он",
    "особливо", "ось", "п'ятий", "п'ятнадцятий", "п'ятнадцять", "п'ять",
    "перед", "перший", "по", "повинно", "подів", "поки", "пора", "поруч",
    "посеред", "потрібно", "потім", "почала", "прекрасне", "прекрасно", "при",
    "про", "просто", "проте", "проти", "під", "пізніше", "пір", "раз", "рано",
    "раніше", "раптом", "роки", "року", "років", "рік", "сам", "сама", "саме",
    "самим", "самими", "самих", "само", "самого", "самому", "саму", "самі",
    "самій", "свого", "свою", "своє", "свої", "своїй", "своїх", "світу",
    "сеаой", "себе", "сказав", "сказала", "сказати", "скрізь", "скільки",
    "собою", "собі", "спасибі", "спочатку", "справ", "став", "суть",
    "сьогодні", "сьомий", "сім", "сімнадцятий", "сімнадцять", "т", "та", "так",
    "така", "таке", "такий", "також", "такі", "там", "твоя", "твоє", "твій",
    "те", "тебе", "теж", "тепер", "ти", "тим", "тими", "тисяч", "тих", "тобою",
    "тобі", "того", "тоді", "той", "том", "тому", "треба", "третій", "три",
    "тринадцятий", "тринадцять", "трохи", "ту", "туди", "тут", "ті", "тільки",
    "тією", "у", "увесь", "уміти", "усе", "усього", "усьому", "усю", "усюди",
    "уся", "усі", "усім", "усіма", "усіх", "усію", "хотіти", "хоч", "хоча",
    "хочеш", "хто", "хіба", "це", "цей", "цим", "цими", "цих", "цього",
    "цьому", "цю", "ця", "ці", "цій", "час", "часто", "частіше", "часу",
    "через", "четвертий", "чи", "чим", "численна", "численне", "численний",
    "численні", "чого", "чому", "чотири", "чотирнадцятий", "чотирнадцять",
    "шостий", "шістнадцятий", "шістнадцять", "шість", "ще", "що", "щоб", "я",
    "як", "яка", "який", "яких", "якого", "якщо", "які", "якій", "ім'я",
    "іноді", "інша", "інше", "інший", "інших", "інші", "їй", "їх", "її"
},

urdu={
    "آئی", "آئے", "آج", "آخر", "آدهی", "آش", "آًب", "آٹھ", "آیب", "اخبزت",
    "اختتبم", "ادھر", "ارد", "اردگرد", "ارکبى", "اش", "اضتعوبل", "اضتعوبلات",
    "اضطرذ", "اضکب", "اضکی", "اضکے", "اطراف", "اغیب", "افراد", "الگ", "اور",
    "اوًچب", "اوًچبئی", "اوًچی", "اوًچے", "اى", "اًذر", "اٹھبًب", "اپٌب",
    "اپٌے", "اچھب", "اچھی", "اچھے", "اکثر", "اکٹھب", "اکٹھی", "اکٹھے", "اکیلا",
    "اکیلی", "اکیلے", "اگرچہ", "اہن", "ایطے", "ایک", "تب", "تبزٍ", "تت", "تر",
    "ترتیت", "تریي", "تعذاد", "تن", "تو", "توبم", "توہیں", "تٌہب", "تک", "تھب",
    "تھوڑا", "تھوڑی", "تھوڑے", "تھی", "تھے", "تیي", "ثب", "ثبری", "ثبرے",
    "ثبعث", "ثبلا", "ثبلترتیت", "ثبہر", "ثدبئے", "ثرآں", "ثراں", "ثرش", "ثعذ",
    "ثغیر", "ثلٌذ", "ثلکہ", "ثٌب", "ثٌبرہب", "ثٌبرہی", "ثٌبرہے", "ثٌبًب",
    "ثٌذ", "ثٌذکرو", "ثٌذکرًب", "ثٌذی", "ثڑا", "ثڑی", "ثڑے", "ثھر", "ثھرا",
    "ثھرپور", "ثھی", "ثہت", "ثہتر", "ثہتری", "ثہتریي", "ثیچ", "خب", "خبر",
    "خبهوظ", "خبًب", "خبًتب", "خبًتی", "خبًتے", "خبًٌب", "ختن", "خجکہ", "خص",
    "خططرذ", "خلذی", "خو", "خواى", "خوًہی", "خوکہ", "خٌبة", "خگہ", "خگہوں",
    "خگہیں", "خیطب", "خیطبکہ", "در", "درخبت", "درخہ", "درخے", "درزقیقت",
    "درضت", "دش", "دفعہ", "دلچطپ", "دلچطپی", "دلچطپیبں", "دو", "دور", "دورى",
    "دوضرا", "دوضروں", "دوضری", "دوضرے", "دوًوں", "دکھبئیں", "دکھبتب",
    "دکھبتی", "دکھبتے", "دکھبو", "دکھبًب", "دی", "دیب", "دیتب", "دیتی", "دیتے",
    "دیر", "دیٌب", "دیکھو", "دیکھٌب", "دیکھی", "دیکھیں", "دے", "راضتوں",
    "راضتہ", "راضتے", "رریعہ", "رریعے", "رکي", "رکھ", "رکھب", "رکھتی", "رکھتے",
    "رکھی", "رکھے", "رہب", "رہی", "رہے", "زبصل", "زبضر", "زبل", "زبلات",
    "زبلیہ", "زصوں", "زصہ", "زصے", "زقبئق", "زقیتیں", "زقیقت", "زکن", "زکویہ",
    "زیبدٍ", "صبف", "صسیر", "صفر", "صورت", "صورتسبل", "صورتوں", "صورتیں",
    "ضبت", "ضبدٍ", "ضبرا", "ضبرے", "ضبل", "ضبلوں", "ضت", "ضرور", "ضرورت",
    "ضروری", "ضلطلہ", "ضوچ", "ضوچب", "ضوچتب", "ضوچتی", "ضوچتے", "ضوچو",
    "ضوچٌب", "ضوچی", "ضوچیں", "ضکب", "ضکتب", "ضکتی", "ضکتے", "ضکٌب", "ضکی",
    "ضکے", "ضیذھب", "ضیذھی", "ضیذھے", "ضیکٌڈ", "ضے", "طب", "طرف", "طریق",
    "طریقوں", "طریقہ", "طریقے", "طور", "طورپر", "ظبہر", "عذد", "عظین",
    "علاقوں", "علاقہ", "علاقے", "علاوٍ", "عووهی", "غبیذ", "غخص", "غذ", "غروع",
    "غروعبت", "غے", "فرد", "فی", "قجل", "قجیلہ", "قطن", "لئے", "لازهی", "لو",
    "لوجب", "لوجی", "لوجے", "لوسبت", "لوسہ", "لوگ", "لوگوں", "لڑکپي", "لگتب",
    "لگتی", "لگتے", "لگی", "لگیں", "لگے", "لی", "لیب", "لیٌب", "لیں", "لے",
    "هتعلق", "هختلف", "هسترم", "هسترهہ", "هسطوش", "هسیذ", "هطئلہ", "هطئلے",
    "هطبئل", "هطتعول", "هطلق", "هعلوم", "هػتول", "هلا", "هوکي", "هوکٌبت",
    "هوکٌہ", "هٌبضت", "هڑا", "هڑًب", "هڑے", "هکول", "هگر", "هہرثبى", "هیرا",
    "هیری", "هیرے", "هیں", "وار", "والے", "ًئی", "ًئے", "ًبپطٌذ", "ًبگسیر",
    "ًطجت", "ًقطہ", "ًوخواى", "ًکبلٌب", "ًکتہ", "ًہیں", "ًیب", "ٹھیک", "پبئے",
    "پبش", "پبًب", "پبًچ", "پر", "پراًب", "پل", "پورا", "پوچھب", "پوچھتب",
    "پوچھتی", "پوچھتے", "پوچھو", "پوچھوں", "پوچھٌب", "پوچھیں", "پچھلا", "پھر",
    "پہلا", "پہلی", "پہلے", "پہلےضی", "پیع", "چبر", "چبہب", "چبہٌب", "چبہے",
    "چلا", "چلو", "چلیں", "چلے", "چکب", "چکی", "چکیں", "چکے", "چھوٹب",
    "چھوٹوں", "چھوٹی", "چھوٹے", "چھہ", "چیسیں", "ڈھوًڈا", "ڈھوًڈلیب", "ڈھوًڈو",
    "ڈھوًڈًب", "ڈھوًڈی", "ڈھوًڈیں", "کئی", "کئے", "کب", "کبفی", "کبم", "کت",
    "کجھی", "کر", "کرا", "کرتب", "کرتبہوں", "کرتی", "کرتے", "کررہی", "کرو",
    "کرًب", "کریں", "کرے", "کل", "کن", "کوئی", "کوتر", "کورا", "کوروں", "کورٍ",
    "کورے", "کوطي", "کوى", "کوًطب", "کوًطی", "کوًطے", "کھولا", "کھولو",
    "کھولٌب", "کھولیں", "کھولے", "کہ", "کہب", "کہتب", "کہتی", "کہتے", "کہو",
    "کہوں", "کہٌب", "کہی", "کہیں", "کہے", "کی", "کیب", "کیطے", "کیوًکہ",
    "کیوں", "کیے", "کے", "گئی", "گئے", "گب", "گرد", "گروٍ", "گروپ", "گروہوں",
    "گٌتی", "گی", "گیب", "گے", "ہر", "ہن", "ہو", "ہوئی", "ہوئے", "ہوا",
    "ہوبرا", "ہوبری", "ہوبرے", "ہوتی", "ہوتے", "ہورہب", "ہورہی", "ہورہے",
    "ہوضکتب", "ہوضکتی", "ہوضکتے", "ہوًب", "ہوًی", "ہوًے", "ہوچکب", "ہوچکی",
    "ہوچکے", "ہوگئی", "ہوگیب", "ہوں", "ہی", "ہیں", "ہے", "یقیٌی", "یہ", "یہبں"
},

vietnamese={
    "", "a_ha", "a_lô", "ai", "ai_ai", "ai_nấy", "ai_đó", "alô", "amen", "anh",
    "anh_ấy", "ba", "ba_bau", "ba_bản", "ba_cùng", "ba_họ", "ba_ngày",
    "ba_ngôi", "ba_tăng", "bao_giờ", "bao_lâu", "bao_nhiêu", "bao_nả",
    "bay_biến", "biết", "biết_bao", "biết_bao_nhiêu", "biết_chắc",
    "biết_chừng_nào", "biết_mình", "biết_mấy", "biết_thế", "biết_trước",
    "biết_việc", "biết_đâu", "biết_đâu_chừng", "biết_đâu_đấy", "biết_được",
    "buổi", "buổi_làm", "buổi_mới", "buổi_ngày", "buổi_sớm", "bà", "bà_ấy",
    "bài", "bài_bác", "bài_bỏ", "bài_cái", "bác", "bán", "bán_cấp", "bán_dạ",
    "bán_thế", "bây_bẩy", "bây_chừ", "bây_giờ", "bây_nhiêu", "bèn", "béng",
    "bên", "bên_bị", "bên_có", "bên_cạnh", "bông", "bước", "bước_khỏi",
    "bước_tới", "bước_đi", "bạn", "bản", "bản_bộ", "bản_riêng", "bản_thân",
    "bản_ý", "bất_chợt", "bất_cứ", "bất_giác", "bất_kì", "bất_kể", "bất_kỳ",
    "bất_luận", "bất_ngờ", "bất_nhược", "bất_quá", "bất_quá_chỉ",
    "bất_thình_lình", "bất_tử", "bất_đồ", "bấy", "bấy_chầy", "bấy_chừ",
    "bấy_giờ", "bấy_lâu", "bấy_lâu_nay", "bấy_nay", "bấy_nhiêu",
    "bập_bà_bập_bõm", "bập_bõm", "bắt_đầu", "bắt_đầu_từ", "bằng", "bằng_cứ",
    "bằng_không", "bằng_người", "bằng_nhau", "bằng_như", "bằng_nào",
    "bằng_nấy", "bằng_vào", "bằng_được", "bằng_ấy", "bển", "bệt", "bị",
    "bị_chú", "bị_vì", "bỏ", "bỏ_bà", "bỏ_cha", "bỏ_cuộc", "bỏ_không",
    "bỏ_lại", "bỏ_mình", "bỏ_mất", "bỏ_mẹ", "bỏ_nhỏ", "bỏ_quá", "bỏ_ra",
    "bỏ_riêng", "bỏ_việc", "bỏ_xa", "bỗng", "bỗng_chốc", "bỗng_dưng",
    "bỗng_không", "bỗng_nhiên", "bỗng_nhưng", "bỗng_thấy", "bỗng_đâu", "bộ",
    "bộ_thuộc", "bộ_điều", "bội_phần", "bớ", "bởi", "bởi_ai", "bởi_chưng",
    "bởi_nhưng", "bởi_sao", "bởi_thế", "bởi_thế_cho_nên", "bởi_tại", "bởi_vì",
    "bởi_vậy", "bởi_đâu", "bức", "cao", "cao_lâu", "cao_ráo", "cao_răng",
    "cao_sang", "cao_số", "cao_thấp", "cao_thế", "cao_xa", "cha", "cha_chả",
    "chao_ôi", "chia_sẻ", "chiếc", "cho", "cho_biết", "cho_chắc", "cho_hay",
    "cho_nhau", "cho_nên", "cho_rằng", "cho_rồi", "cho_thấy", "cho_tin",
    "cho_tới", "cho_tới_khi", "cho_về", "cho_ăn", "cho_đang", "cho_được",
    "cho_đến", "cho_đến_khi", "cho_đến_nỗi", "choa", "chu_cha", "chui_cha",
    "chung", "chung_cho", "chung_chung", "chung_cuộc", "chung_cục",
    "chung_nhau", "chung_qui", "chung_quy", "chung_quy_lại", "chung_ái",
    "chuyển", "chuyển_tự", "chuyển_đạt", "chuyện", "chuẩn_bị", "chành_chạnh",
    "chí_chết", "chính", "chính_bản", "chính_giữa", "chính_là", "chính_thị",
    "chính_điểm", "chùn_chùn", "chùn_chũn", "chú", "chú_dẫn", "chú_khách",
    "chú_mày", "chú_mình", "chúng", "chúng_mình", "chúng_ta", "chúng_tôi",
    "chúng_ông", "chăn_chắn", "chăng", "chăng_chắc", "chăng_nữa", "chơi",
    "chơi_họ", "chưa", "chưa_bao_giờ", "chưa_chắc", "chưa_có", "chưa_cần",
    "chưa_dùng", "chưa_dễ", "chưa_kể", "chưa_tính", "chưa_từng", "chầm_chập",
    "chậc", "chắc", "chắc_chắn", "chắc_dạ", "chắc_hẳn", "chắc_lòng",
    "chắc_người", "chắc_vào", "chắc_ăn", "chẳng_lẽ", "chẳng_những",
    "chẳng_nữa", "chẳng_phải", "chết_nỗi", "chết_thật", "chết_tiệt", "chỉ",
    "chỉ_chính", "chỉ_có", "chỉ_là", "chỉ_tên", "chỉn", "chị", "chị_bộ",
    "chị_ấy", "chịu", "chịu_chưa", "chịu_lời", "chịu_tốt", "chịu_ăn", "chọn",
    "chọn_bên", "chọn_ra", "chốc_chốc", "chớ", "chớ_chi", "chớ_gì",
    "chớ_không", "chớ_kể", "chớ_như", "chợt", "chợt_nghe", "chợt_nhìn", "chủn",
    "chứ", "chứ_ai", "chứ_còn", "chứ_gì", "chứ_không", "chứ_không_phải",
    "chứ_lại", "chứ_lị", "chứ_như", "chứ_sao", "coi_bộ", "coi_mòi", "con",
    "con_con", "con_dạ", "con_nhà", "con_tính", "cu_cậu", "cuối", "cuối_cùng",
    "cuối_điểm", "cuốn", "cuộc", "càng", "càng_càng", "càng_hay", "cá_nhân",
    "các", "các_cậu", "cách", "cách_bức", "cách_không", "cách_nhau",
    "cách_đều", "cái", "cái_gì", "cái_họ", "cái_đã", "cái_đó", "cái_ấy",
    "câu_hỏi", "cây", "cây_nước", "còn", "còn_như", "còn_nữa", "còn_thời_gian",
    "còn_về", "có", "có_ai", "có_chuyện", "có_chăng", "có_chăng_là", "có_chứ",
    "có_cơ", "có_dễ", "có_họ", "có_khi", "có_ngày", "có_người", "có_nhiều",
    "có_nhà", "có_phải", "có_số", "có_tháng", "có_thế", "có_thể", "có_vẻ",
    "có_ý", "có_ăn", "có_điều", "có_điều_kiện", "có_đáng", "có_đâu", "có_được",
    "cóc_khô", "cô", "cô_mình", "cô_quả", "cô_tăng", "cô_ấy", "công_nhiên",
    "cùng", "cùng_chung", "cùng_cực", "cùng_nhau", "cùng_tuổi", "cùng_tột",
    "cùng_với", "cùng_ăn", "căn", "căn_cái", "căn_cắt", "căn_tính", "cũng",
    "cũng_như", "cũng_nên", "cũng_thế", "cũng_vậy", "cũng_vậy_thôi",
    "cũng_được", "cơ", "cơ_chỉ", "cơ_chừng", "cơ_cùng", "cơ_dẫn", "cơ_hồ",
    "cơ_hội", "cơ_mà", "cơn", "cả", "cả_nghe", "cả_nghĩ", "cả_ngày",
    "cả_người", "cả_nhà", "cả_năm", "cả_thảy", "cả_thể", "cả_tin", "cả_ăn",
    "cả_đến", "cảm_thấy", "cảm_ơn", "cấp", "cấp_số", "cấp_trực_tiếp", "cần",
    "cần_cấp", "cần_gì", "cần_số", "cật_lực", "cật_sức", "cậu", "cổ_lai",
    "cụ_thể", "cụ_thể_là", "cụ_thể_như", "của", "của_ngọt", "của_tin", "cứ",
    "cứ_như", "cứ_việc", "cứ_điểm", "cực_lực", "do", "do_vì", "do_vậy",
    "do_đó", "duy", "duy_chỉ", "duy_có", "dài", "dài_lời", "dài_ra", "dành",
    "dành_dành", "dào", "dì", "dù", "dù_cho", "dù_dì", "dù_gì", "dù_rằng",
    "dù_sao", "dùng", "dùng_cho", "dùng_hết", "dùng_làm", "dùng_đến", "dưới",
    "dưới_nước", "dạ", "dạ_bán", "dạ_con", "dạ_dài", "dạ_dạ", "dạ_khách",
    "dần_dà", "dần_dần", "dầu_sao", "dẫn", "dẫu", "dẫu_mà", "dẫu_rằng",
    "dẫu_sao", "dễ", "dễ_dùng", "dễ_gì", "dễ_khiến", "dễ_nghe", "dễ_ngươi",
    "dễ_như_chơi", "dễ_sợ", "dễ_sử_dụng", "dễ_thường", "dễ_thấy", "dễ_ăn",
    "dễ_đâu", "dở_chừng", "dữ", "dữ_cách", "em", "em_em", "giá_trị",
    "giá_trị_thực_tế", "giảm", "giảm_chính", "giảm_thấp", "giảm_thế", "giống",
    "giống_người", "giống_nhau", "giống_như", "giờ", "giờ_lâu", "giờ_này",
    "giờ_đi", "giờ_đây", "giờ_đến", "giữ", "giữ_lấy", "giữ_ý", "giữa",
    "giữa_lúc", "gây", "gây_cho", "gây_giống", "gây_ra", "gây_thêm", "gì",
    "gì_gì", "gì_đó", "gần", "gần_bên", "gần_hết", "gần_ngày", "gần_như",
    "gần_xa", "gần_đây", "gần_đến", "gặp", "gặp_khó_khăn", "gặp_phải", "gồm",
    "hay", "hay_biết", "hay_hay", "hay_không", "hay_là", "hay_làm", "hay_nhỉ",
    "hay_nói", "hay_sao", "hay_tin", "hay_đâu", "hiểu", "hiện_nay", "hiện_tại",
    "hoàn_toàn", "hoặc", "hoặc_là", "hãy", "hãy_còn", "hơn", "hơn_cả",
    "hơn_hết", "hơn_là", "hơn_nữa", "hơn_trước", "hầu_hết", "hết",
    "hết_chuyện", "hết_cả", "hết_của", "hết_nói", "hết_ráo", "hết_rồi",
    "hết_ý", "họ", "họ_gần", "họ_xa", "hỏi", "hỏi_lại", "hỏi_xem", "hỏi_xin",
    "hỗ_trợ", "khi", "khi_khác", "khi_không", "khi_nào", "khi_nên",
    "khi_trước", "khiến", "khoảng", "khoảng_cách", "khoảng_không", "khá",
    "khá_tốt", "khác", "khác_gì", "khác_khác", "khác_nhau", "khác_nào",
    "khác_thường", "khác_xa", "khách", "khó", "khó_biết", "khó_chơi",
    "khó_khăn", "khó_làm", "khó_mở", "khó_nghe", "khó_nghĩ", "khó_nói",
    "khó_thấy", "khó_tránh", "không", "không_ai", "không_bao_giờ",
    "không_bao_lâu", "không_biết", "không_bán", "không_chỉ", "không_còn",
    "không_có", "không_có_gì", "không_cùng", "không_cần", "không_cứ",
    "không_dùng", "không_gì", "không_hay", "không_khỏi", "không_kể",
    "không_ngoài", "không_nhận", "không_những", "không_phải",
    "không_phải_không", "không_thể", "không_tính", "không_điều_kiện",
    "không_được", "không_đầy", "không_để", "khẳng_định", "khỏi", "khỏi_nói",
    "kể", "kể_cả", "kể_như", "kể_tới", "kể_từ", "liên_quan", "loại", "loại_từ",
    "luôn", "luôn_cả", "luôn_luôn", "luôn_tay", "là", "là_cùng", "là_là",
    "là_nhiều", "là_phải", "là_thế_nào", "là_vì", "là_ít", "làm", "làm_bằng",
    "làm_cho", "làm_dần_dần", "làm_gì", "làm_lòng", "làm_lại", "làm_lấy",
    "làm_mất", "làm_ngay", "làm_như", "làm_nên", "làm_ra", "làm_riêng",
    "làm_sao", "làm_theo", "làm_thế_nào", "làm_tin", "làm_tôi", "làm_tăng",
    "làm_tại", "làm_tắp_lự", "làm_vì", "làm_đúng", "làm_được", "lâu",
    "lâu_các", "lâu_lâu", "lâu_nay", "lâu_ngày", "lên", "lên_cao", "lên_cơn",
    "lên_mạnh", "lên_ngôi", "lên_nước", "lên_số", "lên_xuống", "lên_đến",
    "lòng", "lòng_không", "lúc", "lúc_khác", "lúc_lâu", "lúc_nào", "lúc_này",
    "lúc_sáng", "lúc_trước", "lúc_đi", "lúc_đó", "lúc_đến", "lúc_ấy", "lý_do",
    "lượng", "lượng_cả", "lượng_số", "lượng_từ", "lại", "lại_bộ", "lại_cái",
    "lại_còn", "lại_giống", "lại_làm", "lại_người", "lại_nói", "lại_nữa",
    "lại_quả", "lại_thôi", "lại_ăn", "lại_đây", "lấy", "lấy_có", "lấy_cả",
    "lấy_giống", "lấy_làm", "lấy_lý_do", "lấy_lại", "lấy_ra", "lấy_ráo",
    "lấy_sau", "lấy_số", "lấy_thêm", "lấy_thế", "lấy_vào", "lấy_xuống",
    "lấy_được", "lấy_để", "lần", "lần_khác", "lần_lần", "lần_nào", "lần_này",
    "lần_sang", "lần_sau", "lần_theo", "lần_trước", "lần_tìm", "lớn",
    "lớn_lên", "lớn_nhỏ", "lời", "lời_chú", "lời_nói", "mang", "mang_lại",
    "mang_mang", "mang_nặng", "mang_về", "muốn", "mà", "mà_cả", "mà_không",
    "mà_lại", "mà_thôi", "mà_vẫn", "mình", "mạnh", "mất", "mất_còn", "mọi",
    "mọi_giờ", "mọi_khi", "mọi_lúc", "mọi_người", "mọi_nơi", "mọi_sự",
    "mọi_thứ", "mọi_việc", "mối", "mỗi", "mỗi_lúc", "mỗi_lần", "mỗi_một",
    "mỗi_ngày", "mỗi_người", "một", "một_cách", "một_cơn", "một_khi",
    "một_lúc", "một_số", "một_vài", "một_ít", "mới", "mới_hay", "mới_rồi",
    "mới_đây", "mở", "mở_mang", "mở_nước", "mở_ra", "mợ", "mức", "nay", "ngay",
    "ngay_bây_giờ", "ngay_cả", "ngay_khi", "ngay_khi_đến", "ngay_lúc",
    "ngay_lúc_này", "ngay_lập_tức", "ngay_thật", "ngay_tức_khắc",
    "ngay_tức_thì", "ngay_từ", "nghe", "nghe_chừng", "nghe_hiểu", "nghe_không",
    "nghe_lại", "nghe_nhìn", "nghe_như", "nghe_nói", "nghe_ra", "nghe_rõ",
    "nghe_thấy", "nghe_tin", "nghe_trực_tiếp", "nghe_đâu", "nghe_đâu_như",
    "nghe_được", "nghen", "nghiễm_nhiên", "nghĩ", "nghĩ_lại", "nghĩ_ra",
    "nghĩ_tới", "nghĩ_xa", "nghĩ_đến", "nghỉm", "ngoài", "ngoài_này",
    "ngoài_ra", "ngoài_xa", "ngoải", "nguồn", "ngày", "ngày_càng", "ngày_cấp",
    "ngày_giờ", "ngày_ngày", "ngày_nào", "ngày_này", "ngày_nọ", "ngày_qua",
    "ngày_rày", "ngày_tháng", "ngày_xưa", "ngày_xửa", "ngày_đến", "ngày_ấy",
    "ngôi", "ngôi_nhà", "ngôi_thứ", "ngõ_hầu", "ngăn_ngắt", "ngươi", "người",
    "người_hỏi", "người_khác", "người_khách", "người_mình", "người_nghe",
    "người_người", "người_nhận", "ngọn", "ngọn_nguồn", "ngọt", "ngồi",
    "ngồi_bệt", "ngồi_không", "ngồi_sau", "ngồi_trệt", "ngộ_nhỡ", "nhanh",
    "nhanh_lên", "nhanh_tay", "nhau", "nhiên_hậu", "nhiều", "nhiều_ít",
    "nhiệt_liệt", "nhung_nhăng", "nhà", "nhà_chung", "nhà_khó", "nhà_làm",
    "nhà_ngoài", "nhà_ngươi", "nhà_tôi", "nhà_việc", "nhân_dịp", "nhân_tiện",
    "nhé", "nhìn", "nhìn_chung", "nhìn_lại", "nhìn_nhận", "nhìn_theo",
    "nhìn_thấy", "nhìn_xuống", "nhóm", "nhón_nhén", "như", "như_ai",
    "như_chơi", "như_không", "như_là", "như_nhau", "như_quả", "như_sau",
    "như_thường", "như_thế", "như_thế_nào", "như_thể", "như_trên", "như_trước",
    "như_tuồng", "như_vậy", "như_ý", "nhưng", "nhưng_mà", "nhược_bằng", "nhất",
    "nhất_loạt", "nhất_luật", "nhất_là", "nhất_mực", "nhất_nhất", "nhất_quyết",
    "nhất_sinh", "nhất_thiết", "nhất_thì", "nhất_tâm", "nhất_tề", "nhất_đán",
    "nhất_định", "nhận", "nhận_biết", "nhận_họ", "nhận_làm", "nhận_nhau",
    "nhận_ra", "nhận_thấy", "nhận_việc", "nhận_được", "nhằm", "nhằm_khi",
    "nhằm_lúc", "nhằm_vào", "nhằm_để", "nhỉ", "nhỏ", "nhỏ_người", "nhớ",
    "nhớ_bập_bõm", "nhớ_lại", "nhớ_lấy", "nhớ_ra", "nhờ", "nhờ_chuyển",
    "nhờ_có", "nhờ_nhờ", "nhờ_đó", "nhỡ_ra", "những", "những_ai", "những_khi",
    "những_là", "những_lúc", "những_muốn", "những_như", "nào", "nào_cũng",
    "nào_hay", "nào_là", "nào_phải", "nào_đâu", "nào_đó", "này", "này_nọ",
    "nên", "nên_chi", "nên_chăng", "nên_làm", "nên_người", "nên_tránh", "nó",
    "nóc", "nói", "nói_bông", "nói_chung", "nói_khó", "nói_là", "nói_lên",
    "nói_lại", "nói_nhỏ", "nói_phải", "nói_qua", "nói_ra", "nói_riêng",
    "nói_rõ", "nói_thêm", "nói_thật", "nói_toẹt", "nói_trước", "nói_tốt",
    "nói_với", "nói_xa", "nói_ý", "nói_đến", "nói_đủ", "năm", "năm_tháng",
    "nơi", "nơi_nơi", "nước", "nước_bài", "nước_cùng", "nước_lên", "nước_nặng",
    "nước_quả", "nước_xuống", "nước_ăn", "nước_đến", "nấy", "nặng", "nặng_căn",
    "nặng_mình", "nặng_về", "nếu", "nếu_có", "nếu_cần", "nếu_không", "nếu_mà",
    "nếu_như", "nếu_thế", "nếu_vậy", "nếu_được", "nền", "nọ", "nớ", "nức_nở",
    "nữa", "nữa_khi", "nữa_là", "nữa_rồi", "oai_oái", "oái", "pho", "phè",
    "phè_phè", "phía", "phía_bên", "phía_bạn", "phía_dưới", "phía_sau",
    "phía_trong", "phía_trên", "phía_trước", "phóc", "phót", "phù_hợp",
    "phăn_phắt", "phương_chi", "phải", "phải_biết", "phải_chi", "phải_chăng",
    "phải_cách", "phải_cái", "phải_giờ", "phải_khi", "phải_không", "phải_lại",
    "phải_lời", "phải_người", "phải_như", "phải_rồi", "phải_tay", "phần",
    "phần_lớn", "phần_nhiều", "phần_nào", "phần_sau", "phần_việc", "phắt",
    "phỉ_phui", "phỏng", "phỏng_như", "phỏng_nước", "phỏng_theo", "phỏng_tính",
    "phốc", "phụt", "phứt", "qua", "qua_chuyện", "qua_khỏi", "qua_lại",
    "qua_lần", "qua_ngày", "qua_tay", "qua_thì", "qua_đi", "quan_trọng",
    "quan_trọng_vấn_đề", "quan_tâm", "quay", "quay_bước", "quay_lại",
    "quay_số", "quay_đi", "quá", "quá_bán", "quá_bộ", "quá_giờ", "quá_lời",
    "quá_mức", "quá_nhiều", "quá_tay", "quá_thì", "quá_tin", "quá_trình",
    "quá_tuổi", "quá_đáng", "quá_ư", "quả", "quả_là", "quả_thật", "quả_thế",
    "quả_vậy", "quận", "ra", "ra_bài", "ra_bộ", "ra_chơi", "ra_gì", "ra_lại",
    "ra_lời", "ra_ngôi", "ra_người", "ra_sao", "ra_tay", "ra_vào", "ra_ý",
    "ra_điều", "ra_đây", "ren_rén", "riu_ríu", "riêng", "riêng_từng", "riệt",
    "rày", "ráo", "ráo_cả", "ráo_nước", "ráo_trọi", "rén", "rén_bước", "rích",
    "rón_rén", "rõ", "rõ_là", "rõ_thật", "rút_cục", "răng", "răng_răng", "rất",
    "rất_lâu", "rằng", "rằng_là", "rốt_cuộc", "rốt_cục", "rồi", "rồi_nữa",
    "rồi_ra", "rồi_sao", "rồi_sau", "rồi_tay", "rồi_thì", "rồi_xem", "rồi_đây",
    "rứa", "sa_sả", "sang", "sang_năm", "sang_sáng", "sang_tay", "sao",
    "sao_bản", "sao_bằng", "sao_cho", "sao_vậy", "sao_đang", "sau", "sau_chót",
    "sau_cuối", "sau_cùng", "sau_hết", "sau_này", "sau_nữa", "sau_sau",
    "sau_đây", "sau_đó", "so", "so_với", "song_le", "suýt", "suýt_nữa", "sáng",
    "sáng_ngày", "sáng_rõ", "sáng_thế", "sáng_ý", "sì", "sì_sì", "sất", "sắp",
    "sắp_đặt", "sẽ", "sẽ_biết", "sẽ_hay", "số", "số_cho_biết", "số_cụ_thể",
    "số_loại", "số_là", "số_người", "số_phần", "số_thiếu", "sốt_sột", "sớm",
    "sớm_ngày", "sở_dĩ", "sử_dụng", "sự", "sự_thế", "sự_việc", "tanh",
    "tanh_tanh", "tay", "tay_quay", "tha_hồ", "tha_hồ_chơi", "tha_hồ_ăn",
    "than_ôi", "thanh", "thanh_ba", "thanh_chuyển", "thanh_không",
    "thanh_thanh", "thanh_tính", "thanh_điều_kiện", "thanh_điểm", "thay_đổi",
    "thay_đổi_tình_trạng", "theo", "theo_bước", "theo_như", "theo_tin",
    "thi_thoảng", "thiếu", "thiếu_gì", "thiếu_điểm", "thoạt", "thoạt_nghe",
    "thoạt_nhiên", "thoắt", "thuần", "thuần_ái", "thuộc", "thuộc_bài",
    "thuộc_cách", "thuộc_lại", "thuộc_từ", "thà", "thà_là", "thà_rằng",
    "thành_ra", "thành_thử", "thái_quá", "tháng", "tháng_ngày", "tháng_năm",
    "tháng_tháng", "thêm", "thêm_chuyện", "thêm_giờ", "thêm_vào", "thì",
    "thì_giờ", "thì_là", "thì_phải", "thì_ra", "thì_thôi", "thình_lình",
    "thích", "thích_cứ", "thích_thuộc", "thích_tự", "thích_ý", "thím", "thôi",
    "thôi_việc", "thúng_thắng", "thương_ôi", "thường", "thường_bị",
    "thường_hay", "thường_khi", "thường_số", "thường_sự", "thường_thôi",
    "thường_thường", "thường_tính", "thường_tại", "thường_xuất_hiện",
    "thường_đến", "thảo_hèn", "thảo_nào", "thấp", "thấp_cơ", "thấp_thỏm",
    "thấp_xuống", "thấy", "thấy_tháng", "thẩy", "thậm", "thậm_chí", "thậm_cấp",
    "thậm_từ", "thật", "thật_chắc", "thật_là", "thật_lực", "thật_quả",
    "thật_ra", "thật_sự", "thật_thà", "thật_tốt", "thật_vậy", "thế",
    "thế_chuẩn_bị", "thế_là", "thế_lại", "thế_mà", "thế_nào", "thế_nên",
    "thế_ra", "thế_sự", "thế_thì", "thế_thôi", "thế_thường", "thế_thế",
    "thế_à", "thế_đó", "thếch", "thỉnh_thoảng", "thỏm", "thốc", "thốc_tháo",
    "thốt", "thốt_nhiên", "thốt_nói", "thốt_thôi", "thộc", "thời_gian",
    "thời_gian_sử_dụng", "thời_gian_tính", "thời_điểm", "thục_mạng", "thứ",
    "thứ_bản", "thứ_đến", "thửa", "thực_hiện", "thực_hiện_đúng", "thực_ra",
    "thực_sự", "thực_tế", "thực_vậy", "tin", "tin_thêm", "tin_vào",
    "tiếp_theo", "tiếp_tục", "tiếp_đó", "tiện_thể", "toà", "toé_khói", "toẹt",
    "trong", "trong_khi", "trong_lúc", "trong_mình", "trong_ngoài",
    "trong_này", "trong_số", "trong_vùng", "trong_đó", "trong_ấy", "tránh",
    "tránh_khỏi", "tránh_ra", "tránh_tình_trạng", "tránh_xa", "trên",
    "trên_bộ", "trên_dưới", "trước", "trước_hết", "trước_khi", "trước_kia",
    "trước_nay", "trước_ngày", "trước_nhất", "trước_sau", "trước_tiên",
    "trước_tuổi", "trước_đây", "trước_đó", "trả", "trả_của", "trả_lại",
    "trả_ngay", "trả_trước", "trếu_tráo", "trển", "trệt", "trệu_trạo", "trỏng",
    "trời_đất_ơi", "trở_thành", "trừ_phi", "trực_tiếp", "trực_tiếp_làm", "tuy",
    "tuy_có", "tuy_là", "tuy_nhiên", "tuy_rằng", "tuy_thế", "tuy_vậy",
    "tuy_đã", "tuyệt_nhiên", "tuần_tự", "tuốt_luốt", "tuốt_tuồn_tuột",
    "tuốt_tuột", "tuổi", "tuổi_cả", "tuổi_tôi", "tà_tà", "tên", "tên_chính",
    "tên_cái", "tên_họ", "tên_tự", "tênh", "tênh_tênh", "tìm", "tìm_bạn",
    "tìm_cách", "tìm_hiểu", "tìm_ra", "tìm_việc", "tình_trạng", "tính",
    "tính_cách", "tính_căn", "tính_người", "tính_phỏng", "tính_từ", "tít_mù",
    "tò_te", "tôi", "tôi_con", "tông_tốc", "tù_tì", "tăm_tắp", "tăng",
    "tăng_chúng", "tăng_cấp", "tăng_giảm", "tăng_thêm", "tăng_thế", "tại",
    "tại_lòng", "tại_nơi", "tại_sao", "tại_tôi", "tại_vì", "tại_đâu",
    "tại_đây", "tại_đó", "tạo", "tạo_cơ_hội", "tạo_nên", "tạo_ra", "tạo_ý",
    "tạo_điều_kiện", "tấm", "tấm_bản", "tấm_các", "tấn", "tấn_tới", "tất_cả",
    "tất_cả_bao_nhiêu", "tất_thảy", "tất_tần_tật", "tất_tật", "tập_trung",
    "tắp", "tắp_lự", "tắp_tắp", "tọt", "tỏ_ra", "tỏ_vẻ", "tốc_tả", "tối_ư",
    "tốt", "tốt_bạn", "tốt_bộ", "tốt_hơn", "tốt_mối", "tốt_ngày", "tột",
    "tột_cùng", "tớ", "tới", "tới_gần", "tới_mức", "tới_nơi", "tới_thì",
    "tức_thì", "tức_tốc", "từ", "từ_căn", "từ_giờ", "từ_khi", "từ_loại",
    "từ_nay", "từ_thế", "từ_tính", "từ_tại", "từ_từ", "từ_ái", "từ_điều",
    "từ_đó", "từ_ấy", "từng", "từng_cái", "từng_giờ", "từng_nhà", "từng_phần",
    "từng_thời_gian", "từng_đơn_vị", "từng_ấy", "tự", "tự_cao", "tự_khi",
    "tự_lượng", "tự_tính", "tự_tạo", "tự_vì", "tự_ý", "tự_ăn", "tựu_trung",
    "veo", "veo_veo", "việc", "việc_gì", "vung_thiên_địa", "vung_tàn_tán",
    "vung_tán_tàn", "và", "vài", "vài_ba", "vài_người", "vài_nhà", "vài_nơi",
    "vài_tên", "vài_điều", "vào", "vào_gặp", "vào_khoảng", "vào_lúc",
    "vào_vùng", "vào_đến", "vâng", "vâng_chịu", "vâng_dạ", "vâng_vâng",
    "vâng_ý", "vèo", "vèo_vèo", "vì", "vì_chưng", "vì_rằng", "vì_sao",
    "vì_thế", "vì_vậy", "ví_bằng", "ví_dù", "ví_phỏng", "ví_thử",
    "vô_hình_trung", "vô_kể", "vô_luận", "vô_vàn", "vùng", "vùng_lên",
    "vùng_nước", "văng_tê", "vượt", "vượt_khỏi", "vượt_quá", "vạn_nhất",
    "vả_chăng", "vả_lại", "vấn_đề", "vấn_đề_quan_trọng", "vẫn", "vẫn_thế",
    "vậy", "vậy_là", "vậy_mà", "vậy_nên", "vậy_ra", "vậy_thì", "vậy_ư", "về",
    "về_không", "về_nước", "về_phần", "về_sau", "về_tay", "vị_trí", "vị_tất",
    "vốn_dĩ", "với", "với_lại", "với_nhau", "vở", "vụt", "vừa", "vừa_khi",
    "vừa_lúc", "vừa_mới", "vừa_qua", "vừa_rồi", "vừa_vừa", "xa", "xa_cách",
    "xa_gần", "xa_nhà", "xa_tanh", "xa_tắp", "xa_xa", "xa_xả", "xem",
    "xem_lại", "xem_ra", "xem_số", "xin", "xin_gặp", "xin_vâng", "xiết_bao",
    "xon_xón", "xoành_xoạch", "xoét", "xoẳn", "xoẹt", "xuất_hiện",
    "xuất_kì_bất_ý", "xuất_kỳ_bất_ý", "xuể", "xuống", "xăm_xúi", "xăm_xăm",
    "xăm_xắm", "xảy_ra", "xềnh_xệch", "xệp", "xử_lý", "yêu_cầu", "à", "à_này",
    "à_ơi", "ào", "ào_vào", "ào_ào", "á", "á_à", "ái", "ái_chà", "ái_dà",
    "áng", "áng_như", "âu_là", "ít", "ít_biết", "ít_có", "ít_hơn", "ít_khi",
    "ít_lâu", "ít_nhiều", "ít_nhất", "ít_nữa", "ít_quá", "ít_ra", "ít_thôi",
    "ít_thấy", "ô_hay", "ô_hô", "ô_kê", "ô_kìa", "ôi_chao", "ôi_thôi", "ông",
    "ông_nhỏ", "ông_tạo", "ông_từ", "ông_ấy", "ông_ổng", "úi", "úi_chà",
    "úi_dào", "ý", "ý_chừng", "ý_da", "ý_hoặc", "ăn", "ăn_chung", "ăn_chắc",
    "ăn_chịu", "ăn_cuộc", "ăn_hết", "ăn_hỏi", "ăn_làm", "ăn_người", "ăn_ngồi",
    "ăn_quá", "ăn_riêng", "ăn_sáng", "ăn_tay", "ăn_trên", "ăn_về", "đang",
    "đang_tay", "đang_thì", "điều", "điều_gì", "điều_kiện", "điểm",
    "điểm_chính", "điểm_gặp", "điểm_đầu_tiên", "đành_đạch", "đáng", "đáng_kể",
    "đáng_lí", "đáng_lý", "đáng_lẽ", "đáng_số", "đánh_giá", "đánh_đùng",
    "đáo_để", "đâu", "đâu_có", "đâu_cũng", "đâu_như", "đâu_nào", "đâu_phải",
    "đâu_đâu", "đâu_đây", "đâu_đó", "đây", "đây_này", "đây_rồi", "đây_đó",
    "đã", "đã_hay", "đã_không", "đã_là", "đã_lâu", "đã_thế", "đã_vậy", "đã_đủ",
    "đó", "đó_đây", "đúng", "đúng_ngày", "đúng_ra", "đúng_tuổi", "đúng_với",
    "đơn_vị", "đưa", "đưa_cho", "đưa_chuyện", "đưa_em", "đưa_ra", "đưa_tay",
    "đưa_tin", "đưa_tới", "đưa_vào", "đưa_về", "đưa_xuống", "đưa_đến", "được",
    "được_cái", "được_lời", "được_nước", "được_tin", "đại_loại", "đại_nhân",
    "đại_phàm", "đại_để", "đạt", "đảm_bảo", "đầu_tiên", "đầy", "đầy_năm",
    "đầy_phè", "đầy_tuổi", "đặc_biệt", "đặt", "đặt_làm", "đặt_mình", "đặt_mức",
    "đặt_ra", "đặt_trước", "đặt_để", "đến", "đến_bao_giờ", "đến_cùng",
    "đến_cùng_cực", "đến_cả", "đến_giờ", "đến_gần", "đến_hay", "đến_khi",
    "đến_lúc", "đến_lời", "đến_nay", "đến_ngày", "đến_nơi", "đến_nỗi",
    "đến_thì", "đến_thế", "đến_tuổi", "đến_xem", "đến_điều", "đến_đâu", "đều",
    "đều_bước", "đều_nhau", "đều_đều", "để", "để_cho", "để_giống", "để_không",
    "để_lòng", "để_lại", "để_mà", "để_phần", "để_được", "để_đến_nỗi",
    "đối_với", "đồng_thời", "đủ", "đủ_dùng", "đủ_nơi", "đủ_số", "đủ_điều",
    "đủ_điểm", "ơ", "ơ_hay", "ơ_kìa", "ơi", "ơi_là", "ư", "ạ", "ạ_ơi", "ấy",
    "ấy_là", "ầu_ơ", "ắt", "ắt_hẳn", "ắt_là", "ắt_phải", "ắt_thật", "ối_dào",
    "ối_giời", "ối_giời_ơi", "ồ", "ồ_ồ", "ổng", "ớ", "ớ_này", "ờ", "ờ_ờ", "ở",
    "ở_lại", "ở_như", "ở_nhờ", "ở_năm", "ở_trên", "ở_vào", "ở_đây", "ở_đó",
    "ở_được", "ủa", "ứ_hự", "ứ_ừ", "ừ", "ừ_nhé", "ừ_thì", "ừ_ào", "ừ_ừ", "ử"
},

)
