"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway custom Cloudfront Logging Bucket';
const inProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_14_X,
    handler: 'index.handler'
};
const func = defaults.deployLambdaFunction(stack, inProps);
const [_api] = defaults.RegionalLambdaRestApi(stack, func);
_api.methods.forEach((apiMethod) => {
    // Override the API Gateway Authorization Type from AWS_IAM to NONE
    const child = apiMethod.node.findChild('Resource');
    if (child.authorizationType === 'AWS_IAM') {
        child.addPropertyOverride('AuthorizationType', 'NONE');
        defaults.addCfnSuppressRules(apiMethod, [
            {
                id: "W59",
                reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`,
            },
        ]);
    }
});
new lib_1.CloudFrontToApiGateway(stack, 'cf-apigw', {
    existingApiGatewayObj: _api,
    cloudFrontLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
        versioned: true
    }
});
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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