"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentConfig = exports.MinimumHealthyHosts = void 0;
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value,
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value,
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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