"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationsResourceHandler = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends core_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        this.role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*'],
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: fs.readFileSync(path.join(__dirname, 'lambda/index.py'), 'utf8') },
                Handler: 'index.handler',
                Role: this.role.roleArn,
                Runtime: 'python3.7',
                Timeout: 300,
            },
        });
        resource.node.addDependency(this.role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = cdk.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda;
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
//# sourceMappingURL=data:application/json;base64,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