"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appsync_1 = require("aws-cdk-lib/aws-appsync");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
    /**
     * Lambda authorization type
     */
    AuthorizationType["LAMBDA"] = "AWS_LAMBDA";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApi(api);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resourceArns);
            }
            throw error;
        }
        return this.arns.map((arn) => aws_cdk_lib_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
_a = JSII_RTTI_SYMBOL_1;
IamResource[_a] = { fqn: "@aws-cdk/aws-appsync-alpha.IamResource", version: "2.23.0-alpha.0" };
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApiProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const defaultMode = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) !== null && _d !== void 0 ? _d : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.additionalAuthorizationModes) !== null && _f !== void 0 ? _f : [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new aws_appsync_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(defaultMode.lambdaAuthorizerConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_g = props.schema) !== null && _g !== void 0 ? _g : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (props.domainName) {
            new aws_appsync_1.CfnDomainName(this, 'DomainName', {
                domainName: props.domainName.domainName,
                certificateArn: props.domainName.certificate.certificateArn,
                description: `domain for ${this.name} at ${this.graphqlUrl}`,
            });
            new aws_appsync_1.CfnDomainNameApiAssociation(this, 'DomainAssociation', {
                domainName: props.domainName.domainName,
                apiId: this.apiId,
            });
        }
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_h = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _h === void 0 ? void 0 : _h.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _c;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApiAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromGraphqlApiAttributes);
            }
            throw error;
        }
        const arn = (_c = attrs.graphqlApiArn) !== null && _c !== void 0 ? _c : aws_cdk_lib_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_IamResource(resources);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.LAMBDA).length > 1) {
            throw new Error('You can only have a single AWS Lambda function configured to authorize your API.');
        }
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing User Pool Configuration');
            }
            if (mode.authorizationType === AuthorizationType.LAMBDA && !mode.lambdaAuthorizerConfig) {
                throw new Error('Missing Lambda Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        var _c, _d;
        if (!config)
            return undefined;
        const logsRoleArn = (_d = (_c = config.role) === null || _c === void 0 ? void 0 : _c.roleArn) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        }).roleArn;
        return {
            cloudWatchLogsRoleArn: logsRoleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupLambdaAuthorizerConfig(config) {
        var _c;
        if (!config)
            return undefined;
        return {
            authorizerResultTtlInSeconds: (_c = config.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            authorizerUri: config.handler.functionArn,
            identityValidationExpression: config.validationRegex,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
                lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(mode.lambdaAuthorizerConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        var _c, _d;
        if (((_c = config === null || config === void 0 ? void 0 : config.expires) === null || _c === void 0 ? void 0 : _c.isBefore(aws_cdk_lib_1.Duration.days(1))) || ((_d = config === null || config === void 0 ? void 0 : config.expires) === null || _d === void 0 ? void 0 : _d.isAfter(aws_cdk_lib_1.Duration.days(365)))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = (config === null || config === void 0 ? void 0 : config.expires) ? config === null || config === void 0 ? void 0 : config.expires.toEpoch() : undefined;
        return new aws_appsync_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_IIntermediateType(type);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addType);
            }
            throw error;
        }
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addQuery);
            }
            throw error;
        }
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMutation);
            }
            throw error;
        }
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSubscription);
            }
            throw error;
        }
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
_b = JSII_RTTI_SYMBOL_1;
GraphqlApi[_b] = { fqn: "@aws-cdk/aws-appsync-alpha.GraphqlApi", version: "2.23.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhcGhxbGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdyYXBocWxhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBRUEsaURBQXNHO0FBRXRHLDZDQUErRjtBQUUvRix5REFBaUk7QUFDakksdURBQWdFO0FBQ2hFLHFDQUFrQztBQUtsQzs7R0FFRztBQUNILElBQVksaUJBcUJYO0FBckJELFdBQVksaUJBQWlCO0lBQzNCOztPQUVHO0lBQ0gsd0NBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCxvQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCw0REFBdUMsQ0FBQTtJQUN2Qzs7T0FFRztJQUNILDRDQUF1QixDQUFBO0lBQ3ZCOztPQUVHO0lBQ0gsMENBQXFCLENBQUE7QUFDdkIsQ0FBQyxFQXJCVyxpQkFBaUIsR0FBakIseUJBQWlCLEtBQWpCLHlCQUFpQixRQXFCNUI7QUFvQ0Q7O0dBRUc7QUFDSCxJQUFZLHFCQVNYO0FBVEQsV0FBWSxxQkFBcUI7SUFDL0I7O09BRUc7SUFDSCx3Q0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCxzQ0FBYSxDQUFBO0FBQ2YsQ0FBQyxFQVRXLHFCQUFxQixHQUFyQiw2QkFBcUIsS0FBckIsNkJBQXFCLFFBU2hDO0FBaUlEOztHQUVHO0FBQ0gsSUFBWSxhQWFYO0FBYkQsV0FBWSxhQUFhO0lBQ3ZCOztPQUVHO0lBQ0gsOEJBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsZ0NBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsNEJBQVcsQ0FBQTtBQUNiLENBQUMsRUFiVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQWF4QjtBQTRGRDs7R0FFRztBQUNILE1BQWEsV0FBVztJQXFDdEIsWUFBb0IsSUFBYztRQUNoQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztLQUNsQjtJQXRDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsSUFBYztRQUNwQyxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUM1RDtRQUNELE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDOUI7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFZLEVBQUUsR0FBRyxNQUFnQjtRQUNwRCxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxTQUFTLElBQUksV0FBVyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsQ0FBQztRQUMxRyxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQzlCO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRztRQUNmLE9BQU8sSUFBSSxXQUFXLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0tBQy9CO0lBUUQ7Ozs7T0FJRztJQUNJLFlBQVksQ0FBQyxHQUFlOzs7Ozs7Ozs7O1FBQ2pDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLG1CQUFLLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNwRCxPQUFPLEVBQUUsU0FBUztZQUNsQixRQUFRLEVBQUUsUUFBUSxHQUFHLENBQUMsS0FBSyxFQUFFO1lBQzdCLFNBQVMsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQjtZQUN4QyxZQUFZLEVBQUUsR0FBRyxHQUFHLEVBQUU7U0FDdkIsQ0FBQyxDQUFDLENBQUM7S0FDTDs7QUFyREgsa0NBc0RDOzs7QUFtQkQ7Ozs7R0FJRztBQUNILE1BQWEsVUFBVyxTQUFRLGdDQUFjO0lBbUU1QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCOztRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDOzs7Ozs7Ozs7O1FBRWpCLE1BQU0sV0FBVyxlQUFHLEtBQUssQ0FBQyxtQkFBbUIsMENBQUUsb0JBQW9CLG1DQUNqRSxFQUFFLGlCQUFpQixFQUFFLGlCQUFpQixDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ25ELE1BQU0sZUFBZSxlQUFHLEtBQUssQ0FBQyxtQkFBbUIsMENBQUUsNEJBQTRCLG1DQUFJLEVBQUUsQ0FBQztRQUN0RixNQUFNLEtBQUssR0FBRyxDQUFDLFdBQVcsRUFBRSxHQUFHLGVBQWUsQ0FBQyxDQUFDO1FBRWhELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFFekQsSUFBSSxDQUFDLDBCQUEwQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXZDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSwyQkFBYSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDN0MsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGtCQUFrQixFQUFFLFdBQVcsQ0FBQyxpQkFBaUI7WUFDakQsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUMvQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDO1lBQ25GLGNBQWMsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQztZQUNwRSxzQkFBc0IsRUFBRSxJQUFJLENBQUMsMkJBQTJCLENBQUMsV0FBVyxDQUFDLHNCQUFzQixDQUFDO1lBQzVGLGlDQUFpQyxFQUFFLElBQUksQ0FBQyxpQ0FBaUMsQ0FBQyxlQUFlLENBQUM7WUFDMUYsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQy9CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUM7UUFDaEMsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQztRQUM1QixJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDO1FBQzFDLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7UUFDMUIsSUFBSSxDQUFDLE1BQU0sU0FBRyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxJQUFJLGVBQU0sRUFBRSxDQUFDO1FBQzNDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0MsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLElBQUksMkJBQWEsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO2dCQUNwQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxVQUFVO2dCQUN2QyxjQUFjLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsY0FBYztnQkFDM0QsV0FBVyxFQUFFLGNBQWMsSUFBSSxDQUFDLElBQUksT0FBTyxJQUFJLENBQUMsVUFBVSxFQUFFO2FBQzdELENBQUMsQ0FBQztZQUVILElBQUkseUNBQTJCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO2dCQUN6RCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxVQUFVO2dCQUN2QyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7YUFDbEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUM5RSxNQUFNLE1BQU0sU0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBdUIsRUFBRSxFQUFFO2dCQUNwRCxPQUFPLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQztZQUNuRixDQUFDLENBQUMsMENBQUUsWUFBWSxDQUFDO1lBQ2pCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNoRCxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7WUFDdEQsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQztTQUM5QztLQUNGO0lBckhEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEyQjs7Ozs7Ozs7Ozs7UUFDOUYsTUFBTSxHQUFHLFNBQUcsS0FBSyxDQUFDLGFBQWEsbUNBQUksbUJBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQzNELE9BQU8sRUFBRSxTQUFTO1lBQ2xCLFFBQVEsRUFBRSxRQUFRLEtBQUssQ0FBQyxZQUFZLEVBQUU7U0FDdkMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxNQUFPLFNBQVEsZ0NBQWM7WUFHakMsWUFBWSxDQUFZLEVBQUUsQ0FBUztnQkFDakMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFIRSxVQUFLLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztnQkFDM0IsUUFBRyxHQUFHLEdBQUcsQ0FBQztZQUcxQixDQUFDO1NBQ0Y7UUFDRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5QjtJQW1HRDs7Ozs7OztPQU9HO0lBQ0ksS0FBSyxDQUFDLE9BQW1CLEVBQUUsU0FBc0IsRUFBRSxHQUFHLE9BQWlCOzs7Ozs7Ozs7O1FBQzVFLE9BQU8sZUFBSyxDQUFDLGNBQWMsQ0FBQztZQUMxQixPQUFPO1lBQ1AsT0FBTztZQUNQLFlBQVksRUFBRSxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQztZQUMxQyxLQUFLLEVBQUUsSUFBSTtTQUNaLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYSxDQUFDLE9BQW1CLEVBQUUsR0FBRyxNQUFnQjtRQUMzRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLEdBQUcsTUFBTSxDQUFDLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztLQUMxRjtJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVUsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDeEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7S0FDdkY7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUIsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDL0QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7S0FDOUY7SUFFTywwQkFBMEIsQ0FBQyxLQUEwQjtRQUMzRCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzFGLE1BQU0sSUFBSSxLQUFLLENBQUMsa0ZBQWtGLENBQUMsQ0FBQztTQUNyRztRQUNELEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUNqQixJQUFJLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUU7Z0JBQ2xGLE1BQU0sSUFBSSxLQUFLLENBQUMsNEJBQTRCLENBQUMsQ0FBQzthQUMvQztZQUNELElBQUksSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLFNBQVMsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUU7Z0JBQ2xGLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQzthQUNwRDtZQUNELElBQUksSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRTtnQkFDdkYsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO2FBQ2pEO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzNGLE1BQU0sSUFBSSxLQUFLLENBQUMsbUhBQW1ILENBQUMsQ0FBQztTQUN0STtRQUNELElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDdkYsTUFBTSxJQUFJLEtBQUssQ0FBQywrR0FBK0csQ0FBQyxDQUFDO1NBQ2xJO0tBQ0Y7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQW1CLENBQUMsU0FBc0I7UUFDL0MsU0FBUyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUVPLGNBQWMsQ0FBQyxNQUFrQjs7UUFDdkMsSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUM5QixNQUFNLFdBQVcsZUFBVyxNQUFNLENBQUMsSUFBSSwwQ0FBRSxPQUFPLG1DQUFJLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDaEYsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsdUJBQXVCLENBQUM7WUFDeEQsZUFBZSxFQUFFO2dCQUNmLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsNkNBQTZDLENBQUM7YUFDdEY7U0FDRixDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ1gsT0FBTztZQUNMLHFCQUFxQixFQUFFLFdBQVc7WUFDbEMscUJBQXFCLEVBQUUsTUFBTSxDQUFDLHFCQUFxQjtZQUNuRCxhQUFhLEVBQUUsTUFBTSxDQUFDLGFBQWE7U0FDcEMsQ0FBQztLQUNIO0lBRU8sd0JBQXdCLENBQUMsTUFBNEI7UUFDM0QsSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUM5QixPQUFPO1lBQ0wsT0FBTyxFQUFFLE1BQU0sQ0FBQyxtQkFBbUI7WUFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1lBQ3pCLE1BQU0sRUFBRSxNQUFNLENBQUMsb0JBQW9CO1lBQ25DLE1BQU0sRUFBRSxNQUFNLENBQUMsWUFBWTtTQUM1QixDQUFDO0tBQ0g7SUFFTyxtQkFBbUIsQ0FBQyxNQUF1QjtRQUNqRCxJQUFJLENBQUMsTUFBTTtZQUFFLE9BQU8sU0FBUyxDQUFDO1FBQzlCLE9BQU87WUFDTCxVQUFVLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVO1lBQ3RDLFNBQVMsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxNQUFNO1lBQ3ZDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxnQkFBZ0I7WUFDekMsYUFBYSxFQUFFLE1BQU0sQ0FBQyxhQUFhLElBQUkscUJBQXFCLENBQUMsS0FBSztTQUNuRSxDQUFDO0tBQ0g7SUFFTywyQkFBMkIsQ0FBQyxNQUErQjs7UUFDakUsSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUM5QixPQUFPO1lBQ0wsNEJBQTRCLFFBQUUsTUFBTSxDQUFDLGVBQWUsMENBQUUsU0FBUyxFQUFFO1lBQ2pFLGFBQWEsRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDekMsNEJBQTRCLEVBQUUsTUFBTSxDQUFDLGVBQWU7U0FDckQsQ0FBQztLQUNIO0lBRU8saUNBQWlDLENBQUMsS0FBMkI7UUFDbkUsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUM7WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUNuRCxPQUFPLEtBQUssQ0FBQyxNQUFNLENBQTJELENBQUMsR0FBRyxFQUFFLElBQUksRUFBRSxFQUFFLENBQUM7WUFDM0YsR0FBRyxHQUFHLEVBQUU7Z0JBQ04sa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjtnQkFDMUMsY0FBYyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDO2dCQUM3RCxtQkFBbUIsRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDO2dCQUM1RSxzQkFBc0IsRUFBRSxJQUFJLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDO2FBQ3RGO1NBQ0YsRUFBRSxFQUFFLENBQUMsQ0FBQztLQUNSO0lBRU8sWUFBWSxDQUFDLE1BQXFCOztRQUN4QyxJQUFJLE9BQUEsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLE9BQU8sMENBQUUsUUFBUSxDQUFDLHNCQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxhQUFLLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxPQUFPLDBDQUFFLE9BQU8sQ0FBQyxzQkFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBQyxFQUFFO1lBQy9GLE1BQU0sS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7U0FDbkU7UUFDRCxNQUFNLE9BQU8sR0FBRyxDQUFBLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxPQUFPLEVBQUMsQ0FBQyxDQUFDLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxPQUFPLENBQUMsT0FBTyxHQUFHLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDeEUsT0FBTyxJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQSxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsSUFBSSxLQUFJLFNBQVMsUUFBUSxFQUFFO1lBQy9ELE9BQU87WUFDUCxXQUFXLEVBQUUsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLFdBQVc7WUFDaEMsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLO1NBQ2xCLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxXQUFXLENBQUMsUUFBZ0IsRUFBRSxTQUFrQjtRQUNyRCxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsU0FBUyxDQUFDLENBQUM7S0FDOUM7SUFFRDs7Ozs7T0FLRztJQUNJLE9BQU8sQ0FBQyxJQUF1Qjs7Ozs7Ozs7OztRQUNwQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQ2xDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFFBQVEsQ0FBQyxTQUFpQixFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQ3ZELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQy9DO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVcsQ0FBQyxTQUFpQixFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQzFELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ2xEO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWUsQ0FBQyxTQUFpQixFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQzlELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ3REOztBQWpWSCxnQ0FrVkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQ2VydGlmaWNhdGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCB7IElVc2VyUG9vbCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jb2duaXRvJztcbmltcG9ydCB7IE1hbmFnZWRQb2xpY3ksIFJvbGUsIElSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsLCBHcmFudCwgSUdyYW50YWJsZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgSUZ1bmN0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIENmblJlc291cmNlLCBEdXJhdGlvbiwgRXhwaXJhdGlvbiwgSVJlc29sdmFibGUsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5BcGlLZXksIENmbkdyYXBoUUxBcGksIENmbkdyYXBoUUxTY2hlbWEsIENmbkRvbWFpbk5hbWUsIENmbkRvbWFpbk5hbWVBcGlBc3NvY2lhdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcHBzeW5jJztcbmltcG9ydCB7IElHcmFwaHFsQXBpLCBHcmFwaHFsQXBpQmFzZSB9IGZyb20gJy4vZ3JhcGhxbGFwaS1iYXNlJztcbmltcG9ydCB7IFNjaGVtYSB9IGZyb20gJy4vc2NoZW1hJztcbmltcG9ydCB7IElJbnRlcm1lZGlhdGVUeXBlIH0gZnJvbSAnLi9zY2hlbWEtYmFzZSc7XG5pbXBvcnQgeyBSZXNvbHZhYmxlRmllbGQgfSBmcm9tICcuL3NjaGVtYS1maWVsZCc7XG5pbXBvcnQgeyBPYmplY3RUeXBlIH0gZnJvbSAnLi9zY2hlbWEtaW50ZXJtZWRpYXRlJztcblxuLyoqXG4gKiBlbnVtIHdpdGggYWxsIHBvc3NpYmxlIHZhbHVlcyBmb3IgQXBwU3luYyBhdXRob3JpemF0aW9uIHR5cGVcbiAqL1xuZXhwb3J0IGVudW0gQXV0aG9yaXphdGlvblR5cGUge1xuICAvKipcbiAgICogQVBJIEtleSBhdXRob3JpemF0aW9uIHR5cGVcbiAgICovXG4gIEFQSV9LRVkgPSAnQVBJX0tFWScsXG4gIC8qKlxuICAgKiBBV1MgSUFNIGF1dGhvcml6YXRpb24gdHlwZS4gQ2FuIGJlIHVzZWQgd2l0aCBDb2duaXRvIElkZW50aXR5IFBvb2wgZmVkZXJhdGVkIGNyZWRlbnRpYWxzXG4gICAqL1xuICBJQU0gPSAnQVdTX0lBTScsXG4gIC8qKlxuICAgKiBDb2duaXRvIFVzZXIgUG9vbCBhdXRob3JpemF0aW9uIHR5cGVcbiAgICovXG4gIFVTRVJfUE9PTCA9ICdBTUFaT05fQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgLyoqXG4gICAqIE9wZW5JRCBDb25uZWN0IGF1dGhvcml6YXRpb24gdHlwZVxuICAgKi9cbiAgT0lEQyA9ICdPUEVOSURfQ09OTkVDVCcsXG4gIC8qKlxuICAgKiBMYW1iZGEgYXV0aG9yaXphdGlvbiB0eXBlXG4gICAqL1xuICBMQU1CREEgPSAnQVdTX0xBTUJEQScsXG59XG5cbi8qKlxuICogSW50ZXJmYWNlIHRvIHNwZWNpZnkgZGVmYXVsdCBvciBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24ocylcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBdXRob3JpemF0aW9uTW9kZSB7XG4gIC8qKlxuICAgKiBPbmUgb2YgcG9zc2libGUgZm91ciB2YWx1ZXMgQXBwU3luYyBzdXBwb3J0c1xuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9zZWN1cml0eS5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVlgXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGU7XG4gIC8qKlxuICAgKiBJZiBhdXRob3JpemF0aW9uVHlwZSBpcyBgQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MYCwgdGhpcyBvcHRpb24gaXMgcmVxdWlyZWQuXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgdXNlclBvb2xDb25maWc/OiBVc2VyUG9vbENvbmZpZztcbiAgLyoqXG4gICAqIElmIGF1dGhvcml6YXRpb25UeXBlIGlzIGBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZYCwgdGhpcyBvcHRpb24gY2FuIGJlIGNvbmZpZ3VyZWQuXG4gICAqIEBkZWZhdWx0IC0gbmFtZTogJ0RlZmF1bHRBUElLZXknIHwgZGVzY3JpcHRpb246ICdEZWZhdWx0IEFQSSBLZXkgY3JlYXRlZCBieSBDREsnXG4gICAqL1xuICByZWFkb25seSBhcGlLZXlDb25maWc/OiBBcGlLZXlDb25maWc7XG4gIC8qKlxuICAgKiBJZiBhdXRob3JpemF0aW9uVHlwZSBpcyBgQXV0aG9yaXphdGlvblR5cGUuT0lEQ2AsIHRoaXMgb3B0aW9uIGlzIHJlcXVpcmVkLlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IG9wZW5JZENvbm5lY3RDb25maWc/OiBPcGVuSWRDb25uZWN0Q29uZmlnO1xuICAvKipcbiAgICogSWYgYXV0aG9yaXphdGlvblR5cGUgaXMgYEF1dGhvcml6YXRpb25UeXBlLkxBTUJEQWAsIHRoaXMgb3B0aW9uIGlzIHJlcXVpcmVkLlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGxhbWJkYUF1dGhvcml6ZXJDb25maWc/OiBMYW1iZGFBdXRob3JpemVyQ29uZmlnO1xufVxuXG4vKipcbiAqIGVudW0gd2l0aCBhbGwgcG9zc2libGUgdmFsdWVzIGZvciBDb2duaXRvIHVzZXItcG9vbCBkZWZhdWx0IGFjdGlvbnNcbiAqL1xuZXhwb3J0IGVudW0gVXNlclBvb2xEZWZhdWx0QWN0aW9uIHtcbiAgLyoqXG4gICAqIEFMTE9XIGFjY2VzcyB0byBBUElcbiAgICovXG4gIEFMTE9XID0gJ0FMTE9XJyxcbiAgLyoqXG4gICAqIERFTlkgYWNjZXNzIHRvIEFQSVxuICAgKi9cbiAgREVOWSA9ICdERU5ZJyxcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBDb2duaXRvIHVzZXItcG9vbHMgaW4gQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sQ29uZmlnIHtcbiAgLyoqXG4gICAqIFRoZSBDb2duaXRvIHVzZXIgcG9vbCB0byB1c2UgYXMgaWRlbnRpdHkgc291cmNlXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbDogSVVzZXJQb29sO1xuICAvKipcbiAgICogdGhlIG9wdGlvbmFsIGFwcCBpZCByZWdleFxuICAgKlxuICAgKiBAZGVmYXVsdCAtICBOb25lXG4gICAqL1xuICByZWFkb25seSBhcHBJZENsaWVudFJlZ2V4Pzogc3RyaW5nO1xuICAvKipcbiAgICogRGVmYXVsdCBhdXRoIGFjdGlvblxuICAgKlxuICAgKiBAZGVmYXVsdCBBTExPV1xuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdEFjdGlvbj86IFVzZXJQb29sRGVmYXVsdEFjdGlvbjtcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBBUEkgS2V5IGF1dGhvcml6YXRpb24gaW4gQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEFwaUtleUNvbmZpZyB7XG4gIC8qKlxuICAgKiBVbmlxdWUgbmFtZSBvZiB0aGUgQVBJIEtleVxuICAgKiBAZGVmYXVsdCAtICdEZWZhdWx0QVBJS2V5J1xuICAgKi9cbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAqIERlc2NyaXB0aW9uIG9mIEFQSSBrZXlcbiAgICogQGRlZmF1bHQgLSAnRGVmYXVsdCBBUEkgS2V5IGNyZWF0ZWQgYnkgQ0RLJ1xuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIGZyb20gY3JlYXRpb24gdGltZSBhZnRlciB3aGljaCB0aGUgQVBJIGtleSBleHBpcmVzLlxuICAgKiBJdCBtdXN0IGJlIGEgbWluaW11bSBvZiAxIGRheSBhbmQgYSBtYXhpbXVtIG9mIDM2NSBkYXlzIGZyb20gZGF0ZSBvZiBjcmVhdGlvbi5cbiAgICogUm91bmRlZCBkb3duIHRvIHRoZSBuZWFyZXN0IGhvdXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gNyBkYXlzIHJvdW5kZWQgZG93biB0byBuZWFyZXN0IGhvdXJcbiAgICovXG4gIHJlYWRvbmx5IGV4cGlyZXM/OiBFeHBpcmF0aW9uO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIE9wZW5JRCBDb25uZWN0IGF1dGhvcml6YXRpb24gaW4gQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5JZENvbm5lY3RDb25maWcge1xuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBtaWxsaXNlY29uZHMgYW4gT0lEQyB0b2tlbiBpcyB2YWxpZCBhZnRlciBiZWluZyBhdXRoZW50aWNhdGVkIGJ5IE9JREMgcHJvdmlkZXIuXG4gICAqIGBhdXRoX3RpbWVgIGNsYWltIGluIE9JREMgdG9rZW4gaXMgcmVxdWlyZWQgZm9yIHRoaXMgdmFsaWRhdGlvbiB0byB3b3JrLlxuICAgKiBAZGVmYXVsdCAtIG5vIHZhbGlkYXRpb25cbiAgICovXG4gIHJlYWRvbmx5IHRva2VuRXhwaXJ5RnJvbUF1dGg/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIG1pbGxpc2Vjb25kcyBhbiBPSURDIHRva2VuIGlzIHZhbGlkIGFmdGVyIGJlaW5nIGlzc3VlZCB0byBhIHVzZXIuXG4gICAqIFRoaXMgdmFsaWRhdGlvbiB1c2VzIGBpYXRgIGNsYWltIG9mIE9JREMgdG9rZW4uXG4gICAqIEBkZWZhdWx0IC0gbm8gdmFsaWRhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgdG9rZW5FeHBpcnlGcm9tSXNzdWU/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgY2xpZW50IGlkZW50aWZpZXIgb2YgdGhlIFJlbHlpbmcgcGFydHkgYXQgdGhlIE9wZW5JRCBpZGVudGl0eSBwcm92aWRlci5cbiAgICogQSByZWd1bGFyIGV4cHJlc3Npb24gY2FuIGJlIHNwZWNpZmllZCBzbyBBcHBTeW5jIGNhbiB2YWxpZGF0ZSBhZ2FpbnN0IG11bHRpcGxlIGNsaWVudCBpZGVudGlmaWVycyBhdCBhIHRpbWUuXG4gICAqIEBleGFtcGxlIC0gJ0FCQ0R8Q0RFRicgLy8gd2hlcmUgQUJDRCBhbmQgQ0RFRiBhcmUgdHdvIGRpZmZlcmVudCBjbGllbnRJZFxuICAgKiBAZGVmYXVsdCAtICogKEFsbClcbiAgICovXG4gIHJlYWRvbmx5IGNsaWVudElkPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGlzc3VlciBmb3IgdGhlIE9JREMgY29uZmlndXJhdGlvbi4gVGhlIGlzc3VlciByZXR1cm5lZCBieSBkaXNjb3ZlcnkgbXVzdCBleGFjdGx5IG1hdGNoIHRoZSB2YWx1ZSBvZiBgaXNzYCBpbiB0aGUgT0lEQyB0b2tlbi5cbiAgICovXG4gIHJlYWRvbmx5IG9pZGNQcm92aWRlcjogc3RyaW5nO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIExhbWJkYSBhdXRob3JpemF0aW9uIGluIEFwcFN5bmMuIE5vdGUgdGhhdCB5b3UgY2FuIG9ubHkgaGF2ZSBhIHNpbmdsZSBBV1MgTGFtYmRhIGZ1bmN0aW9uIGNvbmZpZ3VyZWQgdG8gYXV0aG9yaXplIHlvdXIgQVBJLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExhbWJkYUF1dGhvcml6ZXJDb25maWcge1xuICAvKipcbiAgICogVGhlIGF1dGhvcml6ZXIgbGFtYmRhIGZ1bmN0aW9uLlxuICAgKiBOb3RlOiBUaGlzIExhbWJkYSBmdW5jdGlvbiBtdXN0IGhhdmUgdGhlIGZvbGxvd2luZyByZXNvdXJjZS1iYXNlZCBwb2xpY3kgYXNzaWduZWQgdG8gaXQuXG4gICAqIFdoZW4gY29uZmlndXJpbmcgTGFtYmRhIGF1dGhvcml6ZXJzIGluIHRoZSBjb25zb2xlLCB0aGlzIGlzIGRvbmUgZm9yIHlvdS5cbiAgICogVG8gZG8gc28gd2l0aCB0aGUgQVdTIENMSSwgcnVuIHRoZSBmb2xsb3dpbmc6XG4gICAqXG4gICAqIGBhd3MgbGFtYmRhIGFkZC1wZXJtaXNzaW9uIC0tZnVuY3Rpb24tbmFtZSBcImFybjphd3M6bGFtYmRhOnVzLWVhc3QtMjoxMTExMjIyMjMzMzM6ZnVuY3Rpb246bXktZnVuY3Rpb25cIiAtLXN0YXRlbWVudC1pZCBcImFwcHN5bmNcIiAtLXByaW5jaXBhbCBhcHBzeW5jLmFtYXpvbmF3cy5jb20gLS1hY3Rpb24gbGFtYmRhOkludm9rZUZ1bmN0aW9uYFxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWFwcHN5bmMtZ3JhcGhxbGFwaS1sYW1iZGFhdXRob3JpemVyY29uZmlnLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IGhhbmRsZXI6IElGdW5jdGlvbjtcblxuICAvKipcbiAgICogSG93IGxvbmcgdGhlIHJlc3VsdHMgYXJlIGNhY2hlZC5cbiAgICogRGlzYWJsZSBjYWNoaW5nIGJ5IHNldHRpbmcgdGhpcyB0byAwLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5taW51dGVzKDUpXG4gICAqL1xuICByZWFkb25seSByZXN1bHRzQ2FjaGVUdGw/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogQSByZWd1bGFyIGV4cHJlc3Npb24gZm9yIHZhbGlkYXRpb24gb2YgdG9rZW5zIGJlZm9yZSB0aGUgTGFtYmRhIGZ1bmN0aW9uIGlzIGNhbGxlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyByZWdleCBmaWx0ZXIgd2lsbCBiZSBhcHBsaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdmFsaWRhdGlvblJlZ2V4Pzogc3RyaW5nO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gb2YgdGhlIEFQSSBhdXRob3JpemF0aW9uIG1vZGVzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhvcml6YXRpb25Db25maWcge1xuICAvKipcbiAgICogT3B0aW9uYWwgYXV0aG9yaXphdGlvbiBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVBJIEtleSBhdXRob3JpemF0aW9uXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0QXV0aG9yaXphdGlvbj86IEF1dGhvcml6YXRpb25Nb2RlO1xuXG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIGF1dGhvcml6YXRpb24gbW9kZXNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBvdGhlciBtb2Rlc1xuICAgKi9cbiAgcmVhZG9ubHkgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2Rlcz86IEF1dGhvcml6YXRpb25Nb2RlW107XG59XG5cbi8qKlxuICogbG9nLWxldmVsIGZvciBmaWVsZHMgaW4gQXBwU3luY1xuICovXG5leHBvcnQgZW51bSBGaWVsZExvZ0xldmVsIHtcbiAgLyoqXG4gICAqIE5vIGxvZ2dpbmdcbiAgICovXG4gIE5PTkUgPSAnTk9ORScsXG4gIC8qKlxuICAgKiBFcnJvciBsb2dnaW5nXG4gICAqL1xuICBFUlJPUiA9ICdFUlJPUicsXG4gIC8qKlxuICAgKiBBbGwgbG9nZ2luZ1xuICAgKi9cbiAgQUxMID0gJ0FMTCcsXG59XG5cbi8qKlxuICogTG9nZ2luZyBjb25maWd1cmF0aW9uIGZvciBBcHBTeW5jXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTG9nQ29uZmlnIHtcbiAgLyoqXG4gICAqIGV4Y2x1ZGUgdmVyYm9zZSBjb250ZW50XG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBleGNsdWRlVmVyYm9zZUNvbnRlbnQ/OiBib29sZWFuIHwgSVJlc29sdmFibGU7XG4gIC8qKlxuICAgKiBsb2cgbGV2ZWwgZm9yIGZpZWxkc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIFVzZSBBcHBTeW5jIGRlZmF1bHRcbiAgICovXG4gIHJlYWRvbmx5IGZpZWxkTG9nTGV2ZWw/OiBGaWVsZExvZ0xldmVsO1xuXG4gIC8qKlxuICAgKiBUaGUgcm9sZSBmb3IgQ2xvdWRXYXRjaCBMb2dzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IElSb2xlO1xufVxuXG4vKipcbiAqIERvbWFpbiBuYW1lIGNvbmZpZ3VyYXRpb24gZm9yIEFwcFN5bmNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBjZXJ0aWZpY2F0ZSB0byB1c2Ugd2l0aCB0aGUgZG9tYWluIG5hbWUuXG4gICAqL1xuICByZWFkb25seSBjZXJ0aWZpY2F0ZTogSUNlcnRpZmljYXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgYWN0dWFsIGRvbWFpbiBuYW1lLiBGb3IgZXhhbXBsZSwgYGFwaS5leGFtcGxlLmNvbWAuXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gQXBwU3luYyBHcmFwaFFMIEFQSVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdyYXBocWxBcGlQcm9wcyB7XG4gIC8qKlxuICAgKiB0aGUgbmFtZSBvZiB0aGUgR3JhcGhRTCBBUElcbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogT3B0aW9uYWwgYXV0aG9yaXphdGlvbiBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVBJIEtleSBhdXRob3JpemF0aW9uXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uQ29uZmlnPzogQXV0aG9yaXphdGlvbkNvbmZpZztcblxuICAvKipcbiAgICogTG9nZ2luZyBjb25maWd1cmF0aW9uIGZvciB0aGlzIGFwaVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGxvZ0NvbmZpZz86IExvZ0NvbmZpZztcblxuICAvKipcbiAgICogR3JhcGhRTCBzY2hlbWEgZGVmaW5pdGlvbi4gU3BlY2lmeSBob3cgeW91IHdhbnQgdG8gZGVmaW5lIHlvdXIgc2NoZW1hLlxuICAgKlxuICAgKiBTY2hlbWEuZnJvbUZpbGUoZmlsZVBhdGg6IHN0cmluZykgYWxsb3dzIHNjaGVtYSBkZWZpbml0aW9uIHRocm91Z2ggc2NoZW1hLmdyYXBocWwgZmlsZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHNjaGVtYSB3aWxsIGJlIGdlbmVyYXRlZCBjb2RlLWZpcnN0IChpLmUuIGFkZFR5cGUsIGFkZE9iamVjdFR5cGUsIGV0Yy4pXG4gICAqXG4gICAqL1xuICByZWFkb25seSBzY2hlbWE/OiBTY2hlbWE7XG4gIC8qKlxuICAgKiBBIGZsYWcgaW5kaWNhdGluZyB3aGV0aGVyIG9yIG5vdCBYLVJheSB0cmFjaW5nIGlzIGVuYWJsZWQgZm9yIHRoZSBHcmFwaFFMIEFQSS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgeHJheUVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIG5hbWUgY29uZmlndXJhdGlvbiBmb3IgdGhlIEdyYXBoUUwgQVBJXG4gICAqXG4gICAqIFRoZSBSb3V0ZSA1MyBob3N0ZWQgem9uZSBhbmQgQ05hbWUgRE5TIHJlY29yZCBtdXN0IGJlIGNvbmZpZ3VyZWQgaW4gYWRkaXRpb24gdG8gdGhpcyBzZXR0aW5nIHRvXG4gICAqIGVuYWJsZSBjdXN0b20gZG9tYWluIFVSTFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRvbWFpbiBuYW1lXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lPzogRG9tYWluT3B0aW9ucztcbn1cblxuLyoqXG4gKiBBIGNsYXNzIHVzZWQgdG8gZ2VuZXJhdGUgcmVzb3VyY2UgYXJucyBmb3IgQXBwU3luY1xuICovXG5leHBvcnQgY2xhc3MgSWFtUmVzb3VyY2Uge1xuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIHJlc291cmNlIG5hbWVzIGdpdmVuIGN1c3RvbSBhcm5zXG4gICAqXG4gICAqIEBwYXJhbSBhcm5zIFRoZSBjdXN0b20gYXJucyB0aGF0IG5lZWQgdG8gYmUgcGVybWlzc2lvbmVkXG4gICAqXG4gICAqIEV4YW1wbGU6IGN1c3RvbSgnL3R5cGVzL1F1ZXJ5L2ZpZWxkcy9nZXRFeGFtcGxlJylcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY3VzdG9tKC4uLmFybnM6IHN0cmluZ1tdKTogSWFtUmVzb3VyY2Uge1xuICAgIGlmIChhcm5zLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCAxIGN1c3RvbSBBUk4gbXVzdCBiZSBwcm92aWRlZC4nKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBJYW1SZXNvdXJjZShhcm5zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgcmVzb3VyY2UgbmFtZXMgZ2l2ZW4gYSB0eXBlIGFuZCBmaWVsZHNcbiAgICpcbiAgICogQHBhcmFtIHR5cGUgVGhlIHR5cGUgdGhhdCBuZWVkcyB0byBiZSBhbGxvd2VkXG4gICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0aGF0IG5lZWQgdG8gYmUgYWxsb3dlZCwgaWYgZW1wdHkgZ3JhbnQgcGVybWlzc2lvbnMgdG8gQUxMIGZpZWxkc1xuICAgKlxuICAgKiBFeGFtcGxlOiBvZlR5cGUoJ1F1ZXJ5JywgJ0dldEV4YW1wbGUnKVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvZlR5cGUodHlwZTogc3RyaW5nLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogSWFtUmVzb3VyY2Uge1xuICAgIGNvbnN0IGFybnMgPSBmaWVsZHMubGVuZ3RoID8gZmllbGRzLm1hcCgoZmllbGQpID0+IGB0eXBlcy8ke3R5cGV9L2ZpZWxkcy8ke2ZpZWxkfWApIDogW2B0eXBlcy8ke3R5cGV9LypgXTtcbiAgICByZXR1cm4gbmV3IElhbVJlc291cmNlKGFybnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSByZXNvdXJjZSBuYW1lcyB0aGF0IGFjY2VwdHMgYWxsIHR5cGVzOiBgKmBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYWxsKCk6IElhbVJlc291cmNlIHtcbiAgICByZXR1cm4gbmV3IElhbVJlc291cmNlKFsnKiddKTtcbiAgfVxuXG4gIHByaXZhdGUgYXJuczogc3RyaW5nW107XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihhcm5zOiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYXJucyA9IGFybnM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIHRoZSBSZXNvdXJjZSBBUk5cbiAgICpcbiAgICogQHBhcmFtIGFwaSBUaGUgR3JhcGhRTCBBUEkgdG8gZ2l2ZSBwZXJtaXNzaW9uc1xuICAgKi9cbiAgcHVibGljIHJlc291cmNlQXJucyhhcGk6IEdyYXBocWxBcGkpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIHRoaXMuYXJucy5tYXAoKGFybikgPT4gU3RhY2sub2YoYXBpKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2FwcHN5bmMnLFxuICAgICAgcmVzb3VyY2U6IGBhcGlzLyR7YXBpLmFwaUlkfWAsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHthcm59YCxcbiAgICB9KSk7XG4gIH1cbn1cblxuLyoqXG4gKiBBdHRyaWJ1dGVzIGZvciBHcmFwaFFMIGltcG9ydHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHcmFwaHFsQXBpQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBhbiB1bmlxdWUgQVdTIEFwcFN5bmMgR3JhcGhRTCBBUEkgaWRlbnRpZmllclxuICAgKiBpLmUuICdseHo3NzVsd2RyZ2NuZGd6M251cnZhYzdvYSdcbiAgICovXG4gIHJlYWRvbmx5IGdyYXBocWxBcGlJZDogc3RyaW5nLFxuXG4gIC8qKlxuICAgKiB0aGUgYXJuIGZvciB0aGUgR3JhcGhRTCBBcGlcbiAgICogQGRlZmF1bHQgLSBhdXRvZ2VuZXJhdGVkIGFyblxuICAgKi9cbiAgcmVhZG9ubHkgZ3JhcGhxbEFwaUFybj86IHN0cmluZyxcbn1cblxuLyoqXG4gKiBBbiBBcHBTeW5jIEdyYXBoUUwgQVBJXG4gKlxuICogQHJlc291cmNlIEFXUzo6QXBwU3luYzo6R3JhcGhRTEFwaVxuICovXG5leHBvcnQgY2xhc3MgR3JhcGhxbEFwaSBleHRlbmRzIEdyYXBocWxBcGlCYXNlIHtcbiAgLyoqXG4gICAqIEltcG9ydCBhIEdyYXBoUUwgQVBJIHRocm91Z2ggdGhpcyBmdW5jdGlvblxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgc2NvcGVcbiAgICogQHBhcmFtIGlkIGlkXG4gICAqIEBwYXJhbSBhdHRycyBHcmFwaFFMIEFQSSBBdHRyaWJ1dGVzIG9mIGFuIEFQSVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tR3JhcGhxbEFwaUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEdyYXBocWxBcGlBdHRyaWJ1dGVzKTogSUdyYXBocWxBcGkge1xuICAgIGNvbnN0IGFybiA9IGF0dHJzLmdyYXBocWxBcGlBcm4gPz8gU3RhY2sub2Yoc2NvcGUpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnYXBwc3luYycsXG4gICAgICByZXNvdXJjZTogYGFwaXMvJHthdHRycy5ncmFwaHFsQXBpSWR9YCxcbiAgICB9KTtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBHcmFwaHFsQXBpQmFzZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgYXBpSWQgPSBhdHRycy5ncmFwaHFsQXBpSWQ7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgYXJuID0gYXJuO1xuICAgICAgY29uc3RydWN0b3IoczogQ29uc3RydWN0LCBpOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIocywgaSk7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogYW4gdW5pcXVlIEFXUyBBcHBTeW5jIEdyYXBoUUwgQVBJIGlkZW50aWZpZXJcbiAgICogaS5lLiAnbHh6Nzc1bHdkcmdjbmRnejNudXJ2YWM3b2EnXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXBpSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogdGhlIEFSTiBvZiB0aGUgQVBJXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIHRoZSBVUkwgb2YgdGhlIGVuZHBvaW50IGNyZWF0ZWQgYnkgQXBwU3luY1xuICAgKlxuICAgKiBAYXR0cmlidXRlIEdyYXBoUWxVcmxcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBncmFwaHFsVXJsOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIHRoZSBuYW1lIG9mIHRoZSBBUElcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIHRoZSBzY2hlbWEgYXR0YWNoZWQgdG8gdGhpcyBhcGlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzY2hlbWE6IFNjaGVtYTtcblxuICAvKipcbiAgICogVGhlIEF1dGhvcml6YXRpb24gVHlwZXMgZm9yIHRoaXMgR3JhcGhRTCBBcGlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtb2RlczogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICAvKipcbiAgICogdGhlIGNvbmZpZ3VyZWQgQVBJIGtleSwgaWYgcHJlc2VudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFwaSBrZXlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcGlLZXk/OiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBzY2hlbWFSZXNvdXJjZTogQ2ZuR3JhcGhRTFNjaGVtYTtcbiAgcHJpdmF0ZSBhcGk6IENmbkdyYXBoUUxBcGk7XG4gIHByaXZhdGUgYXBpS2V5UmVzb3VyY2U/OiBDZm5BcGlLZXk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEdyYXBocWxBcGlQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBkZWZhdWx0TW9kZSA9IHByb3BzLmF1dGhvcml6YXRpb25Db25maWc/LmRlZmF1bHRBdXRob3JpemF0aW9uID8/XG4gICAgICB7IGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZIH07XG4gICAgY29uc3QgYWRkaXRpb25hbE1vZGVzID0gcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZz8uYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcyA/PyBbXTtcbiAgICBjb25zdCBtb2RlcyA9IFtkZWZhdWx0TW9kZSwgLi4uYWRkaXRpb25hbE1vZGVzXTtcblxuICAgIHRoaXMubW9kZXMgPSBtb2Rlcy5tYXAoKG1vZGUpID0+IG1vZGUuYXV0aG9yaXphdGlvblR5cGUpO1xuXG4gICAgdGhpcy52YWxpZGF0ZUF1dGhvcml6YXRpb25Qcm9wcyhtb2Rlcyk7XG5cbiAgICB0aGlzLmFwaSA9IG5ldyBDZm5HcmFwaFFMQXBpKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG5hbWU6IHByb3BzLm5hbWUsXG4gICAgICBhdXRoZW50aWNhdGlvblR5cGU6IGRlZmF1bHRNb2RlLmF1dGhvcml6YXRpb25UeXBlLFxuICAgICAgbG9nQ29uZmlnOiB0aGlzLnNldHVwTG9nQ29uZmlnKHByb3BzLmxvZ0NvbmZpZyksXG4gICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB0aGlzLnNldHVwT3BlbklkQ29ubmVjdENvbmZpZyhkZWZhdWx0TW9kZS5vcGVuSWRDb25uZWN0Q29uZmlnKSxcbiAgICAgIHVzZXJQb29sQ29uZmlnOiB0aGlzLnNldHVwVXNlclBvb2xDb25maWcoZGVmYXVsdE1vZGUudXNlclBvb2xDb25maWcpLFxuICAgICAgbGFtYmRhQXV0aG9yaXplckNvbmZpZzogdGhpcy5zZXR1cExhbWJkYUF1dGhvcml6ZXJDb25maWcoZGVmYXVsdE1vZGUubGFtYmRhQXV0aG9yaXplckNvbmZpZyksXG4gICAgICBhZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlcnM6IHRoaXMuc2V0dXBBZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzKGFkZGl0aW9uYWxNb2RlcyksXG4gICAgICB4cmF5RW5hYmxlZDogcHJvcHMueHJheUVuYWJsZWQsXG4gICAgfSk7XG5cbiAgICB0aGlzLmFwaUlkID0gdGhpcy5hcGkuYXR0ckFwaUlkO1xuICAgIHRoaXMuYXJuID0gdGhpcy5hcGkuYXR0ckFybjtcbiAgICB0aGlzLmdyYXBocWxVcmwgPSB0aGlzLmFwaS5hdHRyR3JhcGhRbFVybDtcbiAgICB0aGlzLm5hbWUgPSB0aGlzLmFwaS5uYW1lO1xuICAgIHRoaXMuc2NoZW1hID0gcHJvcHMuc2NoZW1hID8/IG5ldyBTY2hlbWEoKTtcbiAgICB0aGlzLnNjaGVtYVJlc291cmNlID0gdGhpcy5zY2hlbWEuYmluZCh0aGlzKTtcblxuICAgIGlmIChwcm9wcy5kb21haW5OYW1lKSB7XG4gICAgICBuZXcgQ2ZuRG9tYWluTmFtZSh0aGlzLCAnRG9tYWluTmFtZScsIHtcbiAgICAgICAgZG9tYWluTmFtZTogcHJvcHMuZG9tYWluTmFtZS5kb21haW5OYW1lLFxuICAgICAgICBjZXJ0aWZpY2F0ZUFybjogcHJvcHMuZG9tYWluTmFtZS5jZXJ0aWZpY2F0ZS5jZXJ0aWZpY2F0ZUFybixcbiAgICAgICAgZGVzY3JpcHRpb246IGBkb21haW4gZm9yICR7dGhpcy5uYW1lfSBhdCAke3RoaXMuZ3JhcGhxbFVybH1gLFxuICAgICAgfSk7XG5cbiAgICAgIG5ldyBDZm5Eb21haW5OYW1lQXBpQXNzb2NpYXRpb24odGhpcywgJ0RvbWFpbkFzc29jaWF0aW9uJywge1xuICAgICAgICBkb21haW5OYW1lOiBwcm9wcy5kb21haW5OYW1lLmRvbWFpbk5hbWUsXG4gICAgICAgIGFwaUlkOiB0aGlzLmFwaUlkLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKG1vZGVzLnNvbWUoKG1vZGUpID0+IG1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkpKSB7XG4gICAgICBjb25zdCBjb25maWcgPSBtb2Rlcy5maW5kKChtb2RlOiBBdXRob3JpemF0aW9uTW9kZSkgPT4ge1xuICAgICAgICByZXR1cm4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSAmJiBtb2RlLmFwaUtleUNvbmZpZztcbiAgICAgIH0pPy5hcGlLZXlDb25maWc7XG4gICAgICB0aGlzLmFwaUtleVJlc291cmNlID0gdGhpcy5jcmVhdGVBUElLZXkoY29uZmlnKTtcbiAgICAgIHRoaXMuYXBpS2V5UmVzb3VyY2UuYWRkRGVwZW5kc09uKHRoaXMuc2NoZW1hUmVzb3VyY2UpO1xuICAgICAgdGhpcy5hcGlLZXkgPSB0aGlzLmFwaUtleVJlc291cmNlLmF0dHJBcGlLZXk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoaXMgR3JhcGhRTEFwaSB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsXG4gICAqIEBwYXJhbSByZXNvdXJjZXMgVGhlIHNldCBvZiByZXNvdXJjZXMgdG8gYWxsb3cgKGkuZS4gLi4uOltyZWdpb25dOlthY2NvdW50SWRdOmFwaXMvR3JhcGhRTElkLy4uLilcbiAgICogQHBhcmFtIGFjdGlvbnMgVGhlIGFjdGlvbnMgdGhhdCBzaG91bGQgYmUgZ3JhbnRlZCB0byB0aGUgcHJpbmNpcGFsIChpLmUuIGFwcHN5bmM6Z3JhcGhxbCApXG4gICAqL1xuICBwdWJsaWMgZ3JhbnQoZ3JhbnRlZTogSUdyYW50YWJsZSwgcmVzb3VyY2VzOiBJYW1SZXNvdXJjZSwgLi4uYWN0aW9uczogc3RyaW5nW10pOiBHcmFudCB7XG4gICAgcmV0dXJuIEdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zLFxuICAgICAgcmVzb3VyY2VBcm5zOiByZXNvdXJjZXMucmVzb3VyY2VBcm5zKHRoaXMpLFxuICAgICAgc2NvcGU6IHRoaXMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBmb3IgTXV0YXRpb24gYWNjZXNzIHRvIHRoaXMgR3JhcGhRTEFwaSB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsXG4gICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBncmFudCBhY2Nlc3MgdG8gdGhhdCBhcmUgTXV0YXRpb25zIChsZWF2ZSBibGFuayBmb3IgYWxsKVxuICAgKi9cbiAgcHVibGljIGdyYW50TXV0YXRpb24oZ3JhbnRlZTogSUdyYW50YWJsZSwgLi4uZmllbGRzOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCBJYW1SZXNvdXJjZS5vZlR5cGUoJ011dGF0aW9uJywgLi4uZmllbGRzKSwgJ2FwcHN5bmM6R3JhcGhRTCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgZm9yIFF1ZXJ5IGFjY2VzcyB0byB0aGlzIEdyYXBoUUxBcGkgdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCdzIHBvbGljeS5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbFxuICAgKiBAcGFyYW0gZmllbGRzIFRoZSBmaWVsZHMgdG8gZ3JhbnQgYWNjZXNzIHRvIHRoYXQgYXJlIFF1ZXJpZXMgKGxlYXZlIGJsYW5rIGZvciBhbGwpXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRRdWVyeShncmFudGVlOiBJR3JhbnRhYmxlLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KGdyYW50ZWUsIElhbVJlc291cmNlLm9mVHlwZSgnUXVlcnknLCAuLi5maWVsZHMpLCAnYXBwc3luYzpHcmFwaFFMJyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBmb3IgU3Vic2NyaXB0aW9uIGFjY2VzcyB0byB0aGlzIEdyYXBoUUxBcGkgdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCdzIHBvbGljeS5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbFxuICAgKiBAcGFyYW0gZmllbGRzIFRoZSBmaWVsZHMgdG8gZ3JhbnQgYWNjZXNzIHRvIHRoYXQgYXJlIFN1YnNjcmlwdGlvbnMgKGxlYXZlIGJsYW5rIGZvciBhbGwpXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRTdWJzY3JpcHRpb24oZ3JhbnRlZTogSUdyYW50YWJsZSwgLi4uZmllbGRzOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCBJYW1SZXNvdXJjZS5vZlR5cGUoJ1N1YnNjcmlwdGlvbicsIC4uLmZpZWxkcyksICdhcHBzeW5jOkdyYXBoUUwnKTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVBdXRob3JpemF0aW9uUHJvcHMobW9kZXM6IEF1dGhvcml6YXRpb25Nb2RlW10pIHtcbiAgICBpZiAobW9kZXMuZmlsdGVyKChtb2RlKSA9PiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5MQU1CREEpLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IGNhbiBvbmx5IGhhdmUgYSBzaW5nbGUgQVdTIExhbWJkYSBmdW5jdGlvbiBjb25maWd1cmVkIHRvIGF1dGhvcml6ZSB5b3VyIEFQSS4nKTtcbiAgICB9XG4gICAgbW9kZXMubWFwKChtb2RlKSA9PiB7XG4gICAgICBpZiAobW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuT0lEQyAmJiAhbW9kZS5vcGVuSWRDb25uZWN0Q29uZmlnKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignTWlzc2luZyBPSURDIENvbmZpZ3VyYXRpb24nKTtcbiAgICAgIH1cbiAgICAgIGlmIChtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wgJiYgIW1vZGUudXNlclBvb2xDb25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaXNzaW5nIFVzZXIgUG9vbCBDb25maWd1cmF0aW9uJyk7XG4gICAgICB9XG4gICAgICBpZiAobW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuTEFNQkRBICYmICFtb2RlLmxhbWJkYUF1dGhvcml6ZXJDb25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaXNzaW5nIExhbWJkYSBDb25maWd1cmF0aW9uJyk7XG4gICAgICB9XG4gICAgfSk7XG4gICAgaWYgKG1vZGVzLmZpbHRlcigobW9kZSkgPT4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSkubGVuZ3RoID4gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2FuXFwndCBkdXBsaWNhdGUgQVBJX0tFWSBjb25maWd1cmF0aW9uLiBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL3NlY3VyaXR5Lmh0bWwnKTtcbiAgICB9XG4gICAgaWYgKG1vZGVzLmZpbHRlcigobW9kZSkgPT4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuSUFNKS5sZW5ndGggPiAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBJQU0gY29uZmlndXJhdGlvbi4gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9zZWN1cml0eS5odG1sJyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBzY2hlbWEgZGVwZW5kZW5jeSB0byBhIGdpdmVuIGNvbnN0cnVjdFxuICAgKlxuICAgKiBAcGFyYW0gY29uc3RydWN0IHRoZSBkZXBlbmRlZVxuICAgKi9cbiAgcHVibGljIGFkZFNjaGVtYURlcGVuZGVuY3koY29uc3RydWN0OiBDZm5SZXNvdXJjZSk6IGJvb2xlYW4ge1xuICAgIGNvbnN0cnVjdC5hZGREZXBlbmRzT24odGhpcy5zY2hlbWFSZXNvdXJjZSk7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBwcml2YXRlIHNldHVwTG9nQ29uZmlnKGNvbmZpZz86IExvZ0NvbmZpZykge1xuICAgIGlmICghY29uZmlnKSByZXR1cm4gdW5kZWZpbmVkO1xuICAgIGNvbnN0IGxvZ3NSb2xlQXJuOiBzdHJpbmcgPSBjb25maWcucm9sZT8ucm9sZUFybiA/PyBuZXcgUm9sZSh0aGlzLCAnQXBpTG9nc1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdhcHBzeW5jLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICBNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0FwcFN5bmNQdXNoVG9DbG91ZFdhdGNoTG9ncycpLFxuICAgICAgXSxcbiAgICB9KS5yb2xlQXJuO1xuICAgIHJldHVybiB7XG4gICAgICBjbG91ZFdhdGNoTG9nc1JvbGVBcm46IGxvZ3NSb2xlQXJuLFxuICAgICAgZXhjbHVkZVZlcmJvc2VDb250ZW50OiBjb25maWcuZXhjbHVkZVZlcmJvc2VDb250ZW50LFxuICAgICAgZmllbGRMb2dMZXZlbDogY29uZmlnLmZpZWxkTG9nTGV2ZWwsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBPcGVuSWRDb25uZWN0Q29uZmlnKGNvbmZpZz86IE9wZW5JZENvbm5lY3RDb25maWcpIHtcbiAgICBpZiAoIWNvbmZpZykgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICByZXR1cm4ge1xuICAgICAgYXV0aFR0bDogY29uZmlnLnRva2VuRXhwaXJ5RnJvbUF1dGgsXG4gICAgICBjbGllbnRJZDogY29uZmlnLmNsaWVudElkLFxuICAgICAgaWF0VHRsOiBjb25maWcudG9rZW5FeHBpcnlGcm9tSXNzdWUsXG4gICAgICBpc3N1ZXI6IGNvbmZpZy5vaWRjUHJvdmlkZXIsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBVc2VyUG9vbENvbmZpZyhjb25maWc/OiBVc2VyUG9vbENvbmZpZykge1xuICAgIGlmICghY29uZmlnKSByZXR1cm4gdW5kZWZpbmVkO1xuICAgIHJldHVybiB7XG4gICAgICB1c2VyUG9vbElkOiBjb25maWcudXNlclBvb2wudXNlclBvb2xJZCxcbiAgICAgIGF3c1JlZ2lvbjogY29uZmlnLnVzZXJQb29sLnN0YWNrLnJlZ2lvbixcbiAgICAgIGFwcElkQ2xpZW50UmVnZXg6IGNvbmZpZy5hcHBJZENsaWVudFJlZ2V4LFxuICAgICAgZGVmYXVsdEFjdGlvbjogY29uZmlnLmRlZmF1bHRBY3Rpb24gfHwgVXNlclBvb2xEZWZhdWx0QWN0aW9uLkFMTE9XLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHNldHVwTGFtYmRhQXV0aG9yaXplckNvbmZpZyhjb25maWc/OiBMYW1iZGFBdXRob3JpemVyQ29uZmlnKSB7XG4gICAgaWYgKCFjb25maWcpIHJldHVybiB1bmRlZmluZWQ7XG4gICAgcmV0dXJuIHtcbiAgICAgIGF1dGhvcml6ZXJSZXN1bHRUdGxJblNlY29uZHM6IGNvbmZpZy5yZXN1bHRzQ2FjaGVUdGw/LnRvU2Vjb25kcygpLFxuICAgICAgYXV0aG9yaXplclVyaTogY29uZmlnLmhhbmRsZXIuZnVuY3Rpb25Bcm4sXG4gICAgICBpZGVudGl0eVZhbGlkYXRpb25FeHByZXNzaW9uOiBjb25maWcudmFsaWRhdGlvblJlZ2V4LFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHNldHVwQWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2Rlcyhtb2Rlcz86IEF1dGhvcml6YXRpb25Nb2RlW10pIHtcbiAgICBpZiAoIW1vZGVzIHx8IG1vZGVzLmxlbmd0aCA9PT0gMCkgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICByZXR1cm4gbW9kZXMucmVkdWNlPENmbkdyYXBoUUxBcGkuQWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJQcm9wZXJ0eVtdPigoYWNjLCBtb2RlKSA9PiBbXG4gICAgICAuLi5hY2MsIHtcbiAgICAgICAgYXV0aGVudGljYXRpb25UeXBlOiBtb2RlLmF1dGhvcml6YXRpb25UeXBlLFxuICAgICAgICB1c2VyUG9vbENvbmZpZzogdGhpcy5zZXR1cFVzZXJQb29sQ29uZmlnKG1vZGUudXNlclBvb2xDb25maWcpLFxuICAgICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB0aGlzLnNldHVwT3BlbklkQ29ubmVjdENvbmZpZyhtb2RlLm9wZW5JZENvbm5lY3RDb25maWcpLFxuICAgICAgICBsYW1iZGFBdXRob3JpemVyQ29uZmlnOiB0aGlzLnNldHVwTGFtYmRhQXV0aG9yaXplckNvbmZpZyhtb2RlLmxhbWJkYUF1dGhvcml6ZXJDb25maWcpLFxuICAgICAgfSxcbiAgICBdLCBbXSk7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUFQSUtleShjb25maWc/OiBBcGlLZXlDb25maWcpIHtcbiAgICBpZiAoY29uZmlnPy5leHBpcmVzPy5pc0JlZm9yZShEdXJhdGlvbi5kYXlzKDEpKSB8fCBjb25maWc/LmV4cGlyZXM/LmlzQWZ0ZXIoRHVyYXRpb24uZGF5cygzNjUpKSkge1xuICAgICAgdGhyb3cgRXJyb3IoJ0FQSSBrZXkgZXhwaXJhdGlvbiBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMzY1IGRheXMuJyk7XG4gICAgfVxuICAgIGNvbnN0IGV4cGlyZXMgPSBjb25maWc/LmV4cGlyZXMgPyBjb25maWc/LmV4cGlyZXMudG9FcG9jaCgpIDogdW5kZWZpbmVkO1xuICAgIHJldHVybiBuZXcgQ2ZuQXBpS2V5KHRoaXMsIGAke2NvbmZpZz8ubmFtZSB8fCAnRGVmYXVsdCd9QXBpS2V5YCwge1xuICAgICAgZXhwaXJlcyxcbiAgICAgIGRlc2NyaXB0aW9uOiBjb25maWc/LmRlc2NyaXB0aW9uLFxuICAgICAgYXBpSWQ6IHRoaXMuYXBpSWQsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRXNjYXBlIGhhdGNoIHRvIGFwcGVuZCB0byBTY2hlbWEgYXMgZGVzaXJlZC4gV2lsbCBhbHdheXMgcmVzdWx0XG4gICAqIGluIGEgbmV3bGluZS5cbiAgICpcbiAgICogQHBhcmFtIGFkZGl0aW9uIHRoZSBhZGRpdGlvbiB0byBhZGQgdG8gc2NoZW1hXG4gICAqIEBwYXJhbSBkZWxpbWl0ZXIgdGhlIGRlbGltaXRlciBiZXR3ZWVuIHNjaGVtYSBhbmQgYWRkaXRpb25cbiAgICogQGRlZmF1bHQgLSAnJ1xuICAgKlxuICAgKi9cbiAgcHVibGljIGFkZFRvU2NoZW1hKGFkZGl0aW9uOiBzdHJpbmcsIGRlbGltaXRlcj86IHN0cmluZyk6IHZvaWQge1xuICAgIHRoaXMuc2NoZW1hLmFkZFRvU2NoZW1hKGFkZGl0aW9uLCBkZWxpbWl0ZXIpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCB0eXBlIHRvIHRoZSBzY2hlbWFcbiAgICpcbiAgICogQHBhcmFtIHR5cGUgdGhlIGludGVybWVkaWF0ZSB0eXBlIHRvIGFkZCB0byB0aGUgc2NoZW1hXG4gICAqXG4gICAqL1xuICBwdWJsaWMgYWRkVHlwZSh0eXBlOiBJSW50ZXJtZWRpYXRlVHlwZSk6IElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgICByZXR1cm4gdGhpcy5zY2hlbWEuYWRkVHlwZSh0eXBlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBxdWVyeSBmaWVsZCB0byB0aGUgc2NoZW1hJ3MgUXVlcnkuIENESyB3aWxsIGNyZWF0ZSBhblxuICAgKiBPYmplY3QgVHlwZSBjYWxsZWQgJ1F1ZXJ5Jy4gRm9yIGV4YW1wbGUsXG4gICAqXG4gICAqIHR5cGUgUXVlcnkge1xuICAgKiAgIGZpZWxkTmFtZTogRmllbGQucmV0dXJuVHlwZVxuICAgKiB9XG4gICAqXG4gICAqIEBwYXJhbSBmaWVsZE5hbWUgdGhlIG5hbWUgb2YgdGhlIHF1ZXJ5XG4gICAqIEBwYXJhbSBmaWVsZCB0aGUgcmVzb2x2YWJsZSBmaWVsZCB0byBmb3IgdGhpcyBxdWVyeVxuICAgKi9cbiAgcHVibGljIGFkZFF1ZXJ5KGZpZWxkTmFtZTogc3RyaW5nLCBmaWVsZDogUmVzb2x2YWJsZUZpZWxkKTogT2JqZWN0VHlwZSB7XG4gICAgcmV0dXJuIHRoaXMuc2NoZW1hLmFkZFF1ZXJ5KGZpZWxkTmFtZSwgZmllbGQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIG11dGF0aW9uIGZpZWxkIHRvIHRoZSBzY2hlbWEncyBNdXRhdGlvbi4gQ0RLIHdpbGwgY3JlYXRlIGFuXG4gICAqIE9iamVjdCBUeXBlIGNhbGxlZCAnTXV0YXRpb24nLiBGb3IgZXhhbXBsZSxcbiAgICpcbiAgICogdHlwZSBNdXRhdGlvbiB7XG4gICAqICAgZmllbGROYW1lOiBGaWVsZC5yZXR1cm5UeXBlXG4gICAqIH1cbiAgICpcbiAgICogQHBhcmFtIGZpZWxkTmFtZSB0aGUgbmFtZSBvZiB0aGUgTXV0YXRpb25cbiAgICogQHBhcmFtIGZpZWxkIHRoZSByZXNvbHZhYmxlIGZpZWxkIHRvIGZvciB0aGlzIE11dGF0aW9uXG4gICAqL1xuICBwdWJsaWMgYWRkTXV0YXRpb24oZmllbGROYW1lOiBzdHJpbmcsIGZpZWxkOiBSZXNvbHZhYmxlRmllbGQpOiBPYmplY3RUeXBlIHtcbiAgICByZXR1cm4gdGhpcy5zY2hlbWEuYWRkTXV0YXRpb24oZmllbGROYW1lLCBmaWVsZCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgc3Vic2NyaXB0aW9uIGZpZWxkIHRvIHRoZSBzY2hlbWEncyBTdWJzY3JpcHRpb24uIENESyB3aWxsIGNyZWF0ZSBhblxuICAgKiBPYmplY3QgVHlwZSBjYWxsZWQgJ1N1YnNjcmlwdGlvbicuIEZvciBleGFtcGxlLFxuICAgKlxuICAgKiB0eXBlIFN1YnNjcmlwdGlvbiB7XG4gICAqICAgZmllbGROYW1lOiBGaWVsZC5yZXR1cm5UeXBlXG4gICAqIH1cbiAgICpcbiAgICogQHBhcmFtIGZpZWxkTmFtZSB0aGUgbmFtZSBvZiB0aGUgU3Vic2NyaXB0aW9uXG4gICAqIEBwYXJhbSBmaWVsZCB0aGUgcmVzb2x2YWJsZSBmaWVsZCB0byBmb3IgdGhpcyBTdWJzY3JpcHRpb25cbiAgICovXG4gIHB1YmxpYyBhZGRTdWJzY3JpcHRpb24oZmllbGROYW1lOiBzdHJpbmcsIGZpZWxkOiBSZXNvbHZhYmxlRmllbGQpOiBPYmplY3RUeXBlIHtcbiAgICByZXR1cm4gdGhpcy5zY2hlbWEuYWRkU3Vic2NyaXB0aW9uKGZpZWxkTmFtZSwgZmllbGQpO1xuICB9XG59XG4iXX0=