"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationsResourceHandler = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*'],
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs10.x',
                Timeout: 300,
            },
        });
        resource.node.addDependency(role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = cdk.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports, import/no-extraneous-dependencies
    const s3 = new (require('aws-sdk').S3)();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const https = require('https');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const url = require('url');
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration,
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse('FAILED', err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse('SUCCESS');
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || 'See the details in CloudWatch Log Stream: ' + context.logStreamName,
            PhysicalResourceId: event.PhysicalResourceId || event.LogicalResourceId,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': responseBody.length,
            },
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on('error', (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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