"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Pattern deployment with new Lambda function
// --------------------------------------------------------------
test('Pattern deployment with new Lambda function', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        }
    };
    new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Pattern deployment with existing Lambda function
// --------------------------------------------------------------
test('Pattern deployment with existing Lambda function', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const fn = new lambda.Function(stack, 'ExistingLambdaFunction', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    });
    const props = {
        deployLambda: false,
        existingLambdaObj: fn
    };
    new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test for error with deployLambda=false and
// existingLambdaObj=undefined (not supplied by user).
// --------------------------------------------------------------
test('Error on deployLambda=false and existingLambdaObj=undefined', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployLambda: false
    };
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test deployLambda=true with lambdaFunctionProps.
// --------------------------------------------------------------
test('Test deployLambda=true with lambdaFunctionProps', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`),
            environment: {
                OVERRIDE_STATUS: 'true'
            }
        },
        apiGatewayProps: {
            description: "sampleApiProp"
        }
    };
    const app = new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction).toHaveProperty('environment.OVERRIDE_STATUS', 'true');
});
// --------------------------------------------------------------
// Test getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        }
    };
    const app = new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction !== null);
    // Assertion 2
    expect(app.apiGateway !== null);
});
// --------------------------------------------------------------
// Test for error with deployLambda=true and
// lambdaFunctionProps=undefined (not supplied by user).
// --------------------------------------------------------------
test('Error on deployLambda=true and lambdaFunctionProps=undefined', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployLambda: true
    };
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Pattern deployment with two ApiGatewayToLambda constructs
// --------------------------------------------------------------
test('Pattern deployment with two ApiGatewayToLambda constructs', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props1 = {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        }
    };
    new lib_1.ApiGatewayToLambda(stack, 'pattern1', props1);
    const props2 = {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        }
    };
    new lib_1.ApiGatewayToLambda(stack, 'pattern2', props2);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
//# sourceMappingURL=data:application/json;base64,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