"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const kinesis = require("@aws-cdk/aws-kinesis");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const dests = require("../lib");
test('stream can be subscription destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const stream = new kinesis.Stream(stack, 'MyStream');
    const logGroup = new logs.LogGroup(stack, 'LogGroup');
    // WHEN
    new logs.SubscriptionFilter(stack, 'Subscription', {
        logGroup,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents()
    });
    // THEN: subscription target is Stream
    expect(stack).toHaveResource('AWS::Logs::SubscriptionFilter', {
        DestinationArn: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] },
        RoleArn: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] },
    });
    // THEN: we have a role to write to the Stream
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [{
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: {
                            'Fn::Join': ['', [
                                    'logs.',
                                    { Ref: 'AWS::Region' },
                                    '.',
                                    { Ref: 'AWS::URLSuffix' }
                                ]
                            ]
                        }
                    }
                }],
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords',
                    ],
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] }
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] }
                }
            ],
        }
    });
});
test('stream can be subscription destination twice, without duplicating permissions', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const stream = new kinesis.Stream(stack, 'MyStream');
    const logGroup1 = new logs.LogGroup(stack, 'LogGroup');
    const logGroup2 = new logs.LogGroup(stack, 'LogGroup2');
    // WHEN
    new logs.SubscriptionFilter(stack, 'Subscription', {
        logGroup: logGroup1,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents()
    });
    new logs.SubscriptionFilter(stack, 'Subscription2', {
        logGroup: logGroup2,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents()
    });
    // THEN: subscription target is Stream
    expect(stack).toHaveResource('AWS::Logs::SubscriptionFilter', {
        DestinationArn: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] },
        RoleArn: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] },
    });
    // THEN: we have a role to write to the Stream
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [{
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: {
                            'Fn::Join': ['', [
                                    'logs.',
                                    { Ref: 'AWS::Region' },
                                    '.',
                                    { Ref: 'AWS::URLSuffix' }
                                ]
                            ]
                        }
                    }
                }],
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords',
                    ],
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] }
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] }
                }
            ],
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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