"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.DatabaseClusterBase = exports.EngineVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const endpoint_1 = require("./endpoint");
const neptune_generated_1 = require("./neptune.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Instances Types to use in Neptune cluster
 * used for defining {@link DatabaseClusterProps.engineVersion}.
 */
class EngineVersion {
    /**
     * Constructor for specifying a custom engine version
     * @param version the engine version of Neptune
     */
    constructor(version) {
        this.version = version;
    }
}
exports.EngineVersion = EngineVersion;
_a = JSII_RTTI_SYMBOL_1;
EngineVersion[_a] = { fqn: "@aws-cdk/aws-neptune.EngineVersion", version: "1.157.0" };
/**
 * Neptune engine version 1.0.1.0
 */
EngineVersion.V1_0_1_0 = new EngineVersion('1.0.1.0');
/**
 * Neptune engine version 1.0.1.1
 */
EngineVersion.V1_0_1_1 = new EngineVersion('1.0.1.1');
/**
 * Neptune engine version 1.0.1.2
 */
EngineVersion.V1_0_1_2 = new EngineVersion('1.0.1.2');
/**
 * Neptune engine version 1.0.2.1
 */
EngineVersion.V1_0_2_1 = new EngineVersion('1.0.2.1');
/**
 * Neptune engine version 1.0.2.2
 */
EngineVersion.V1_0_2_2 = new EngineVersion('1.0.2.2');
/**
 * Neptune engine version 1.0.3.0
 */
EngineVersion.V1_0_3_0 = new EngineVersion('1.0.3.0');
/**
 * Neptune engine version 1.0.4.0
 */
EngineVersion.V1_0_4_0 = new EngineVersion('1.0.4.0');
/**
 * Neptune engine version 1.0.4.1
 */
EngineVersion.V1_0_4_1 = new EngineVersion('1.0.4.1');
/**
 * Neptune engine version 1.0.5.0
 */
EngineVersion.V1_0_5_0 = new EngineVersion('1.0.5.0');
/**
 * Neptune engine version 1.1.0.0
 */
EngineVersion.V1_1_0_0 = new EngineVersion('1.1.0.0');
/**
 * A new or imported database cluster.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_neptune_DatabaseClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseClusterAttributes);
            }
            throw error;
        }
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.clusterResourceIdentifier = attrs.clusterResourceIdentifier;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['neptune-db:*'],
            resourceArns: [
                [
                    'arn',
                    core_1.Aws.PARTITION,
                    'neptune-db',
                    core_1.Aws.REGION,
                    core_1.Aws.ACCOUNT_ID,
                    `${this.clusterResourceIdentifier}/*`,
                ].join(':'),
            ],
        });
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_b = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_b] = { fqn: "@aws-cdk/aws-neptune.DatabaseClusterBase", version: "1.157.0" };
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::Neptune::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(scope, id);
        /**
         * Identifiers of the instance
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual instance.
         */
        this.instanceEndpoints = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_neptune_DatabaseClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = (_d = props.vpcSubnets) !== null && _d !== void 0 ? _d : { subnetType: ec2.SubnetType.PRIVATE_WITH_NAT };
        // Determine the subnet(s) to deploy the Neptune cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_e = props.subnetGroup) !== null && _e !== void 0 ? _e : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        const securityGroups = (_f = props.securityGroups) !== null && _f !== void 0 ? _f : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Neptune security group',
                vpc: this.vpc,
            }),
        ];
        // Default to encrypted storage
        const storageEncrypted = (_g = props.storageEncrypted) !== null && _g !== void 0 ? _g : true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        const deletionProtection = (_h = props.deletionProtection) !== null && _h !== void 0 ? _h : (props.removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
        this.enableIamAuthentication = props.iamAuthentication;
        // Create the Neptune cluster
        const cluster = new neptune_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: (_j = props.engineVersion) === null || _j === void 0 ? void 0 : _j.version,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            port: props.port,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
            dbClusterParameterGroupName: (_k = props.clusterParameterGroup) === null || _k === void 0 ? void 0 : _k.clusterParameterGroupName,
            deletionProtection: deletionProtection,
            associatedRoles: props.associatedRoles ? props.associatedRoles.map(role => ({ roleArn: role.roleArn })) : undefined,
            iamAuthEnabled: core_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            // Backup
            backupRetentionPeriod: (_l = props.backupRetention) === null || _l === void 0 ? void 0 : _l.toDays(),
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // Encryption
            kmsKeyId: (_m = props.kmsKey) === null || _m === void 0 ? void 0 : _m.keyArn,
            storageEncrypted,
        });
        cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrClusterResourceId;
        const port = core_1.Token.asNumber(cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpoint, port);
        // Create the instances
        const instanceCount = (_o = props.instances) !== null && _o !== void 0 ? _o : DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new neptune_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: props.instanceType._instanceType,
                dbParameterGroupName: (_p = props.parameterGroup) === null || _p === void 0 ? void 0 : _p.parameterGroupName,
                autoMinorVersionUpgrade: props.autoMinorVersionUpgrade === true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: securityGroups,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
_c = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_c] = { fqn: "@aws-cdk/aws-neptune.DatabaseCluster", version: "1.157.0" };
/**
 * The default number of instances in the Neptune cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
//# sourceMappingURL=data:application/json;base64,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