from __future__ import annotations

from typing import TYPE_CHECKING

from lamin_utils import logger
from lamindb_setup.core import deprecated

from bionty.base.dev._handle_sources import LAMINDB_INSTANCE_LOADED

from ._settings import settings

if TYPE_CHECKING:
    from pandas import DataFrame
from .dev._handle_sources import parse_currently_used_sources


def display_sources() -> DataFrame:
    """Displays all available sources.

    Example::

        import bionty.base as bt_base

        bt.display_sources()
    """
    from .dev._handle_sources import parse_sources_yaml

    return parse_sources_yaml(settings.public_sources).set_index("entity")  # type: ignore


def display_currently_used_sources(mute: bool = False) -> DataFrame:
    """Displays all currently used sources.

    Active version is unique for entity + organism.

    Example::

        import bionty.base as bt_base

        bt.display_currently_used_sources()
    """
    if LAMINDB_INSTANCE_LOADED():
        if not mute:
            logger.error(
                "You have a LaminDB instance loaded, please run the following to check default sources:\n"
                "    → bt.Source.filter(currently_used=True).to_dataframe()"
            )

    versions = parse_currently_used_sources(settings.public_sources)

    df_rows = []
    for bionty_class, bionty_class_data in versions.items():
        for organism, organism_data in bionty_class_data.items():
            for source, version in organism_data.items():
                df_rows.append(
                    {
                        "entity": bionty_class,
                        "organism": organism,
                        "name": source,
                        "version": version,
                    }
                )

    import pandas as pd

    return pd.DataFrame(df_rows).set_index("entity")
