"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentGroup = exports.InstanceTagSet = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 *
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 *
 * @stability stable
 */
class InstanceTagSet {
    /**
     * @stability stable
     */
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    /**
     * @stability stable
     */
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
_a = JSII_RTTI_SYMBOL_1;
InstanceTagSet[_a] = { fqn: "@aws-cdk/aws-codedeploy.InstanceTagSet", version: "1.139.0" };
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 *
 * @stability stable
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_codedeploy_ServerDeploymentGroupProps(props);
        this.application = props.application || new application_1.ServerApplication(this, 'Application', {
            applicationName: props.deploymentGroupName === cdk.PhysicalName.GENERATE_IF_NEEDED ? cdk.PhysicalName.GENERATE_IF_NEEDED : undefined,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = (_c = props.installAgent) !== null && _c !== void 0 ? _c : true;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.list({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app, or in a different region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param attrs the properties of the referenced Deployment Group.
     * @returns a Construct representing a reference to an existing Deployment Group
     * @stability stable
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_codedeploy_ServerDeploymentGroupAttributes(attrs);
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * @stability stable
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group.
     * @stability stable
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * @stability stable
     */
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('set +e', // make sure we don't exit on the `which` failing
                'PKG_CMD=`which yum 2>/dev/null`', 'set -e', // continue with failing on error
                'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG_CMD=yum', 'fi', '$PKG_CMD update -y', 'set +e', // make sure we don't exit on the next command failing (we check its exit code below)
                '$PKG_CMD install -y ruby2.0', 'RUBY2_INSTALL=$?', 'set -e', // continue with failing on error
                'if [ $RUBY2_INSTALL -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', 'AWS_CLI_PACKAGE_NAME=awscli', 'if [ "$PKG_CMD" = "yum" ]; then', 'AWS_CLI_PACKAGE_NAME=aws-cli', 'fi', '$PKG_CMD install -y $AWS_CLI_PACKAGE_NAME', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, 'cd $TEMPDIR', '.\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
_b = JSII_RTTI_SYMBOL_1;
ServerDeploymentGroup[_b] = { fqn: "@aws-cdk/aws-codedeploy.ServerDeploymentGroup", version: "1.139.0" };
//# sourceMappingURL=data:application/json;base64,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