"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-konstruk/core");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const core_2 = require("@aws-solutions-konstruk/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let partitionKeyName;
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            const dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            partitionKeyName = dynamoTableProps.partitionKey.name;
            this.table = new dynamodb.Table(this, 'DynamoTable', dynamoTableProps);
        }
        else {
            partitionKeyName = defaults.DefaultTableProps.partitionKey.name;
            this.table = new dynamodb.Table(this, 'DynamoTable', defaults.DefaultTableProps);
        }
        // Setup the API Gateway
        this.apiGateway = defaults.GlobalRestApi(this);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.table.tableName);
            this.addActiontoPlicy("dynamodb:PutItem");
            this.addMethod(this.apiGateway.root, createRequestTemplate, "PutItem", "POST");
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "{\r\n\"TableName\": \"" + this.table.tableName + "\",\r\n \"KeyConditionExpression\": \"" + partitionKeyName + " = :v1\",\r\n    \"ExpressionAttributeValues\": {\r\n        \":v1\": {\r\n            \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n        }\r\n    }\r\n}";
            this.addActiontoPlicy("dynamodb:Query");
            this.addMethod(apiGatewayResource, getRequestTemplate, "Query", "GET");
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.table.tableName);
            this.addActiontoPlicy("dynamodb:UpdateItem");
            this.addMethod(apiGatewayResource, updateRequestTemplate, "UpdateItem", "PUT");
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "{\r\n  \"TableName\": \"" + this.table.tableName + "\",\r\n  \"Key\": {\r\n    \"" + partitionKeyName + "\": {\r\n      \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n    }\r\n  },\r\n  \"ConditionExpression\": \"attribute_not_exists(Replies)\",\r\n  \"ReturnValues\": \"ALL_OLD\"\r\n}";
            this.addActiontoPlicy("dynamodb:DeleteItem");
            this.addMethod(apiGatewayResource, deleteRequestTemplate, "DeleteItem", "DELETE");
        }
    }
    addActiontoPlicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.table.tableArn
            ],
            actions: [`${action}`]
        }));
    }
    addMethod(apiResource, requestTemplate, dynamodbAction, apiMethod) {
        // Setup the API Gateway Integration
        const apiGatewayIntegration = new api.AwsIntegration({
            service: "dynamodb",
            action: dynamodbAction,
            integrationHttpMethod: "POST",
            options: {
                passthroughBehavior: api.PassthroughBehavior.NEVER,
                credentialsRole: this.apiGatewayRole,
                requestParameters: {
                    "integration.request.header.Content-Type": "'application/json'"
                },
                requestTemplates: {
                    "application/json": requestTemplate
                },
                integrationResponses: [
                    {
                        statusCode: "200"
                    },
                    {
                        statusCode: "500",
                        responseTemplates: {
                            "text/html": "Error"
                        },
                        selectionPattern: "500"
                    }
                ]
            }
        });
        // Setup the API Gateway method(s)
        apiResource.addMethod(apiMethod, apiGatewayIntegration, {
            methodResponses: [
                {
                    statusCode: "200",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    }
                },
                {
                    statusCode: "500",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    },
                }
            ]
        });
    }
    /**
     * @summary Returns an instance of the api.RestApi created by the construct.
     * @returns {api.RestApi} Instance of the RestApi created by the construct.
     * @since 0.8.0
     * @access public
     */
    restApi() {
        return this.apiGateway;
    }
    /**
     * @summary Retruns an instance of dynamodb.Table created by the construct.
     * @returns {dynamodb.Table} Instance of dynamodb.Table created by the construct
     * @since 0.8.0
     * @access public
     */
    dynamoTable() {
        return this.table;
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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