"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
const validation_1 = require("./private/validation");
const aws_servicecatalogappregistry_1 = require("aws-cdk-lib/aws-servicecatalogappregistry");
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedAttributeGroups = new Set();
        this.associatedResources = new Set();
    }
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup) {
        if (!this.associatedAttributeGroups.has(attributeGroup.node.addr)) {
            const hashId = this.generateUniqueHash(attributeGroup.node.addr);
            new aws_servicecatalogappregistry_1.CfnAttributeGroupAssociation(this, `AttributeGroupAssociation${hashId}`, {
                application: this.applicationId,
                attributeGroup: attributeGroup.attributeGroupId,
            });
            this.associatedAttributeGroups.add(attributeGroup.node.addr);
        }
    }
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     */
    associateStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            const hashId = this.generateUniqueHash(stack.node.addr);
            new aws_servicecatalogappregistry_1.CfnResourceAssociation(this, `ResourceAssociation${hashId}`, {
                application: this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
        }
    }
}
/**
 * A Service Catalog AppRegistry Application.
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalogappregistry_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validateApplicationProps(props);
        const application = new aws_servicecatalogappregistry_1.CfnApplication(this, 'Resource', {
            name: props.applicationName,
            description: props.description,
        });
        this.applicationArn = application.attrArn;
        this.applicationId = application.attrId;
        this.nodeAddress = cdk.Names.nodeUniqueId(application.node);
    }
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const arn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME);
        const applicationId = arn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required Application ID from Application ARN: ' + applicationArn);
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationArn = applicationArn;
                this.applicationId = applicationId;
            }
            generateUniqueHash(resourceAddress) {
                return hashValues(this.applicationArn, resourceAddress);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: applicationArn,
        });
    }
    generateUniqueHash(resourceAddress) {
        return hashValues(this.nodeAddress, resourceAddress);
    }
    validateApplicationProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'application name', 1, 256, props.applicationName);
        validation_1.InputValidator.validateRegex(this.node.path, 'application name', /^[a-zA-Z0-9-_]+$/, props.applicationName);
        validation_1.InputValidator.validateLength(this.node.path, 'application description', 0, 1024, props.description);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-servicecatalogappregistry-alpha.Application", version: "2.26.0-alpha.0" };
/**
 * Generates a unique hash identfifer using SHA256 encryption algorithm
 */
function hashValues(...values) {
    const sha256 = crypto.createHash('sha256');
    values.forEach(val => sha256.update(val));
    return sha256.digest('hex').slice(0, 12);
}
//# sourceMappingURL=data:application/json;base64,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