"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StageStackAssociator = exports.CheckedStageStackAssociator = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("../common");
const utils_1 = require("../private/utils");
/**
 * Aspect class, this will visit each node from the provided construct once.
 *
 * For every stack node visited, this class will be responsible to associate
 * the stack to the application.
 */
class StackAssociatorBase {
    constructor() {
        this.sharedAccounts = new Set();
    }
    visit(node) {
        // verify if a stage in a particular stack is associated to Application.
        node.node.children.forEach((childNode) => {
            if (aws_cdk_lib_1.Stage.isStage(childNode)) {
                var stageAssociated = this.applicationAssociator?.isStageAssociated(childNode);
                if (stageAssociated === false) {
                    this.error(childNode, 'Associate Stage: ' + childNode.stageName + ' to ensure all stacks in your cdk app are associated with AppRegistry. '
                        + 'You can use ApplicationAssociator.associateStage to associate any stage.');
                }
            }
        });
        if (aws_cdk_lib_1.Stack.isStack(node)) {
            this.handleCrossRegionStack(node);
            this.handleCrossAccountStack(node);
            this.associate(node);
        }
    }
    /**
     * Associate a stage stack to the given application.
     *
     * @param node A Stage stack.
     */
    associate(node) {
        this.application.associateApplicationWithStack(node);
    }
    /**
     * Adds an error annotation to a node.
     *
     * @param node The scope to add the error to.
     * @param message The error message.
     */
    error(node, message) {
        aws_cdk_lib_1.Annotations.of(node).addError(message);
    }
    /**
     * Adds a warning annotation to a node.
     *
     * @param node The scope to add the warning to.
     * @param message The error message.
     */
    warning(node, message) {
        aws_cdk_lib_1.Annotations.of(node).addWarning(message);
    }
    /**
     * Handle cross-region association. AppRegistry do not support
     * cross region association at this moment,
     * If any stack is evaluated as cross-region than that of application,
     * we will throw an error.
     *
     * @param node Cfn stack.
     */
    handleCrossRegionStack(node) {
        if (utils_1.isRegionUnresolved(this.application.env.region, node.region)) {
            this.warning(node, 'Environment agnostic stack determined, AppRegistry association might not work as expected in case you deploy cross-region or cross-account stack.');
            return;
        }
        if (node.region != this.application.env.region) {
            this.error(node, 'AppRegistry does not support cross region associations. Application region '
                + this.application.env.region + ', stack region ' + node.region);
        }
    }
    /**
     * Handle cross-account association.
     * If any stack is evaluated as cross-account than that of application,
     * then we will share the application to the stack owning account.
     *
     * @param node Cfn stack.
     */
    handleCrossAccountStack(node) {
        if (utils_1.isAccountUnresolved(this.application.env.account, node.account)) {
            this.warning(node, 'Environment agnostic stack determined, AppRegistry association might not work as expected in case you deploy cross-region or cross-account stack.');
            return;
        }
        if (node.account != this.application.env.account && !this.sharedAccounts.has(node.account)) {
            this.application.shareApplication({
                accounts: [node.account],
                sharePermission: common_1.SharePermission.ALLOW_ACCESS,
            });
            this.sharedAccounts.add(node.account);
        }
    }
}
class CheckedStageStackAssociator extends StackAssociatorBase {
    constructor(app) {
        super();
        this.application = app.appRegistryApplication;
        this.applicationAssociator = app;
    }
}
exports.CheckedStageStackAssociator = CheckedStageStackAssociator;
class StageStackAssociator extends StackAssociatorBase {
    constructor(app) {
        super();
        this.application = app;
    }
}
exports.StageStackAssociator = StageStackAssociator;
//# sourceMappingURL=data:application/json;base64,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