"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ram_1 = require("aws-cdk-lib/aws-ram");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const stack_associator_1 = require("./aspects/stack-associator");
const common_1 = require("./common");
const utils_1 = require("./private/utils");
const validation_1 = require("./private/validation");
const aws_servicecatalogappregistry_1 = require("aws-cdk-lib/aws-servicecatalogappregistry");
const APPLICATION_READ_ONLY_RAM_PERMISSION_ARN = 'arn:aws:ram::aws:permission/AWSRAMPermissionServiceCatalogAppRegistryApplicationReadOnly';
const APPLICATION_ALLOW_ACCESS_RAM_PERMISSION_ARN = 'arn:aws:ram::aws:permission/AWSRAMPermissionServiceCatalogAppRegistryApplicationAllowAssociation';
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedAttributeGroups = new Set();
        this.associatedResources = new Set();
    }
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup) {
        if (!this.associatedAttributeGroups.has(attributeGroup.node.addr)) {
            const hashId = this.generateUniqueHash(attributeGroup.node.addr);
            new aws_servicecatalogappregistry_1.CfnAttributeGroupAssociation(this, `AttributeGroupAssociation${hashId}`, {
                application: this.applicationId,
                attributeGroup: attributeGroup.attributeGroupId,
            });
            this.associatedAttributeGroups.add(attributeGroup.node.addr);
        }
    }
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     *
     * @deprecated Use `associateApplicationWithStack` instead.
     */
    associateStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            const hashId = this.generateUniqueHash(stack.node.addr);
            new aws_servicecatalogappregistry_1.CfnResourceAssociation(this, `ResourceAssociation${hashId}`, {
                application: this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
        }
    }
    /**
     * Associate stack with the application in the stack passed as parameter.
     *
     * If the stack is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     */
    associateApplicationWithStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            new aws_servicecatalogappregistry_1.CfnResourceAssociation(stack, 'AppRegistryAssociation', {
                application: stack === cdk.Stack.of(this) ? this.applicationId : this.applicationName ?? this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
            if (stack !== cdk.Stack.of(this) && this.isSameAccount(stack) && !this.isStageScope(stack)) {
                stack.addDependency(cdk.Stack.of(this));
            }
        }
    }
    /**
     * Share an application with accounts, organizations and OUs, and IAM roles and users.
     * The application will become available to end users within those principals.
     *
     * @param shareOptions The options for the share.
     */
    shareApplication(shareOptions) {
        const principals = common_1.getPrincipalsforSharing(shareOptions);
        const shareName = `RAMShare${common_1.hashValues(aws_cdk_lib_1.Names.nodeUniqueId(this.node), this.node.children.length.toString())}`;
        new aws_ram_1.CfnResourceShare(this, shareName, {
            name: shareName,
            allowExternalPrincipals: false,
            principals: principals,
            resourceArns: [this.applicationArn],
            permissionArns: [this.getApplicationSharePermissionARN(shareOptions)],
        });
    }
    /**
     * Associate all stacks present in construct's aspect with application.
     *
     * NOTE: This method won't automatically register stacks under pipeline stages,
     * and requires association of each pipeline stage by calling this method with stage Construct.
     *
     */
    associateAllStacksInScope(scope) {
        cdk.Aspects.of(scope).add(new stack_associator_1.StageStackAssociator(this));
    }
    /**
     * Get the correct permission ARN based on the SharePermission
     */
    getApplicationSharePermissionARN(shareOptions) {
        switch (shareOptions.sharePermission) {
            case common_1.SharePermission.ALLOW_ACCESS:
                return APPLICATION_ALLOW_ACCESS_RAM_PERMISSION_ARN;
            case common_1.SharePermission.READ_ONLY:
                return APPLICATION_READ_ONLY_RAM_PERMISSION_ARN;
            default:
                return shareOptions.sharePermission ?? APPLICATION_READ_ONLY_RAM_PERMISSION_ARN;
        }
    }
    /**
    *  Checks whether a stack is defined in a Stage or not.
    */
    isStageScope(stack) {
        return !(stack.node.scope instanceof cdk.App) && (stack.node.scope instanceof cdk.Stage);
    }
    /**
     * Verifies if application and the visited node is deployed in different account.
     */
    isSameAccount(stack) {
        return utils_1.isAccountUnresolved(this.env.account, stack.account) || this.env.account === stack.account;
    }
}
/**
 * A Service Catalog AppRegistry Application.
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalogappregistry_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        this.validateApplicationProps(props);
        const application = new aws_servicecatalogappregistry_1.CfnApplication(this, 'Resource', {
            name: props.applicationName,
            description: props.description,
        });
        this.applicationArn = application.attrArn;
        this.applicationId = application.attrId;
        this.applicationName = props.applicationName;
        this.nodeAddress = cdk.Names.nodeUniqueId(application.node);
    }
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const arn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME);
        const applicationId = arn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required Application ID from Application ARN: ' + applicationArn);
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationArn = applicationArn;
                this.applicationId = applicationId;
                this.applicationName = undefined;
            }
            generateUniqueHash(resourceAddress) {
                return common_1.hashValues(this.applicationArn, resourceAddress);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: applicationArn,
        });
    }
    generateUniqueHash(resourceAddress) {
        return common_1.hashValues(this.nodeAddress, resourceAddress);
    }
    validateApplicationProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'application name', 1, 256, props.applicationName);
        validation_1.InputValidator.validateRegex(this.node.path, 'application name', /^[a-zA-Z0-9-_]+$/, props.applicationName);
        validation_1.InputValidator.validateLength(this.node.path, 'application description', 0, 1024, props.description);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-servicecatalogappregistry-alpha.Application", version: "2.47.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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