import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IAttributeGroup } from './attribute-group';
import { ShareOptions } from './common';
/**
 * A Service Catalog AppRegistry Application.
 */
export interface IApplication extends cdk.IResource {
    /**
     * The ARN of the application.
     * @attribute
     */
    readonly applicationArn: string;
    /**
     * The ID of the application.
     * @attribute
     */
    readonly applicationId: string;
    /**
     * The name of the application.
     * @attribute
     */
    readonly applicationName?: string;
    /**
     * Associate this application with an attribute group.
     *
     * @param attributeGroup AppRegistry attribute group
     */
    associateAttributeGroup(attributeGroup: IAttributeGroup): void;
    /**
     * Associate this application with a CloudFormation stack.
     *
     * @deprecated Use `associateApplicationWithStack` instead.
     * @param stack a CFN stack
     */
    associateStack(stack: cdk.Stack): void;
    /**
     * Associate a Cloudformation statck with the application in the given stack.
     *
     * @param stack a CFN stack
     */
    associateApplicationWithStack(stack: cdk.Stack): void;
    /**
     * Share this application with other IAM entities, accounts, or OUs.
     *
     * @param shareOptions The options for the share.
     */
    shareApplication(shareOptions: ShareOptions): void;
    /**
     * Associate this application with all stacks under the construct node.
     * NOTE: This method won't automatically register stacks under pipeline stages,
     * and requires association of each pipeline stage by calling this method with stage Construct.
     *
     * @param construct cdk Construct
     */
    associateAllStacksInScope(construct: Construct): void;
}
/**
 * Properties for a Service Catalog AppRegistry Application
 */
export interface ApplicationProps {
    /**
     * Enforces a particular physical application name.
     */
    readonly applicationName: string;
    /**
     * Description for application.
     * @default - No description provided
     */
    readonly description?: string;
}
declare abstract class ApplicationBase extends cdk.Resource implements IApplication {
    abstract readonly applicationArn: string;
    abstract readonly applicationId: string;
    abstract readonly applicationName?: string;
    private readonly associatedAttributeGroups;
    private readonly associatedResources;
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup: IAttributeGroup): void;
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     *
     * @deprecated Use `associateApplicationWithStack` instead.
     */
    associateStack(stack: cdk.Stack): void;
    /**
     * Associate stack with the application in the stack passed as parameter.
     *
     * If the stack is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     */
    associateApplicationWithStack(stack: cdk.Stack): void;
    /**
     * Share an application with accounts, organizations and OUs, and IAM roles and users.
     * The application will become available to end users within those principals.
     *
     * @param shareOptions The options for the share.
     */
    shareApplication(shareOptions: ShareOptions): void;
    /**
     * Associate all stacks present in construct's aspect with application.
     *
     * NOTE: This method won't automatically register stacks under pipeline stages,
     * and requires association of each pipeline stage by calling this method with stage Construct.
     *
     */
    associateAllStacksInScope(scope: Construct): void;
    /**
     * Create a unique id
     */
    protected abstract generateUniqueHash(resourceAddress: string): string;
    /**
     * Get the correct permission ARN based on the SharePermission
     */
    private getApplicationSharePermissionARN;
    /**
    *  Checks whether a stack is defined in a Stage or not.
    */
    private isStageScope;
    /**
     * Verifies if application and the visited node is deployed in different account.
     */
    private isSameAccount;
}
/**
 * A Service Catalog AppRegistry Application.
 */
export declare class Application extends ApplicationBase {
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope: Construct, id: string, applicationArn: string): IApplication;
    readonly applicationArn: string;
    readonly applicationId: string;
    readonly applicationName?: string;
    private readonly nodeAddress;
    constructor(scope: Construct, id: string, props: ApplicationProps);
    protected generateUniqueHash(resourceAddress: string): string;
    private validateApplicationProps;
}
export {};
