"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('State Machine', () => {
    test('Instantiate Default State Machine', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            stateMachineName: 'MyStateMachine',
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass'))
        });
        // THEN
        expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
            StateMachineName: 'MyStateMachine',
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}'
        });
    }),
        test('Instantiate Standard State Machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new stepfunctions.StateMachine(stack, 'MyStateMachine', {
                stateMachineName: 'MyStateMachine',
                definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
                stateMachineType: stepfunctions.StateMachineType.STANDARD
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                StateMachineName: 'MyStateMachine',
                StateMachineType: 'STANDARD',
                DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}'
            });
        }),
        test('Instantiate Express State Machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new stepfunctions.StateMachine(stack, 'MyStateMachine', {
                stateMachineName: 'MyStateMachine',
                definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
                stateMachineType: stepfunctions.StateMachineType.EXPRESS
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                StateMachineName: 'MyStateMachine',
                StateMachineType: 'EXPRESS',
                DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}'
            });
        }),
        test('log configuration', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const logGroup = new logs.LogGroup(stack, 'MyLogGroup');
            new stepfunctions.StateMachine(stack, 'MyStateMachine', {
                definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
                logs: {
                    destination: logGroup,
                    level: stepfunctions.LogLevel.FATAL,
                    includeExecutionData: false
                }
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
                LoggingConfiguration: {
                    Destinations: [{
                            CloudWatchLogsLogGroup: {
                                LogGroupArn: {
                                    'Fn::GetAtt': ['MyLogGroup5C0DAD85', 'Arn']
                                }
                            }
                        }],
                    IncludeExecutionData: false,
                    Level: 'FATAL'
                }
            });
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [{
                            Action: [
                                'logs:CreateLogDelivery',
                                'logs:GetLogDelivery',
                                'logs:UpdateLogDelivery',
                                'logs:DeleteLogDelivery',
                                'logs:ListLogDeliveries',
                                'logs:PutResourcePolicy',
                                'logs:DescribeResourcePolicies',
                                'logs:DescribeLogGroups'
                            ],
                            Effect: 'Allow',
                            Resource: '*'
                        }],
                    Version: '2012-10-17'
                },
                PolicyName: 'MyStateMachineRoleDefaultPolicyE468EB18',
                Roles: [
                    {
                        Ref: 'MyStateMachineRoleD59FFEBC'
                    }
                ]
            });
        });
});
//# sourceMappingURL=data:application/json;base64,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