"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('State Machine Resources', () => {
    test('Tasks can add permissions to the execution role', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource']
                        })],
                })
            }
        });
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: task
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'resource:Everything',
                        Effect: 'Allow',
                        Resource: 'resource'
                    }
                ],
            }
        });
    }),
        test('Tasks hidden inside a Parallel state are also included', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        policyStatements: [
                            new iam.PolicyStatement({
                                actions: ['resource:Everything'],
                                resources: ['resource']
                            })
                        ]
                    })
                }
            });
            const para = new stepfunctions.Parallel(stack, 'Para');
            para.branch(task);
            // WHEN
            new stepfunctions.StateMachine(stack, 'SM', {
                definition: para
            });
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: 'resource:Everything',
                            Effect: 'Allow',
                            Resource: 'resource'
                        }
                    ],
                }
            });
        }),
        test('Task should render InputPath / Parameters / OutputPath correctly', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                inputPath: '$',
                outputPath: '$.state',
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        parameters: {
                            'input.$': '$',
                            'stringArgument': 'inital-task',
                            'numberArgument': 123,
                            'booleanArgument': true,
                            'arrayArgument': ['a', 'b', 'c']
                        }
                    })
                }
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({ End: true,
                Retry: undefined,
                Catch: undefined,
                InputPath: '$',
                Parameters: { 'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'] },
                OutputPath: '$.state',
                Type: 'Task',
                Comment: undefined,
                Resource: 'resource',
                ResultPath: undefined,
                TimeoutSeconds: undefined,
                HeartbeatSeconds: undefined
            });
        }),
        test('Task combines taskobject parameters with direct parameters', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                inputPath: '$',
                outputPath: '$.state',
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        parameters: {
                            a: 'aa',
                        }
                    })
                },
                parameters: {
                    b: 'bb'
                }
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({ End: true,
                Retry: undefined,
                Catch: undefined,
                InputPath: '$',
                Parameters: { a: 'aa',
                    b: 'bb', },
                OutputPath: '$.state',
                Type: 'Task',
                Comment: undefined,
                Resource: 'resource',
                ResultPath: undefined,
                TimeoutSeconds: undefined,
                HeartbeatSeconds: undefined
            });
        }),
        test('Created state machine can grant start execution to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' })
                }
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
            });
            // WHEN
            stateMachine.grantStartExecution(role);
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:StartExecution',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5'
                            }
                        }
                    ],
                    Version: '2012-10-17',
                },
                PolicyName: 'RoleDefaultPolicy5FFB7DAB',
                Roles: [
                    {
                        Ref: 'Role1ABCC5F0'
                    }
                ]
            });
        }),
        test('Imported state machine can grant start execution to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
            });
            // WHEN
            stateMachine.grantStartExecution(role);
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:StartExecution',
                            Effect: 'Allow',
                            Resource: stateMachineArn
                        }
                    ],
                    Version: '2012-10-17',
                },
                PolicyName: 'RoleDefaultPolicy5FFB7DAB',
                Roles: [
                    {
                        Ref: 'Role1ABCC5F0'
                    }
                ]
            });
        }),
        test('Pass should render InputPath / Parameters / OutputPath correctly', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Pass(stack, 'Pass', {
                inputPath: '$',
                outputPath: '$.state',
                parameters: {
                    'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c']
                }
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({ End: true,
                InputPath: '$',
                OutputPath: '$.state',
                Parameters: { 'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'] },
                Type: 'Pass',
                Comment: undefined,
                Result: undefined,
                ResultPath: undefined,
            });
        }),
        test('State machines must depend on their roles', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        policyStatements: [
                            new iam.PolicyStatement({
                                resources: ['resource'],
                                actions: ['lambda:InvokeFunction'],
                            })
                        ],
                    })
                }
            });
            new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                DependsOn: [
                    'StateMachineRoleDefaultPolicyDF1E6607',
                    'StateMachineRoleB840431D'
                ]
            }, assert_1.ResourcePart.CompleteDefinition);
        });
});
//# sourceMappingURL=data:application/json;base64,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