"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('Map State', () => {
    test('State Machine With Map State', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const map = new stepfunctions.Map(stack, 'Map State', {
            maxConcurrency: 1,
            itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
            parameters: {
                foo: 'foo',
                bar: stepfunctions.Data.stringAt('$.bar')
            }
        });
        map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
        // THEN
        expect(render(map)).toStrictEqual({
            StartAt: 'Map State',
            States: {
                'Map State': {
                    Type: 'Map',
                    End: true,
                    Parameters: { foo: 'foo', bar: '$.bar' },
                    Iterator: {
                        StartAt: 'Pass State',
                        States: {
                            'Pass State': {
                                Type: 'Pass',
                                End: true
                            }
                        }
                    },
                    ItemsPath: '$.inputForMap',
                    MaxConcurrency: 1
                }
            }
        });
    }),
        test('synth is successful', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            app.synth();
        }),
        test('fails in synthesis if iterator is missing', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
                });
                return map;
            });
            expect(() => app.synth()).toThrow(/Map state must have a non-empty iterator/);
        }),
        test('fails in synthesis when maxConcurrency is a float', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1.2,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('fails in synthesis when maxConcurrency is a negative integer', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: -1,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('fails in synthesis when maxConcurrency is too big to be an integer', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: Number.MAX_VALUE,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('isPositiveInteger is false with negative number', () => {
            expect(stepfunctions.isPositiveInteger(-1)).toEqual(false);
        }),
        test('isPositiveInteger is false with decimal number', () => {
            expect(stepfunctions.isPositiveInteger(1.2)).toEqual(false);
        }),
        test('isPositiveInteger is false with a value greater than safe integer', () => {
            const valueToTest = Number.MAX_SAFE_INTEGER + 1;
            expect(stepfunctions.isPositiveInteger(valueToTest)).toEqual(false);
        }),
        test('isPositiveInteger is true with 0', () => {
            expect(stepfunctions.isPositiveInteger(0)).toEqual(true);
        }),
        test('isPositiveInteger is true with 10', () => {
            expect(stepfunctions.isPositiveInteger(10)).toEqual(true);
        }),
        test('isPositiveInteger is true with max integer value', () => {
            expect(stepfunctions.isPositiveInteger(Number.MAX_SAFE_INTEGER)).toEqual(true);
        });
});
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
function createAppWithMap(mapFactory) {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'my-stack');
    const map = mapFactory(stack);
    new stepfunctions.StateGraph(map, 'Test Graph');
    return app;
}
//# sourceMappingURL=data:application/json;base64,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