"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
describe('Fields', () => {
    test('deep replace correctly handles fields in arrays', () => {
        expect(lib_1.FieldUtils.renderObject({
            unknown: undefined,
            bool: true,
            literal: 'literal',
            field: lib_1.Data.stringAt('$.stringField'),
            listField: lib_1.Data.listAt('$.listField'),
            deep: [
                'literal',
                {
                    deepField: lib_1.Data.numberAt('$.numField'),
                },
            ],
        })).toStrictEqual({
            'bool': true,
            'literal': 'literal',
            'field.$': '$.stringField',
            'listField.$': '$.listField',
            'deep': [
                'literal',
                {
                    'deepField.$': '$.numField',
                },
            ],
        });
    }),
        test('exercise contextpaths', () => {
            expect(lib_1.FieldUtils.renderObject({
                str: lib_1.Context.stringAt('$$.Execution.StartTime'),
                count: lib_1.Context.numberAt('$$.State.RetryCount'),
                token: lib_1.Context.taskToken,
                entire: lib_1.Context.entireContext,
            })).toStrictEqual({
                'str.$': '$$.Execution.StartTime',
                'count.$': '$$.State.RetryCount',
                'token.$': '$$.Task.Token',
                'entire.$': '$$',
            });
        }),
        test('find all referenced paths', () => {
            expect(lib_1.FieldUtils.findReferencedPaths({
                bool: false,
                literal: 'literal',
                field: lib_1.Data.stringAt('$.stringField'),
                listField: lib_1.Data.listAt('$.listField'),
                deep: [
                    'literal',
                    {
                        field: lib_1.Data.stringAt('$.stringField'),
                        deepField: lib_1.Data.numberAt('$.numField'),
                    },
                ],
            })).toStrictEqual(['$.listField', '$.numField', '$.stringField']);
        }),
        test('cannot have JsonPath fields in arrays', () => {
            expect(() => lib_1.FieldUtils.renderObject({
                deep: [lib_1.Data.stringAt('$.hello')],
            })).toThrowError(/Cannot use JsonPath fields in an array/);
        }),
        test('datafield path must be correct', () => {
            expect(lib_1.Data.stringAt('$')).toBeDefined();
            expect(() => lib_1.Data.stringAt('$hello')).toThrowError(/exactly equal to '\$' or start with '\$.'/);
            expect(() => lib_1.Data.stringAt('hello')).toThrowError(/exactly equal to '\$' or start with '\$.'/);
        }),
        test('context path must be correct', () => {
            expect(lib_1.Context.stringAt('$$')).toBeDefined();
            expect(() => lib_1.Context.stringAt('$$hello')).toThrowError(/exactly equal to '\$\$' or start with '\$\$.'/);
            expect(() => lib_1.Context.stringAt('hello')).toThrowError(/exactly equal to '\$\$' or start with '\$\$.'/);
        }),
        test('test contains task token', () => {
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.Context.taskToken,
            }));
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.Context.stringAt('$$.Task'),
            }));
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.Context.entireContext,
            }));
            expect(false).toEqual(lib_1.FieldUtils.containsTaskToken({
                oops: 'not here',
            }));
            expect(false).toEqual(lib_1.FieldUtils.containsTaskToken({
                oops: lib_1.Context.stringAt('$$.Execution.StartTime'),
            }));
        }),
        test('arbitrary JSONPath fields are not replaced', () => {
            expect(lib_1.FieldUtils.renderObject({
                field: '$.content',
            })).toStrictEqual({
                field: '$.content',
            });
        }),
        test('fields cannot be used somewhere in a string interpolation', () => {
            expect(() => lib_1.FieldUtils.renderObject({
                field: `contains ${lib_1.Data.stringAt('$.hello')}`,
            })).toThrowError(/Field references must be the entire string/);
        });
});
//# sourceMappingURL=data:application/json;base64,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