"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const chain_1 = require("../chain");
const fields_1 = require("../fields");
const util_1 = require("../util");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Task state in the state machine
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 */
class Task extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.timeout = props.timeout;
        const taskProps = props.task.bind(this);
        this.taskProps = {
            ...taskProps,
            parameters: util_1.noEmptyObject({ ...taskProps.parameters || {}, ...props.parameters || {} }),
        };
        this.endStates = [this];
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderInputOutput(),
            Type: state_type_1.StateType.TASK,
            Comment: this.comment,
            Resource: this.taskProps.resourceArn,
            Parameters: this.taskProps.parameters && fields_1.FieldUtils.renderObject(this.taskProps.parameters),
            ResultPath: state_1.renderJsonPath(this.resultPath),
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
            HeartbeatSeconds: this.taskProps.heartbeat && this.taskProps.heartbeat.toSeconds(),
        };
    }
    /**
     * Return the given named metric for this Task
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: this.taskProps.metricDimensions,
            statistic: 'sum',
            ...props
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricRunTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default average over 5 minutes
     */
    metricScheduleTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default sum over 5 minutes
     */
    metricScheduled(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Started', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Failed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskProps.policyStatements || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('This Task Resource does not expose metrics');
        }
        return this.metric(prefix + suffix, props);
    }
}
exports.Task = Task;
//# sourceMappingURL=data:application/json;base64,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