"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Choice in the state machine
 *
 * A choice state can be used to make decisions based on the execution
 * state.
 */
class Choice extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [];
    }
    /**
     * If the given condition matches, continue execution with the given state
     */
    when(condition, next) {
        super.addChoice(condition, next.startState);
        return this;
    }
    /**
     * If none of the given conditions match, continue execution with the given state
     *
     * If no conditions match and no otherwise() has been given, an execution
     * error will be raised.
     */
    otherwise(def) {
        super.makeDefault(def.startState);
        return this;
    }
    /**
     * Return a Chain that contains all reachable end states from this Choice
     *
     * Use this to combine all possible choice paths back.
     */
    afterwards(options = {}) {
        const endStates = state_1.State.filterNextables(state_1.State.findReachableEndStates(this, { includeErrorHandlers: options.includeErrorHandlers }));
        if (options.includeOtherwise && this.defaultChoice) {
            throw new Error(`'includeOtherwise' set but Choice state ${this.stateId} already has an 'otherwise' transition`);
        }
        if (options.includeOtherwise) {
            endStates.push(new DefaultAsNext(this));
        }
        return chain_1.Chain.custom(this, endStates, this);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.CHOICE,
            Comment: this.comment,
            ...this.renderInputOutput(),
            ...this.renderChoices(),
        };
    }
}
exports.Choice = Choice;
/**
 * Adapter to make the .otherwise() transition settable through .next()
 */
class DefaultAsNext {
    constructor(choice) {
        this.choice = choice;
    }
    next(state) {
        this.choice.otherwise(state);
        return chain_1.Chain.sequence(this.choice, state);
    }
}
//# sourceMappingURL=data:application/json;base64,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