"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const chain_1 = require("./chain");
const parallel_1 = require("./states/parallel");
const state_1 = require("./states/state");
/**
 * Base class for reusable state machine fragments
 */
class StateMachineFragment extends cdk.Construct {
    get id() {
        return this.node.id;
    }
    /**
     * Prefix the IDs of all states in this state machine fragment
     *
     * Use this to avoid multiple copies of the state machine all having the
     * same state IDs.
     *
     * @param prefix The prefix to add. Will use construct ID by default.
     */
    prefixStates(prefix) {
        state_1.State.prefixStates(this, prefix || `${this.id}: `);
        return this;
    }
    /**
     * Wrap all states in this state machine fragment up into a single state.
     *
     * This can be used to add retry or error handling onto this state
     * machine fragment.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(options = {}) {
        const stateId = options.stateId || this.id;
        this.prefixStates(options.prefixStates || `${stateId}: `);
        return new parallel_1.Parallel(this, stateId, options).branch(this);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        return chain_1.Chain.start(this).next(next);
    }
}
exports.StateMachineFragment = StateMachineFragment;
//# sourceMappingURL=data:application/json;base64,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