"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        //
        // It may be that a construct already exists with the given action name (CDK Pipelines
        // may do this to maintain construct tree compatibility between versions).
        //
        // If so, we simply reuse it.
        let actionScope = constructs_1.Node.of(this.scope).tryFindChild(action.actionProperties.actionName);
        if (!actionScope) {
            actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        }
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.any({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.any({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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