"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderInExpression = exports.jsonPathFromAny = exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("@aws-cdk/core");
const intrinstics_1 = require("./intrinstics");
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
        handleResolvable: renderResolvable,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathString(x))) {
                found.add(p);
            }
            return {};
        },
        handleList(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathStringList(x))) {
                found.add(p);
            }
            return {};
        },
        handleNumber(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathNumber(x))) {
                found.add(p);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
        handleResolvable(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathFromAny(x))) {
                found.add(p);
            }
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
/**
 * From an expression, return the list of JSON paths referenced in it
 */
function findPathsInIntrinsicFunctions(expression) {
    if (!expression) {
        return [];
    }
    const ret = new Array();
    try {
        const parsed = new intrinstics_1.IntrinsicParser(expression).parseTopLevelIntrinsic();
        recurse(parsed);
        return ret;
    }
    catch (e) {
        // Not sure that our parsing is 100% correct. We don't want to break anyone, so
        // fall back to legacy behavior if we can't parse this string.
        return [expression];
    }
    function recurse(p) {
        switch (p.type) {
            case 'path':
                ret.push(p.path);
                break;
            case 'fncall':
                for (const arg of p.arguments) {
                    recurse(arg);
                }
        }
    }
}
function recurseObject(obj, handlers, visited = []) {
    // If the argument received is not actually an object (string, number, boolean, undefined, ...) or null
    // just return it as is as there's nothing to be rendered. This should only happen in the original call to
    // recurseObject as any recursive calls to it are checking for typeof value === 'object' && value !== null
    if (typeof obj !== 'object' || obj === null) {
        return obj;
    }
    // Avoiding infinite recursion
    if (visited.includes(obj)) {
        return {};
    }
    // Marking current object as visited for the current recursion path
    visited.push(obj);
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers, visited));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            if (core_1.Tokenization.isResolvable(value)) {
                Object.assign(ret, handlers.handleResolvable(key, value));
            }
            else {
                ret[key] = recurseObject(value, handlers, visited);
            }
        }
    }
    // Removing from visited after leaving the current recursion path
    // Allowing it to be visited again if it's not causing a recursion (circular reference)
    visited.pop();
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers, visited = []) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers, visited);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a resolvable
 *
 * If we can extract a Path from it, render as a path string, otherwise as itself (will
 * be resolved later
 */
function renderResolvable(key, value) {
    const path = jsonPathFromAny(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
function jsonPathFromAny(x) {
    if (!x) {
        return undefined;
    }
    if (typeof x === 'string') {
        return jsonPathString(x);
    }
    return pathFromToken(core_1.Tokenization.reverse(x));
}
exports.jsonPathFromAny = jsonPathFromAny;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
/**
 * Render the string in a valid JSON Path expression.
 *
 * If the string is a Tokenized JSON path reference -- return the JSON path reference inside it.
 * Otherwise, single-quote it.
 *
 * Call this function whenever you're building compound JSONPath expressions, in
 * order to avoid having tokens-in-tokens-in-tokens which become very hard to parse.
 */
function renderInExpression(x) {
    const path = jsonPathString(x);
    return path !== null && path !== void 0 ? path : singleQuotestring(x);
}
exports.renderInExpression = renderInExpression;
function singleQuotestring(x) {
    const ret = new Array();
    ret.push("'");
    for (const c of x) {
        if (c === "'") {
            ret.push("\\'");
        }
        else if (c === '\\') {
            ret.push('\\\\');
        }
        else if (c === '\n') {
            ret.push('\\n');
        }
        else {
            ret.push(c);
        }
    }
    ret.push("'");
    return ret.join('');
}
//# sourceMappingURL=data:application/json;base64,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