from __future__ import annotations
from asyncio.futures import Future

import sys
from os import environ, execl

from asyncio import (
    wait_for,
    new_event_loop,
    AbstractEventLoop,
    iscoroutinefunction,
    run_coroutine_threadsafe
)
from dotenv import load_dotenv
from typing import (
    Any,
    Callable,
    Awaitable,
    Coroutine,
    NoReturn
)
from pathlib import Path
from threading import Thread
from subprocess import run

from ujson import loads
from aiohttp import ClientSession
from colorama import Fore, Style, init

from .ws import Ws
from .db import DB
from .obj import Message, clear
from .exceptions import (
    AccountNotFoundInDotenv,
    EventIsntAsync,
    InvalidDotenvKeys,
    CommandIsntAsync,
    InvalidChatChoice,
    InvalidEvent
)

Coro_return_chat_message = Callable[[], Coroutine[Message.from_ws, None, None]]
Coro_return_Any = Callable[[], Coroutine[Any, None, None]]
Coro_return_None = Callable[[], Coroutine[None, None, None]]

version = '0.0.40'

class Bot:
    __slots__ = (
        '_db',
        '_email',
        '_loop',
        '_msg',
        '_password',
        '_ws',
        'commands',
        'events',
        'id',
        'ignore_chats',
        'only_chats',
        'prefix'
    )

    def __init__(
        self,
        email:        str | None           = None,
        password:     str | None           = None,
        prefix:       str                  = '/',
        only_chats:   dict[str, list[str]] = {},
        ignore_chats: dict[str, list[str]] = {}
    ):
        init()
        # load_dotenv is not identifying the .env on project folder,
        # so I use Path to get the absolute .env path
        load_dotenv(Path('.env').absolute())
        try:
            self._email    = email    or environ['EMAIL']
            self._password = password or environ['PASSWORD']
        except KeyError:
            raise InvalidDotenvKeys('Your .env must have the keys: EMAIL and PASSWORD')

        if not self._email or not self._password:
            raise AccountNotFoundInDotenv('Put your email and password in .env')
        if only_chats and ignore_chats:
            raise InvalidChatChoice('Enter chats only in "only_chats" or "ignore_chats"')

        clear()

        self.id:    str | None        = None
        self._db:   DB                = DB()
        self._msg:  Message           = Message()
        self._loop: AbstractEventLoop = new_event_loop()
        self.prefix = prefix

        self.only_chats   = only_chats
        self.ignore_chats = ignore_chats

        self.commands: dict[str, dict[str, list[str], Coro_return_None, str]] = {}
        self.events:   dict[str, list[Coro_return_None]] = {
                                                    'ready':      [],
                                                    'close':      [],
                                                    'message':    [],
                                                    'join_chat':  [],
                                                    'leave_chat': [],
                                                    'image':      []
                                                }

    def add(
        self,
        help:    str       = 'No help',
        aliases: list[str] = []
    ) -> Callable[[Coro_return_chat_message], None]:

        def foo(f: Coro_return_chat_message) -> None:
            if not iscoroutinefunction(f):
                raise CommandIsntAsync('Command must be async: "async def ..."')
            self.commands[f.__name__] = {
                                    'aliases': aliases,
                                    'def': f,
                                    'help': help
                                }
        return foo

    def on(self) -> Callable[[Coro_return_chat_message], None]:
        def foo(f: Coro_return_chat_message) -> None:
            if f.__name__ not in self.events:
                raise InvalidEvent(f.__name__)

            if not iscoroutinefunction(f):
                raise EventIsntAsync('Event must be async: "async def ..."')
            self.events[f.__name__].append(f)

        return foo

    async def check_update(self) -> NoReturn | None:
        def try_update() -> NoReturn | None:
            if self._db.deps_need_update():
                cmd = run('pip install -U amsync', capture_output=True, text=True)
            else:
                cmd = run('pip install -U amsync --no-deps', capture_output=True, text=True)

            if cmd.returncode:
                clear()
                print(f'Error updating from version {Style.BRIGHT}{version}{Style.NORMAL} to {Fore.CYAN}{new}{Fore.WHITE}\n\n')
                print(cmd.stdout or cmd.stderr)
                sys.exit(1)

        if (
            'DYNO' not in environ   # not in heroku
            and self._db.lib_need_update()
        ):
            async with ClientSession(json_serialize=loads) as s:
                async with s.get('https://pypi.org/pypi/Amsync/json') as res:
                    new = (await res.json(loads=loads))['info']['version']
                    if new != version:
                        print(f'There is a new version: {Fore.CYAN}{new}{Fore.WHITE}')
                        print(f'Actual version: {Style.BRIGHT}{version}{Style.NORMAL}\n')
                        print(f'Do you want to update it? (Y/n) ', end='')
                        if input().lower() == 'y':
                            clear()
                            print('Updating')
                            try_update()
                            clear()
                            print('Restarting...\n')
                            execl(sys.executable, Path(__file__).absolute(), *sys.argv)
                        clear()

    def run(self) -> None:
        self._loop.run_until_complete(self.check_update())
        self._ws: Ws = Ws(
            loop         = self._loop,
            email        = self._email,
            password     = self._password,
            only_chats   = self.only_chats,
            ignore_chats = self.ignore_chats
        )

        Thread(target=self._loop.run_forever).start()
        fut = run_coroutine_threadsafe(
            self._ws.run(
                call   = self._call,
                events = self.events,
                bot    = self),
            self._loop
        )

        # On error "run_coroutine_threadsafe" pauses the program as a raise Exception,
        # but does not print the exception on the screen.
        # So it is necessary to take the exception and raise it to show
        try:
            fut.result()
        except:
            raise fut.exception()

    async def send(
        self,
        *msgs: list[str],
        files: str     | None = None,
        type_: int     | None = 0,
        embed: 'Embed' | None = None, # type: ignore
        com:   str     | None = None,
        chat:  str     | None = None
    ) -> Awaitable[list[dict[str, Any]]]:

        return await self._msg.send(
            *msgs,
            files = files,
            type_ = type_,
            embed = embed,
            com   = com,
            chat  = chat
        )

    async def wait_for(
        self,
        check: Callable[[Message.from_ws], bool] = lambda _: True,
        timeout: int | None = None
    ) -> Awaitable[Future, int | None] | None:

        future = self._loop.create_future()
        self._ws.futures.append(future)

        try:
            if check(msg := await wait_for(future, timeout)):
                return msg

            # Calls wait_for until the condition is met
            return await self.wait_for(check, timeout)
        except TimeoutError:
            # Delete the future canceled by asyncio.wait_for
            del self._ws.futures[self._ws.futures.index(future)]

    def _is_alias(self, name):
        for command_name, args in self.commands.items():
            if name in args['aliases']:
                return command_name

    async def _call(self, m: Message) -> None:
        if m.text and m.text.startswith(self.prefix):
            splited      = m.text.split()
            name         = splited[0][len(self.prefix):]
            command_name = self._is_alias(name) or name

            if command_name in self.commands:
                if (
                    len(splited) > 1 
                    and splited[1] in (f'{self.prefix}h', f'{self.prefix}help')
                ):
                    await self.send(self.commands[command_name]['help'])
                else:
                    # Remove command name from text
                    m.text = ' '.join(splited[1:])
                    self._loop.create_task(self.commands[command_name]['def'](m))
