"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
module.exports = {
    'Pipeline': {
        'can be passed an IAM role during pipeline creation'(test) {
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                role,
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'RoleArn': {
                    'Fn::GetAtt': [
                        'Role1ABCC5F0',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'can be imported by ARN'(test) {
            const stack = new cdk.Stack();
            const pipeline = codepipeline.Pipeline.fromPipelineArn(stack, 'Pipeline', 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineArn, 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineName, 'MyPipeline');
            test.done();
        },
        'that is cross-region': {
            'validates that source actions are in the same region as the pipeline'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { region: 'us-west-1', account: '123456789012' } });
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
                const sourceStage = pipeline.addStage({
                    stageName: 'Source',
                });
                const sourceAction = new fake_source_action_1.FakeSourceAction({
                    actionName: 'FakeSource',
                    output: new codepipeline.Artifact(),
                    region: 'ap-southeast-1',
                });
                test.throws(() => {
                    sourceStage.addAction(sourceAction);
                }, /Source action 'FakeSource' must be in the same region as the pipeline/);
                test.done();
            },
            'allows passing an Alias in place of the KMS Key in the replication Bucket'(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const replicationStack = new cdk.Stack(app, 'ReplicationStack', {
                    env: { region: replicationRegion, account: '123456789012' },
                });
                const replicationKey = new kms.Key(replicationStack, 'ReplicationKey');
                const replicationAlias = replicationKey.addAlias('alias/my-replication-alias');
                const replicationBucket = new s3.Bucket(replicationStack, 'ReplicationBucket', {
                    encryptionKey: replicationAlias,
                    bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: replicationBucket,
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':kms:us-west-1:123456789012:alias/my-replication-alias',
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                }));
                assert_1.expect(replicationStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
                    'KeyPolicy': {
                        'Statement': [
                            {
                            // owning account management permissions - we don't care about them in this test
                            },
                            {
                                // KMS verifies whether the principal given in its key policy exists when creating that key.
                                // Since the replication bucket must be deployed before the pipeline,
                                // we cannot put the pipeline role as the principal here -
                                // hence, we put the account itself
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': ['', [
                                                'arn:',
                                                { 'Ref': 'AWS::Partition' },
                                                ':iam::123456789012:root',
                                            ]],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                    },
                }));
                test.done();
            },
            "generates ArtifactStores with the alias' ARN as the KeyID"(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'MyStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':kms:us-west-1:123456789012:alias/s-west-1tencryptionalias9b344b2b8e6825cb1f7d',
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                }));
                assert_1.expect(pipeline.crossRegionSupport[replicationRegion].stack).to(assert_1.haveResourceLike('AWS::KMS::Alias', {
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                }, assert_1.ResourcePart.CompleteDefinition));
                test.done();
            },
            'allows passing an imported Bucket and Key for the replication Bucket'(test) {
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(undefined, undefined, {
                    env: { region: pipelineRegion },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: s3.Bucket.fromBucketAttributes(pipelineStack, 'ReplicationBucket', {
                            bucketArn: 'arn:aws:s3:::my-us-west-1-replication-bucket',
                            encryptionKey: kms.Key.fromKeyArn(pipelineStack, 'ReplicationKey', `arn:aws:kms:${replicationRegion}:123456789012:key/1234-5678-9012`),
                        }),
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'Location': 'my-us-west-1-replication-bucket',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': 'arn:aws:kms:us-west-1:123456789012:key/1234-5678-9012',
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                }));
                test.done();
            },
            'generates the support stack containing the replication Bucket without the need to bootstrap in that environment'(test) {
                const app = new cdk.App({
                    treeMetadata: false,
                });
                app.node.setContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT, true);
                const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                    env: { region: 'us-west-2', account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: 'eu-south-1',
                                })],
                        },
                    ],
                });
                const assembly = app.synth();
                const supportStackArtifact = assembly.getStackByName('PipelineStack-support-eu-south-1');
                test.equal(supportStackArtifact.assumeRoleArn, 'arn:${AWS::Partition}:iam::123456789012:role/cdk-hnb659fds-deploy-role-123456789012-us-west-2');
                test.equal(supportStackArtifact.cloudFormationExecutionRoleArn, 'arn:${AWS::Partition}:iam::123456789012:role/cdk-hnb659fds-cfn-exec-role-123456789012-us-west-2');
                test.done();
            },
        },
        'that is cross-account': {
            'does not allow passing a dynamic value in the Action account property'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { account: '123456789012' } });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: cdk.Aws.ACCOUNT_ID,
                    }));
                }, /The 'account' property must be a concrete value \(action: 'FakeBuild'\)/);
                test.done();
            },
            'does not allow an env-agnostic Pipeline Stack if an Action account has been provided'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack');
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: '123456789012',
                    }));
                }, /Pipeline stack which uses cross-environment actions must have an explicitly set account/);
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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