"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const events = require("@aws-cdk/aws-events");
const ecs = require("@aws-cdk/aws-ecs");
const clusterName = "custom-cluster-name";
const containerName = "custom-container-name";
const serviceName = "custom-service-name";
const familyName = "family-name";
const testCidr = "172.0.0.0/16";
test('Check for construct properties', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = createFargateConstructWithNewResources(stack, publicApi);
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.eventBus).toBeDefined();
});
test('Check for new service', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".events"
                ]
            ]
        }
    });
});
test('Check for an existing service', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const existingVpc = defaults.getTestVpc(stack);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, { clusterName }, defaults.fakeEcrRepoArn, undefined, { family: familyName }, { containerName }, { serviceName });
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingVpc,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".events"
                ]
            ]
        }
    });
    expect(stack).toCountResources("AWS::ECS::Service", 1);
});
test('Check for IAM put events policy for created bus', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructcustomnameA27657F0",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for IAM put events policy for default event bus', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":events:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":event-bus/default"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for public/private VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::Events::EventBus', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Check for isolated VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::Events::EventBus', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Check for an existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingVpc,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16"
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
});
test('Check for custom ARN resource', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customEnvName = "TEST_CUSTOM_ARN";
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        eventBusEnvironmentVariableName: customEnvName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: familyName,
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customEnvName,
                        Value: "default"
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
});
test('Check for an existing event bus', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customName = 'MyCustomEventBus';
    const existingEventBus = new events.EventBus(stack, 'bus', {
        eventBusName: customName
    });
    const construct = new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingEventBusInterface: existingEventBus
    });
    expect(construct.eventBus).toBeDefined();
    expect(stack).toHaveResource("AWS::Events::EventBus", {
        Name: customName
    });
});
test('Check for custom event bus props', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const eventBusName = "custom-name";
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::Events::EventBus", {
        Name: eventBusName
    });
});
function createFargateConstructWithNewResources(stack, publicApi) {
    return new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        eventBusProps: {
            eventBusName: 'custom-name'
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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