"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const kinesis = require("@aws-cdk/aws-kinesis");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const dests = require("../lib");
test('stream can be subscription destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const stream = new kinesis.Stream(stack, 'MyStream');
    const logGroup = new logs.LogGroup(stack, 'LogGroup');
    // WHEN
    new logs.SubscriptionFilter(stack, 'Subscription', {
        logGroup,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents(),
    });
    // THEN: subscription target is Stream
    expect(stack).toHaveResource('AWS::Logs::SubscriptionFilter', {
        DestinationArn: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] },
        RoleArn: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] },
    });
    // THEN: we have a role to write to the Stream
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [{
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: {
                            'Fn::Join': ['', [
                                    'logs.',
                                    { Ref: 'AWS::Region' },
                                    '.',
                                    { Ref: 'AWS::URLSuffix' },
                                ]],
                        },
                    },
                }],
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords',
                    ],
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] },
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] },
                },
            ],
        },
    });
});
test('stream can be subscription destination twice, without duplicating permissions', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const stream = new kinesis.Stream(stack, 'MyStream');
    const logGroup1 = new logs.LogGroup(stack, 'LogGroup');
    const logGroup2 = new logs.LogGroup(stack, 'LogGroup2');
    // WHEN
    new logs.SubscriptionFilter(stack, 'Subscription', {
        logGroup: logGroup1,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents(),
    });
    new logs.SubscriptionFilter(stack, 'Subscription2', {
        logGroup: logGroup2,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents(),
    });
    // THEN: subscription target is Stream
    expect(stack).toHaveResource('AWS::Logs::SubscriptionFilter', {
        DestinationArn: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] },
        RoleArn: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] },
    });
    // THEN: we have a role to write to the Stream
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [{
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: {
                            'Fn::Join': ['', [
                                    'logs.',
                                    { Ref: 'AWS::Region' },
                                    '.',
                                    { Ref: 'AWS::URLSuffix' },
                                ]],
                        },
                    },
                }],
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords',
                    ],
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] },
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['SubscriptionCloudWatchLogsCanPutRecords9C1223EC', 'Arn'] },
                },
            ],
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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