"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const r53 = require("@aws-cdk/aws-route53");
const elb = require("@aws-cdk/aws-elasticloadbalancingv2");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
// Helper Functions
function getTestVpc(stack) {
    return defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
    });
}
test('Test Public API, new VPC', () => {
    // Initial Setup
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const newZone = new r53.PublicHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com'
    });
    const props = {
        publicApi: true,
        existingHostedZoneInterface: newZone,
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Test Private API, existing VPC', () => {
    // Initial Setup
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        existingVpc: testExistingVpc
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Test Private API, new VPC', () => {
    // Initial Setup
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        publicApi: false,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Check publicApi and zone props is an error', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const props = {
        publicApi: true,
        existingVpc: testExistingVpc,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check no Zone props and no existing zone interface is an error', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const props = {
        publicApi: false,
        existingVpc: testExistingVpc,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check Zone props with VPC is an error', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const props = {
        publicApi: false,
        existingVpc: testExistingVpc,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
            vpc: testExistingVpc
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Test with privateHostedZoneProps', () => {
    // Initial Setup
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const props = {
        publicApi: false,
        existingVpc: testExistingVpc,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    new lib_1.Route53ToAlb(stack, 'test-error', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Check that passing an existing hosted Zone without passing an existingVPC is an error', () => {
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check that passing an existing Load Balancer without passing an existingVPC is an error', () => {
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingLoadBalancerObj: existingAlb,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check that passing an existing ALB without passing an existingVPC is an error', () => {
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check that passing loadBalancerProps with a vpc is an error', () => {
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        loadBalancerProps: {
            loadBalancerName: 'my-alb',
            vpc: testExistingVpc,
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Test providing loadBalancerProps', () => {
    // Initial Setup
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        existingVpc: testExistingVpc,
        loadBalancerProps: {
            loadBalancerName: 'find-this-name'
        },
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal',
        Name: 'find-this-name'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Test providing an existingLoadBalancer', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        loadBalancerName: 'find-this-name'
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        existingVpc: testExistingVpc,
        existingLoadBalancerObj: existingAlb,
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal',
        Name: 'find-this-name'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Check publicApi and without an existing hosted zone is an error', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const testExistingVpc = getTestVpc(stack);
    const props = {
        publicApi: true,
        existingVpc: testExistingVpc,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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