"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrincipalsforSharing = exports.hashValues = exports.SharePermission = void 0;
const crypto = require("crypto");
/**
 * Supported permissions for sharing applications or attribute groups with principals using AWS RAM.
 */
var SharePermission;
(function (SharePermission) {
    /**
     * Allows principals in the share to only view the application or attribute group.
     */
    SharePermission[SharePermission["READ_ONLY"] = 0] = "READ_ONLY";
    /**
     * Allows principals in the share to associate resources and attribute groups with applications.
     */
    SharePermission[SharePermission["ALLOW_ACCESS"] = 1] = "ALLOW_ACCESS";
})(SharePermission = exports.SharePermission || (exports.SharePermission = {}));
;
/**
 * Generates a unique hash identfifer using SHA256 encryption algorithm.
 */
function hashValues(...values) {
    const sha256 = crypto.createHash('sha256');
    values.forEach(val => sha256.update(val));
    return sha256.digest('hex').slice(0, 12);
}
exports.hashValues = hashValues;
/**
 * Reformats share targets into a collapsed list necessary for handler.
 *
 * @param options The share target options
 * @returns flat list of target ARNs
 */
function getPrincipalsforSharing(options) {
    const principals = [
        ...options.accounts ?? [],
        ...options.organizationArns ?? [],
        ...options.users ? options.users.map(user => user.userArn) : [],
        ...options.roles ? options.roles.map(role => role.roleArn) : [],
    ];
    if (principals.length == 0) {
        throw new Error('An entity must be provided for the share');
    }
    return principals;
}
exports.getPrincipalsforSharing = getPrincipalsforSharing;
//# sourceMappingURL=data:application/json;base64,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