"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StageStackAssociator = exports.CheckedStageStackAssociator = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("../common");
const utils_1 = require("../private/utils");
/**
 * Aspect class, this will visit each node from the provided construct once.
 *
 * For every stack node visited, this class will be responsible to associate
 * the stack to the application.
 */
class StackAssociatorBase {
    constructor() {
        this.sharedAccounts = new Set();
    }
    visit(node) {
        // verify if a stage in a particular stack is associated to Application.
        node.node.children.forEach((childNode) => {
            if (aws_cdk_lib_1.Stage.isStage(childNode)) {
                var stageAssociated = this.applicationAssociator?.isStageAssociated(childNode);
                if (stageAssociated === false) {
                    this.error(childNode, 'Associate Stage: ' + childNode.stageName + ' to ensure all stacks in your cdk app are associated with AppRegistry. '
                        + 'You can use ApplicationAssociator.associateStage to associate any stage.');
                }
            }
        });
        if (aws_cdk_lib_1.Stack.isStack(node)) {
            this.handleCrossRegionStack(node);
            this.handleCrossAccountStack(node);
            this.associate(node);
        }
    }
    /**
     * Associate a stage stack to the given application.
     *
     * @param node A Stage stack.
     */
    associate(node) {
        this.application.associateApplicationWithStack(node);
    }
    /**
     * Adds an error annotation to a node.
     *
     * @param node The scope to add the error to.
     * @param message The error message.
     */
    error(node, message) {
        aws_cdk_lib_1.Annotations.of(node).addError(message);
    }
    /**
     * Adds a warning annotation to a node.
     *
     * @param node The scope to add the warning to.
     * @param message The error message.
     */
    warning(node, message) {
        aws_cdk_lib_1.Annotations.of(node).addWarning(message);
    }
    /**
     * Handle cross-region association. AppRegistry do not support
     * cross region association at this moment,
     * If any stack is evaluated as cross-region than that of application,
     * we will throw an error.
     *
     * @param node Cfn stack.
     */
    handleCrossRegionStack(node) {
        if (utils_1.isRegionUnresolved(this.application.env.region, node.region)) {
            this.warning(node, 'Environment agnostic stack determined, AppRegistry association might not work as expected in case you deploy cross-region or cross-account stack.');
            return;
        }
        if (node.region != this.application.env.region) {
            this.warning(node, 'AppRegistry does not support cross region associations, deployment might fail if there is cross region stacks in the app. Application region '
                + this.application.env.region + ', stack region ' + node.region);
        }
    }
    /**
     * Handle cross-account association.
     * If any stack is evaluated as cross-account than that of application,
     * then we will share the application to the stack owning account.
     *
     * @param node Cfn stack.
     */
    handleCrossAccountStack(node) {
        if (utils_1.isAccountUnresolved(this.application.env.account, node.account)) {
            this.warning(node, 'Environment agnostic stack determined, AppRegistry association might not work as expected in case you deploy cross-region or cross-account stack.');
            return;
        }
        if (node.account != this.application.env.account && !this.sharedAccounts.has(node.account)) {
            this.application.shareApplication({
                accounts: [node.account],
                sharePermission: common_1.SharePermission.ALLOW_ACCESS,
            });
            this.sharedAccounts.add(node.account);
        }
    }
}
class CheckedStageStackAssociator extends StackAssociatorBase {
    constructor(app) {
        super();
        this.application = app.appRegistryApplication();
        this.applicationAssociator = app;
    }
}
exports.CheckedStageStackAssociator = CheckedStageStackAssociator;
class StageStackAssociator extends StackAssociatorBase {
    constructor(app) {
        super();
        this.application = app;
    }
}
exports.StageStackAssociator = StageStackAssociator;
//# sourceMappingURL=data:application/json;base64,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